#!/usr/bin/env python3
"""周报生成模块：每周日 23:59 汇总"""

import json
import logging
from datetime import datetime, timedelta
from pathlib import Path

logger = logging.getLogger("heartbeat.weekly")

WORKSPACE = Path(__file__).parent.parent / "workspace"


def _load_state() -> dict:
    """加载状态文件"""
    state_path = WORKSPACE / "state.json"
    if not state_path.exists():
        return {}
    try:
        return json.loads(state_path.read_text(encoding="utf-8"))
    except (json.JSONDecodeError, Exception):
        return {}


def generate_weekly_report() -> dict:
    """
    生成周报数据

    返回:
        {
            "period": str,
            "health_trend": {"avg": float, "min": int, "max": int, "total_beats": int},
            "task_summary": {"completed": int, "blocked": int, "active": int},
            "highlights": [str],
            "concerns": [str],
        }
    """
    now = datetime.now()
    week_start = now - timedelta(days=7)

    report = {
        "period": f"{week_start.strftime('%m-%d')} ~ {now.strftime('%m-%d')}",
        "health_trend": {"avg": 0, "min": 0, "max": 0, "total_beats": 0},
        "task_summary": {"completed": 0, "blocked": 0, "active": 0},
        "highlights": [],
        "concerns": [],
    }

    # 健康度趋势
    state = _load_state()
    week_scores = []
    week_start_str = week_start.strftime("%Y-%m-%d")
    for h in state.get("health_history", []):
        t = h.get("time", "")
        if t >= week_start_str:
            week_scores.append(h["score"])

    if week_scores:
        report["health_trend"] = {
            "avg": round(sum(week_scores) / len(week_scores), 1),
            "min": min(week_scores),
            "max": max(week_scores),
            "total_beats": len(week_scores),
        }

        # 趋势分析
        mid = len(week_scores) // 2
        first_half_avg = sum(week_scores[:mid]) / max(1, mid)
        second_half_avg = sum(week_scores[mid:]) / max(1, len(week_scores) - mid)

        if second_half_avg > first_half_avg + 5:
            report["highlights"].append("健康度呈上升趋势 📈")
        elif second_half_avg < first_half_avg - 5:
            report["concerns"].append("健康度呈下降趋势 📉")
        else:
            report["highlights"].append("健康度保持稳定 📊")

    # 任务统计
    ongoing_path = WORKSPACE / "ongoing.json"
    if ongoing_path.exists():
        try:
            data = json.loads(ongoing_path.read_text(encoding="utf-8"))
            tasks = data if isinstance(data, list) else data.get("tasks", [])
            for task in tasks:
                st = task.get("status", "")
                if st == "DONE":
                    report["task_summary"]["completed"] += 1
                elif st == "BLOCK":
                    report["task_summary"]["blocked"] += 1
                    report["concerns"].append(f"任务阻塞: {task.get('title', '?')}")
                elif st in ("WIP", "WAIT"):
                    report["task_summary"]["active"] += 1
        except (json.JSONDecodeError, Exception):
            pass

    # 连续 streak
    streak = state.get("streak", 0)
    if streak >= 20:
        report["highlights"].append(f"连续健康心跳: {streak} 次 🔥")
    elif streak >= 10:
        report["highlights"].append(f"连续健康心跳: {streak} 次 ✅")

    return report


def format_weekly_report(report: dict) -> str:
    """将周报数据格式化为邮件文本"""
    lines = []
    lines.append(f"📊 EVA 周报 | {report['period']}")
    lines.append("=" * 40)
    lines.append("")

    # 健康度趋势
    h = report["health_trend"]
    lines.append("## 健康度趋势")
    lines.append(f"  平均分: {h['avg']}")
    lines.append(f"  分数范围: {h['min']} - {h['max']}")
    lines.append(f"  心跳总次数: {h['total_beats']}")
    lines.append("")

    # 任务统计
    t = report["task_summary"]
    lines.append("## 任务统计")
    lines.append(f"  已完成: {t['completed']}")
    lines.append(f"  进行中: {t['active']}")
    lines.append(f"  阻塞中: {t['blocked']}")
    lines.append("")

    # 亮点
    if report["highlights"]:
        lines.append("## 亮点")
        for item in report["highlights"]:
            lines.append(f"  ✨ {item}")
        lines.append("")

    # 关注点
    if report["concerns"]:
        lines.append("## 需关注")
        for item in report["concerns"]:
            lines.append(f"  ⚠️ {item}")
        lines.append("")

    lines.append("---")
    lines.append("Generated by EVA Heartbeat Manager")

    return "\n".join(lines)


def send_weekly_report() -> bool:
    """生成并发送周报"""
    report = generate_weekly_report()
    body = format_weekly_report(report)

    from tools.mail import send_mail
    subject = f"[EVA周报] {report['period']} | avg:{report['health_trend']['avg']}"

    sent = send_mail(subject, body)
    if sent:
        logger.info("周报发送成功")
    else:
        logger.error("周报发送失败")
    return sent
