#!/usr/bin/env python3
"""每日 00:00 重置 + 日报发送模块"""

import json
import logging
from datetime import datetime, timedelta
from pathlib import Path

logger = logging.getLogger("heartbeat.daily_reset")

WORKSPACE = Path(__file__).parent.parent / "workspace"
TEMPLATES = Path(__file__).parent.parent / "templates"


def _load_daily_template() -> str:
    """加载 daily 模板"""
    path = TEMPLATES / "daily_template.md"
    if path.exists():
        return path.read_text(encoding="utf-8")
    # 默认模板
    return (
        "# Daily Tasks | {{ date }}\n\n"
        "- [ ] 晨间邮件检查\n"
        "- [ ] 更新记忆库\n"
        "- [ ] 工作区整理\n"
        "- [ ] 系统状态确认\n"
        "- [ ] 晚间回顾总结\n"
        "- [ ] 📡 同步 Canvas + FSP 日程（打开浏览器后心跳自动抓取）\n"
    )


def _collect_yesterday_summary() -> dict:
    """收集昨日完成情况"""
    summary = {
        "date": (datetime.now() - timedelta(days=1)).strftime("%Y-%m-%d"),
        "daily": {"done": 0, "total": 0, "items": []},
        "todo": {"completed": 0, "items": []},
        "ongoing": {"done": [], "active": []},
        "health": {"scores": [], "average": 0},
    }

    # 读取当前 daily.md（重置前）
    daily_path = WORKSPACE / "daily.md"
    if daily_path.exists():
        import re
        content = daily_path.read_text(encoding="utf-8")
        for line in content.splitlines():
            m = re.match(r"^-\s*\[([ xX])\]\s*(.+)$", line.strip())
            if m:
                done = m.group(1).lower() == "x"
                text = m.group(2).strip()
                summary["daily"]["total"] += 1
                if done:
                    summary["daily"]["done"] += 1
                summary["daily"]["items"].append({"text": text, "done": done})

    # 读取 ongoing.json 中已完成的任务
    ongoing_path = WORKSPACE / "ongoing.json"
    if ongoing_path.exists():
        try:
            data = json.loads(ongoing_path.read_text(encoding="utf-8"))
            tasks = data if isinstance(data, list) else data.get("tasks", [])
            for task in tasks:
                if task.get("status") == "DONE":
                    summary["ongoing"]["done"].append(task.get("title", "?"))
                elif task.get("status") in ("WIP", "WAIT"):
                    summary["ongoing"]["active"].append(task.get("title", "?"))
        except (json.JSONDecodeError, Exception):
            pass

    # 读取健康度历史
    state_path = WORKSPACE / "state.json"
    if state_path.exists():
        try:
            state = json.loads(state_path.read_text(encoding="utf-8"))
            today_str = (datetime.now() - timedelta(days=1)).strftime("%Y-%m-%d")
            for h in state.get("health_history", []):
                if h.get("time", "").startswith(today_str):
                    summary["health"]["scores"].append(h["score"])
            if summary["health"]["scores"]:
                summary["health"]["average"] = round(
                    sum(summary["health"]["scores"]) / len(summary["health"]["scores"]), 1
                )
        except (json.JSONDecodeError, Exception):
            pass

    return summary


def generate_daily_report(summary: dict) -> str:
    """生成日报邮件内容"""
    try:
        from jinja2 import Environment, FileSystemLoader
        env = Environment(loader=FileSystemLoader(str(TEMPLATES)))
        template = env.get_template("email_review.j2")
        return template.render(summary=summary)
    except ImportError:
        logger.info("jinja2 不可用，使用纯文本日报")
    except Exception as e:
        logger.warning("模板渲染失败，回退纯文本: %s", e)

    # 纯文本回退
    lines = []
    lines.append(f"📋 EVA 日报 | {summary['date']}")
    lines.append("=" * 40)
    lines.append("")

    # Daily
    d = summary["daily"]
    lines.append(f"## 例行任务 [{d['done']}/{d['total']}]")
    for item in d["items"]:
        mark = "✅" if item["done"] else "❌"
        lines.append(f"  {mark} {item['text']}")
    lines.append("")

    # Ongoing
    o = summary["ongoing"]
    if o["done"]:
        lines.append("## 已完成任务")
        for t in o["done"]:
            lines.append(f"  ✅ {t}")
        lines.append("")
    if o["active"]:
        lines.append("## 进行中任务")
        for t in o["active"]:
            lines.append(f"  🔄 {t}")
        lines.append("")

    # Health
    h = summary["health"]
    if h["scores"]:
        lines.append(f"## 系统健康度")
        lines.append(f"  平均分: {h['average']}")
        lines.append(f"  心跳次数: {len(h['scores'])}")
        lines.append(f"  分数范围: {min(h['scores'])} - {max(h['scores'])}")
    lines.append("")
    lines.append("---")
    lines.append("Generated by EVA Heartbeat Manager")

    return "\n".join(lines)


def reset_daily():
    """
    执行每日重置:
    1. 收集昨日摘要
    2. 发送日报邮件
    3. 重置 daily.md
    4. 清理已完成的 ongoing 任务

    返回:
        {"report_sent": bool, "daily_reset": bool, "cleanup_count": int, "error": str | None}
    """
    result = {"report_sent": False, "daily_reset": False, "cleanup_count": 0, "error": None}

    try:
        # 1. 收集昨日摘要
        summary = _collect_yesterday_summary()

        # 2. 生成并发送日报
        report = generate_daily_report(summary)
        from tools.mail import send_mail
        subject = f"[EVA日报] {summary['date']} | daily:{summary['daily']['done']}/{summary['daily']['total']}"
        result["report_sent"] = send_mail(subject, report)

        # 3. 重置 daily.md
        template = _load_daily_template()
        today = datetime.now().strftime("%Y-%m-%d")
        new_content = template.replace("{{ date }}", today)

        daily_path = WORKSPACE / "daily.md"
        tmp = daily_path.with_suffix(".tmp")
        tmp.write_text(new_content, encoding="utf-8")
        tmp.rename(daily_path)
        result["daily_reset"] = True
        logger.info("daily.md 已重置")

        # 4. 清理已完成的 ongoing 任务（标记为 DONE 的移除或归档）
        ongoing_path = WORKSPACE / "ongoing.json"
        if ongoing_path.exists():
            data = json.loads(ongoing_path.read_text(encoding="utf-8"))
            tasks = data if isinstance(data, list) else data.get("tasks", [])
            active_tasks = [t for t in tasks if t.get("status") != "DONE"]
            removed = len(tasks) - len(active_tasks)
            result["cleanup_count"] = removed

            if removed > 0:
                tmp = ongoing_path.with_suffix(".tmp")
                tmp.write_text(
                    json.dumps({"tasks": active_tasks}, ensure_ascii=False, indent=2),
                    encoding="utf-8",
                )
                tmp.rename(ongoing_path)
                logger.info("清理了 %d 个已完成 ongoing 任务", removed)

        # 5. 同步网站监控（Canvas + FSP → upcoming.md）
        try:
            from tools.site_monitor import run_sync
            sync_result = run_sync()
            result["site_sync"] = sync_result
            logger.info(
                "网站监控同步: +%d ~%d -%d",
                sync_result["added"], sync_result["updated"], sync_result["removed"],
            )
            if sync_result["errors"]:
                for err in sync_result["errors"]:
                    logger.warning("  同步错误: %s", err)
        except Exception as e:
            logger.warning("网站监控同步失败（非致命）: %s", e)
            result["site_sync"] = {"error": str(e)}

    except Exception as e:
        result["error"] = str(e)
        logger.error("每日重置异常: %s", e)

    return result
