---
name: jina-reader
description: Extract clean, readable markdown content from any URL using Jina Reader API. Use when you need to fetch and parse web pages without dealing with HTML, JavaScript rendering, or paywalls. Ideal for research, article summarization, content analysis, and working with search results from tavily-search, web_search, or searxng skills.
---

# Jina Reader

## Overview

Jina Reader provides clean markdown extraction from any URL, bypassing HTML complexity, JavaScript rendering, and many paywalls. It returns structured text content with metadata (title, URL, published time) that's perfect for AI analysis.

## Quick Start

### Extract markdown content
```bash
scripts/jina-reader.py <url>
```

### Extract with JSON metadata
```bash
scripts/jina-reader.py <url> --format json
```

### Save to file
```bash
scripts/jina-reader.py <url> -o output.md
```

## Core Operations

### 1. Basic Extraction

Extract clean markdown from any URL:
```bash
scripts/jina-reader.py https://example.com/article
```

**Returns:** Full markdown content including title, metadata headers, and structured text.

**Use when:** You need readable text from a webpage for summarization, analysis, or content processing.

### 2. JSON Format

Get structured data with metadata:
```bash
scripts/jina-reader.py https://example.com/article --format json
```

**Returns:**
```json
{
  "status": "success",
  "metadata": {
    "title": "Article Title",
    "url": "https://example.com/article",
    "published": "Mon, 10 Feb 2026 12:00:00 GMT"
  },
  "content": "Markdown content..."
}
```

**Use when:** You need programmatic access to metadata or want to integrate with other tools.

### 3. Shell Script Quick Access

For simple one-liners:
```bash
scripts/jina-reader.sh https://example.com/article
```

**Returns:** Raw markdown content directly to stdout.

**Use when:** Quick extraction without arguments or when piping to other commands.

## Usage Patterns

### Combining with Search

When using tavily-search, web_search, or searxng skills:
1. Get search results with relevant URLs
2. Extract content from top results using jina-reader
3. Process and summarize the extracted content

```bash
# Example workflow
URL="https://example.com/article"
scripts/jina-reader.py "$URL" --format json | jq -r '.content'
```

### Batch Processing

Extract from multiple URLs:
```bash
for url in $(cat urls.txt); do
  scripts/jina-reader.py "$url" -o "output/$(basename $url).md"
done
```

### Content Analysis

Pipe extracted content to analysis tools:
```bash
scripts/jina-reader.py https://example.com/article | wc -w
scripts/jina-reader.py https://example.com/article | grep -i "keyword"
```

## Options

### Python Script (`jina-reader.py`)

- `url` (required): The URL to extract content from
- `-f, --format`: Output format - `markdown` or `json` (default: markdown)
- `-t, --timeout`: Request timeout in seconds (default: 30)
- `-o, --output`: Save output to file instead of stdout

### Shell Script (`jina-reader.sh`)

- `url` (required): The URL to extract content from

## Limitations

- **Timeout:** Default 30 seconds. Increase with `-t` for slow-loading pages
- **Rate limits:** Jina Reader API has rate limits. Use batching strategically
- **Dynamic content:** Can't extract content generated by client-side JavaScript after page load
- **Authentication:** Can't access pages requiring login or special headers

## Troubleshooting

### Timeout errors
```bash
scripts/jina-reader.py <url> -t 60  # Increase timeout
```

### Invalid URLs
The tool auto-prepends `https://` if missing. Use fully qualified URLs for reliability.

### Empty content
Some pages may block scraping. Try the shell script as fallback, or verify the URL is accessible.

## Resources

### scripts/jina-reader.py
Full-featured Python tool with JSON output, metadata extraction, and file saving.

### scripts/jina-reader.sh
Lightweight shell script for quick markdown extraction.
