#!/usr/bin/env bash
set -euo pipefail

# Hacker News CLI — curl + jq, parallel fetching, no auth needed
HN_API="https://hacker-news.firebaseio.com/v0"
ALGOLIA_API="https://hn.algolia.com/api/v1"
JSON_OUTPUT=false

usage() {
  cat <<'EOF'
Usage: hn.sh <command> [options]

Commands:
  top    [--limit N] [--json]          Top stories (default 10)
  new    [--limit N] [--json]          Newest stories
  best   [--limit N] [--json]          Best stories
  ask    [--limit N] [--json]          Ask HN stories
  show   [--limit N] [--json]          Show HN stories
  jobs   [--limit N] [--json]          Job postings
  item   ID [--json]                   Full item details
  comments ID [--limit N] [--depth D] [--json]  Top comments for a story
  user   USERNAME [--json]             User profile
  search QUERY [--limit N] [--sort relevance|date] [--type story|comment] [--period day|week|month|year] [--json]
  whoishiring [--limit N] [--json]     Latest "Who is hiring?" jobs
EOF
  exit 0
}

# --- Helpers ---

relative_time() {
  local ts="${1:-0}"
  local now
  now=$(date +%s)
  local diff=$(( now - ts ))
  if (( diff < 60 )); then echo "${diff}s ago"
  elif (( diff < 3600 )); then echo "$(( diff / 60 ))m ago"
  elif (( diff < 86400 )); then echo "$(( diff / 3600 ))h ago"
  elif (( diff < 2592000 )); then echo "$(( diff / 86400 ))d ago"
  elif (( diff < 31536000 )); then echo "$(( diff / 2592000 ))mo ago"
  else echo "$(( diff / 31536000 ))y ago"
  fi
}

html_to_text() {
  python3 -c "
import html, sys, re
text = html.unescape(sys.stdin.read())
text = re.sub(r'<br\s*/?>', '\n', text)
text = re.sub(r'<p>', '\n\n', text)
text = re.sub(r'<[^>]+>', '', text)
text = re.sub(r'\n{3,}', '\n\n', text)
print(text.strip())
"
}

fetch_item() {
  curl -sf "${HN_API}/item/${1}.json"
}

# Fetch multiple items in parallel, output JSON array
fetch_items_parallel() {
  local ids=("$@")
  local tmpdir
  tmpdir=$(mktemp -d)
  local pids=()

  for i in "${!ids[@]}"; do
    curl -sf "${HN_API}/item/${ids[$i]}.json" > "${tmpdir}/${i}.json" &
    pids+=($!)
  done

  for pid in "${pids[@]}"; do
    wait "$pid" 2>/dev/null || true
  done

  # Combine results in order
  echo "["
  local first=true
  for i in "${!ids[@]}"; do
    if [[ -s "${tmpdir}/${i}.json" ]]; then
      $first || echo ","
      first=false
      cat "${tmpdir}/${i}.json"
    fi
  done
  echo "]"

  rm -rf "$tmpdir"
}

# --- Story list commands ---

cmd_stories() {
  local endpoint="$1"; shift
  local limit=10

  while [[ $# -gt 0 ]]; do
    case "$1" in
      --limit) limit="$2"; shift 2 ;;
      --json) JSON_OUTPUT=true; shift ;;
      *) echo "Unknown option: $1"; exit 1 ;;
    esac
  done

  local ids
  ids=$(curl -sf "${HN_API}/${endpoint}.json" | jq -r ".[0:${limit}][]")

  local id_array=()
  while IFS= read -r id; do
    [[ -n "$id" ]] && id_array+=("$id")
  done <<< "$ids"

  if [[ ${#id_array[@]} -eq 0 ]]; then
    echo "No stories found."
    exit 0
  fi

  local items_json
  items_json=$(fetch_items_parallel "${id_array[@]}")

  if $JSON_OUTPUT; then
    echo "$items_json" | jq .
    return
  fi

  # Table output
  local idx=0
  echo "$items_json" | jq -c '.[]' | while IFS= read -r item; do
    idx=$(( idx + 1 ))
    local title score by descendants url time_val item_type
    title=$(echo "$item" | jq -r '.title // "untitled"')
    score=$(echo "$item" | jq -r '.score // 0')
    by=$(echo "$item" | jq -r '.by // "unknown"')
    descendants=$(echo "$item" | jq -r '.descendants // 0')
    url=$(echo "$item" | jq -r '.url // ""')
    time_val=$(echo "$item" | jq -r '.time // 0')
    item_type=$(echo "$item" | jq -r '.type // "story"')
    local age
    age=$(relative_time "$time_val")

    printf "%2d. ▲ %-4s │ %s\n" "$idx" "$score" "$title"
    if [[ "$item_type" == "job" ]]; then
      printf "          │ 🏢 %s  ⏱ %s\n" "$by" "$age"
    else
      printf "          │ 👤 %s  💬 %s  ⏱ %s\n" "$by" "$descendants" "$age"
    fi
    if [[ -n "$url" && "$url" != "null" ]]; then
      printf "          │ 🔗 %s\n" "$url"
    fi
    echo ""
  done
}

# --- Item command ---

cmd_item() {
  local id="${1:-}"
  if [[ -z "$id" ]]; then
    echo "Usage: hn.sh item ID [--json]"
    exit 1
  fi
  shift

  while [[ $# -gt 0 ]]; do
    case "$1" in
      --json) JSON_OUTPUT=true; shift ;;
      *) echo "Unknown option: $1"; exit 1 ;;
    esac
  done

  local item
  item=$(fetch_item "$id")

  if [[ -z "$item" || "$item" == "null" ]]; then
    echo "Item $id not found."
    exit 1
  fi

  if $JSON_OUTPUT; then
    echo "$item" | jq .
    return
  fi

  local item_type title score by time_val url text descendants kids_count
  item_type=$(echo "$item" | jq -r '.type // "unknown"')
  title=$(echo "$item" | jq -r '.title // ""')
  score=$(echo "$item" | jq -r '.score // 0')
  by=$(echo "$item" | jq -r '.by // "unknown"')
  time_val=$(echo "$item" | jq -r '.time // 0')
  url=$(echo "$item" | jq -r '.url // ""')
  text=$(echo "$item" | jq -r '.text // ""')
  descendants=$(echo "$item" | jq -r '.descendants // 0')
  kids_count=$(echo "$item" | jq -r '.kids // [] | length')
  local age
  age=$(relative_time "$time_val")

  echo "═══════════════════════════════════════════════════"
  echo "  Type: ${item_type}  │  ID: ${id}  │  ${age}"
  echo "═══════════════════════════════════════════════════"
  [[ -n "$title" && "$title" != "null" ]] && echo "  📰 $title"
  echo "  👤 $by  │  ▲ $score  │  💬 $descendants comments ($kids_count direct)"
  [[ -n "$url" && "$url" != "null" ]] && echo "  🔗 $url"
  if [[ -n "$text" && "$text" != "null" ]]; then
    echo ""
    echo "$text" | html_to_text | sed 's/^/  /'
  fi
  echo "═══════════════════════════════════════════════════"
}

# --- Comments command ---

cmd_comments() {
  local id="${1:-}"
  if [[ -z "$id" ]]; then
    echo "Usage: hn.sh comments ID [--limit N] [--depth D] [--json]"
    exit 1
  fi
  shift

  local limit=5 depth=1

  while [[ $# -gt 0 ]]; do
    case "$1" in
      --limit) limit="$2"; shift 2 ;;
      --depth) depth="$2"; shift 2 ;;
      --json) JSON_OUTPUT=true; shift ;;
      *) echo "Unknown option: $1"; exit 1 ;;
    esac
  done

  _fetch_comments "$id" "$limit" "$depth" 0
}

_fetch_comments() {
  local parent_id="$1" limit="$2" max_depth="$3" current_depth="$4"
  local indent=""
  for (( i=0; i<current_depth; i++ )); do indent+="  "; done

  local parent
  parent=$(fetch_item "$parent_id")

  local kid_ids
  kid_ids=$(echo "$parent" | jq -r ".kids // [] | .[0:${limit}][]" 2>/dev/null)

  local id_array=()
  while IFS= read -r kid_id; do
    [[ -n "$kid_id" ]] && id_array+=("$kid_id")
  done <<< "$kid_ids"

  if [[ ${#id_array[@]} -eq 0 ]]; then
    [[ $current_depth -eq 0 ]] && echo "No comments found."
    return
  fi

  local comments_json
  comments_json=$(fetch_items_parallel "${id_array[@]}")

  if $JSON_OUTPUT && [[ $current_depth -eq 0 ]]; then
    echo "$comments_json" | jq .
    return
  fi

  echo "$comments_json" | jq -c '.[]' | while IFS= read -r comment; do
    local cby ctext ctime_val cid ckids_count deleted dead
    deleted=$(echo "$comment" | jq -r '.deleted // false')
    dead=$(echo "$comment" | jq -r '.dead // false')
    [[ "$deleted" == "true" || "$dead" == "true" ]] && continue

    cid=$(echo "$comment" | jq -r '.id')
    cby=$(echo "$comment" | jq -r '.by // "unknown"')
    ctext=$(echo "$comment" | jq -r '.text // ""')
    ctime_val=$(echo "$comment" | jq -r '.time // 0')
    ckids_count=$(echo "$comment" | jq -r '.kids // [] | length')
    local cage
    cage=$(relative_time "$ctime_val")

    echo "${indent}┌─ 👤 ${cby}  ⏱ ${cage}  (${ckids_count} replies)"
    if [[ -n "$ctext" && "$ctext" != "null" ]]; then
      echo "$ctext" | html_to_text | sed "s/^/${indent}│ /"
    fi
    echo "${indent}└─────────"
    echo ""

    if (( current_depth + 1 < max_depth && ckids_count > 0 )); then
      _fetch_comments "$cid" "$limit" "$max_depth" $(( current_depth + 1 ))
    fi
  done
}

# --- User command ---

cmd_user() {
  local username="${1:-}"
  if [[ -z "$username" ]]; then
    echo "Usage: hn.sh user USERNAME [--json]"
    exit 1
  fi
  shift

  while [[ $# -gt 0 ]]; do
    case "$1" in
      --json) JSON_OUTPUT=true; shift ;;
      *) echo "Unknown option: $1"; exit 1 ;;
    esac
  done

  local user
  user=$(curl -sf "${HN_API}/user/${username}.json")

  if [[ -z "$user" || "$user" == "null" ]]; then
    echo "User '$username' not found."
    exit 1
  fi

  if $JSON_OUTPUT; then
    echo "$user" | jq .
    return
  fi

  local karma created about submitted_count
  karma=$(echo "$user" | jq -r '.karma // 0')
  created=$(echo "$user" | jq -r '.created // 0')
  about=$(echo "$user" | jq -r '.about // ""')
  submitted_count=$(echo "$user" | jq -r '.submitted // [] | length')
  local age
  age=$(relative_time "$created")

  echo "═══════════════════════════════════════════════════"
  echo "  👤 $username"
  echo "═══════════════════════════════════════════════════"
  echo "  ⭐ Karma: $karma"
  echo "  📅 Joined: $age"
  echo "  📝 Submissions: $submitted_count"
  if [[ -n "$about" && "$about" != "null" ]]; then
    echo ""
    echo "  About:"
    echo "$about" | html_to_text | sed 's/^/  /'
  fi
  echo "═══════════════════════════════════════════════════"

  # Fetch a few recent submissions
  local recent_ids
  recent_ids=$(echo "$user" | jq -r '.submitted[:5][]' 2>/dev/null)
  local rid_array=()
  while IFS= read -r rid; do
    [[ -n "$rid" ]] && rid_array+=("$rid")
  done <<< "$recent_ids"

  if [[ ${#rid_array[@]} -gt 0 ]]; then
    echo ""
    echo "  Recent activity:"
    local recent_json
    recent_json=$(fetch_items_parallel "${rid_array[@]}")
    echo "$recent_json" | jq -c '.[]' | while IFS= read -r ritem; do
      local rtype rtitle rtime_val
      rtype=$(echo "$ritem" | jq -r '.type // "unknown"')
      rtitle=$(echo "$ritem" | jq -r '.title // ""')
      rtime_val=$(echo "$ritem" | jq -r '.time // 0')
      local rage
      rage=$(relative_time "$rtime_val")

      if [[ -n "$rtitle" && "$rtitle" != "null" ]]; then
        echo "    [$rtype] $rtitle ($rage)"
      else
        echo "    [$rtype] (id: $(echo "$ritem" | jq -r '.id')) ($rage)"
      fi
    done
  fi
}

# --- Search command ---

cmd_search() {
  local query="${1:-}"
  if [[ -z "$query" ]]; then
    echo "Usage: hn.sh search QUERY [--limit N] [--sort relevance|date] [--type story|comment] [--period day|week|month|year] [--json]"
    exit 1
  fi
  shift

  local limit=10 sort="relevance" type="" period=""

  while [[ $# -gt 0 ]]; do
    case "$1" in
      --limit) limit="$2"; shift 2 ;;
      --sort) sort="$2"; shift 2 ;;
      --type) type="$2"; shift 2 ;;
      --period) period="$2"; shift 2 ;;
      --json) JSON_OUTPUT=true; shift ;;
      *) echo "Unknown option: $1"; exit 1 ;;
    esac
  done

  local endpoint="search"
  [[ "$sort" == "date" ]] && endpoint="search_by_date"

  local tags=""
  [[ -n "$type" ]] && tags="&tags=${type}"

  local numeric=""
  if [[ -n "$period" ]]; then
    local now
    now=$(date +%s)
    local since=0
    case "$period" in
      day)   since=$(( now - 86400 )) ;;
      week)  since=$(( now - 604800 )) ;;
      month) since=$(( now - 2592000 )) ;;
      year)  since=$(( now - 31536000 )) ;;
    esac
    numeric="&numericFilters=created_at_i%3E${since}"
  fi

  local encoded_query
  encoded_query=$(python3 -c "import urllib.parse,sys; print(urllib.parse.quote(sys.argv[1]))" "$query")

  local result
  result=$(curl -sf "${ALGOLIA_API}/${endpoint}?query=${encoded_query}&hitsPerPage=${limit}${tags}${numeric}")

  if $JSON_OUTPUT; then
    echo "$result" | jq .
    return
  fi

  local total
  total=$(echo "$result" | jq -r '.nbHits // 0')
  echo "Search: \"$query\" — $total results"
  echo ""

  echo "$result" | jq -c '.hits[]' | while IFS= read -r hit; do
    local htitle hauthor hpoints hcomments hurl hcreated_at
    htitle=$(echo "$hit" | jq -r '.title // .story_title // ""')
    hauthor=$(echo "$hit" | jq -r '.author // "unknown"')
    hpoints=$(echo "$hit" | jq -r '.points // 0')
    hcomments=$(echo "$hit" | jq -r '.num_comments // 0')
    hurl=$(echo "$hit" | jq -r '.url // ""')
    hcreated_at=$(echo "$hit" | jq -r '.created_at // ""')

    if [[ -n "$htitle" && "$htitle" != "null" ]]; then
      printf "  ▲ %-4s │ %s\n" "$hpoints" "$htitle"
      printf "         │ 👤 %s  💬 %s  📅 %s\n" "$hauthor" "$hcomments" "${hcreated_at%%T*}"
      [[ -n "$hurl" && "$hurl" != "null" ]] && printf "         │ 🔗 %s\n" "$hurl"
    else
      # Comment result
      local stext
      stext=$(echo "$hit" | jq -r '.comment_text // .story_text // ""' | head -c 200)
      [[ -n "$stext" ]] && stext=$(echo "$stext" | html_to_text | head -1)
      printf "  💬 %s: %s\n" "$hauthor" "$stext"
      printf "     📅 %s\n" "${hcreated_at%%T*}"
    fi
    echo ""
  done
}

# --- Who is hiring command ---

cmd_whoishiring() {
  local limit=10

  while [[ $# -gt 0 ]]; do
    case "$1" in
      --limit) limit="$2"; shift 2 ;;
      --json) JSON_OUTPUT=true; shift ;;
      *) echo "Unknown option: $1"; exit 1 ;;
    esac
  done

  # Find latest "Who is hiring?" thread
  local search_result
  search_result=$(curl -sf "${ALGOLIA_API}/search_by_date?query=who+is+hiring&tags=story,author_whoishiring&hitsPerPage=1")

  local thread_id thread_title
  thread_id=$(echo "$search_result" | jq -r '.hits[0].objectID // ""')
  thread_title=$(echo "$search_result" | jq -r '.hits[0].title // "Who is hiring?"')

  if [[ -z "$thread_id" ]]; then
    echo "Could not find a 'Who is hiring?' thread."
    exit 1
  fi

  echo "📋 $thread_title"
  echo "   Thread ID: $thread_id"
  echo "   https://news.ycombinator.com/item?id=$thread_id"
  echo ""

  # Fetch thread to get kid IDs
  local thread
  thread=$(fetch_item "$thread_id")
  local kid_ids
  kid_ids=$(echo "$thread" | jq -r ".kids // [] | .[0:${limit}][]")

  local id_array=()
  while IFS= read -r kid_id; do
    [[ -n "$kid_id" ]] && id_array+=("$kid_id")
  done <<< "$kid_ids"

  if [[ ${#id_array[@]} -eq 0 ]]; then
    echo "No job postings found."
    return
  fi

  local jobs_json
  jobs_json=$(fetch_items_parallel "${id_array[@]}")

  if $JSON_OUTPUT; then
    echo "$jobs_json" | jq .
    return
  fi

  local idx=0
  echo "$jobs_json" | jq -c '.[]' | while IFS= read -r job; do
    local deleted dead
    deleted=$(echo "$job" | jq -r '.deleted // false')
    dead=$(echo "$job" | jq -r '.dead // false')
    [[ "$deleted" == "true" || "$dead" == "true" ]] && continue

    idx=$(( idx + 1 ))
    local jby jtext jtime_val
    jby=$(echo "$job" | jq -r '.by // "unknown"')
    jtext=$(echo "$job" | jq -r '.text // ""')
    jtime_val=$(echo "$job" | jq -r '.time // 0')
    local jage
    jage=$(relative_time "$jtime_val")

    # Extract first line as summary
    local first_line
    first_line=$(echo "$jtext" | html_to_text | head -1 | head -c 120)

    printf "%2d. 🏢 %s  ⏱ %s\n" "$idx" "$jby" "$jage"
    echo "    $first_line"
    echo ""
  done
}

# --- Main dispatch ---

[[ $# -eq 0 ]] && usage

cmd="$1"; shift

case "$cmd" in
  top)          cmd_stories "topstories" "$@" ;;
  new)          cmd_stories "newstories" "$@" ;;
  best)         cmd_stories "beststories" "$@" ;;
  ask)          cmd_stories "askstories" "$@" ;;
  show)         cmd_stories "showstories" "$@" ;;
  jobs)         cmd_stories "jobstories" "$@" ;;
  item)         cmd_item "$@" ;;
  comments)     cmd_comments "$@" ;;
  user)         cmd_user "$@" ;;
  search)       cmd_search "$@" ;;
  whoishiring)  cmd_whoishiring "$@" ;;
  help|--help|-h) usage ;;
  *)            echo "Unknown command: $cmd"; usage ;;
esac
