#!/usr/bin/env node
/**
 * Grokipedia Search
 * Search for articles on grokipedia.com
 * 
 * Usage: node search.mjs "query" [--limit N]
 */

const args = process.argv.slice(2);

// Parse arguments
let query = null;
let limit = 10;

for (let i = 0; i < args.length; i++) {
  if (args[i] === '--limit' && args[i + 1]) {
    limit = Math.min(50, Math.max(1, parseInt(args[i + 1], 10) || 10));
    i++;
  } else if (!args[i].startsWith('--')) {
    query = args[i];
  }
}

if (!query) {
  console.error('Usage: node search.mjs "query" [--limit N]');
  console.error('');
  console.error('Options:');
  console.error('  --limit N    Max results (1-50, default: 10)');
  console.error('');
  console.error('Example:');
  console.error('  node search.mjs "artificial intelligence" --limit 5');
  process.exit(1);
}

// Input validation: strip control characters, enforce reasonable length
query = query.replace(/[\x00-\x1f\x7f]/g, '').trim();
if (query.length === 0 || query.length > 200) {
  console.error('Error: Query must be 1-200 characters (no control characters).');
  process.exit(1);
}

async function search(query, limit) {
  const url = new URL('https://grokipedia.com/api/typeahead');
  url.searchParams.set('query', query);
  url.searchParams.set('limit', String(limit));

  const response = await fetch(url.toString(), {
    headers: {
      'Accept': 'application/json',
      'User-Agent': 'Mozilla/5.0 (compatible; GrokipediaParser/1.0)'
    }
  });

  if (!response.ok) {
    throw new Error(`Search failed: ${response.status} ${response.statusText}`);
  }

  const data = await response.json();
  
  return {
    query,
    resultCount: data.results?.length || 0,
    searchTimeMs: data.searchTimeMs,
    results: (data.results || []).map(r => ({
      slug: r.slug,
      title: r.title,
      snippet: r.snippet?.replace(/\n+/g, ' ').substring(0, 200) + (r.snippet?.length > 200 ? '...' : ''),
      relevanceScore: r.relevanceScore,
      viewCount: r.viewCount || 0
    }))
  };
}

try {
  const result = await search(query, limit);
  console.log(JSON.stringify(result, null, 2));
} catch (error) {
  console.error('Error:', error.message);
  process.exit(1);
}
