#!/usr/bin/env node
/**
 * Grokipedia Article Fetcher
 * Fetch and parse articles from grokipedia.com
 * 
 * Usage: node fetch.mjs "Article_Slug" [--raw]
 */

import { JSDOM } from 'jsdom';
import { Readability } from '@mozilla/readability';

const args = process.argv.slice(2);

// Parse arguments
let slug = null;
let raw = false;

for (const arg of args) {
  if (arg === '--raw') {
    raw = true;
  } else if (!arg.startsWith('--')) {
    slug = arg;
  }
}

if (!slug) {
  console.error('Usage: node fetch.mjs "Article_Slug" [--raw]');
  console.error('');
  console.error('Options:');
  console.error('  --raw    Output raw HTML instead of markdown');
  console.error('');
  console.error('Examples:');
  console.error('  node fetch.mjs "Helsinki"');
  console.error('  node fetch.mjs "Artificial_intelligence"');
  process.exit(1);
}

// Input validation: slugs should only contain word characters, hyphens, dots, and percent-encoded sequences
slug = slug.trim();
if (!/^[\w\-.%]+$/.test(slug) || slug.length > 200) {
  console.error('Error: Invalid slug. Use alphanumeric characters, underscores, hyphens only (max 200 chars).');
  process.exit(1);
}

function htmlToMarkdown(html) {
  // Simple HTML to Markdown converter
  let md = html;
  
  // Headers
  md = md.replace(/<h1[^>]*>(.*?)<\/h1>/gi, '# $1\n\n');
  md = md.replace(/<h2[^>]*>(.*?)<\/h2>/gi, '## $1\n\n');
  md = md.replace(/<h3[^>]*>(.*?)<\/h3>/gi, '### $1\n\n');
  md = md.replace(/<h4[^>]*>(.*?)<\/h4>/gi, '#### $1\n\n');
  md = md.replace(/<h5[^>]*>(.*?)<\/h5>/gi, '##### $1\n\n');
  md = md.replace(/<h6[^>]*>(.*?)<\/h6>/gi, '###### $1\n\n');
  
  // Links
  md = md.replace(/<a[^>]*href="([^"]*)"[^>]*>(.*?)<\/a>/gi, '[$2]($1)');
  
  // Bold and italic
  md = md.replace(/<strong[^>]*>(.*?)<\/strong>/gi, '**$1**');
  md = md.replace(/<b[^>]*>(.*?)<\/b>/gi, '**$1**');
  md = md.replace(/<em[^>]*>(.*?)<\/em>/gi, '*$1*');
  md = md.replace(/<i[^>]*>(.*?)<\/i>/gi, '*$1*');
  
  // Lists
  md = md.replace(/<li[^>]*>(.*?)<\/li>/gi, '- $1\n');
  md = md.replace(/<\/?[uo]l[^>]*>/gi, '\n');
  
  // Paragraphs and breaks
  md = md.replace(/<p[^>]*>(.*?)<\/p>/gi, '$1\n\n');
  md = md.replace(/<br\s*\/?>/gi, '\n');
  
  // Blockquotes
  md = md.replace(/<blockquote[^>]*>(.*?)<\/blockquote>/gis, (_, content) => {
    return content.split('\n').map(line => `> ${line}`).join('\n') + '\n\n';
  });
  
  // Code
  md = md.replace(/<code[^>]*>(.*?)<\/code>/gi, '`$1`');
  md = md.replace(/<pre[^>]*>(.*?)<\/pre>/gis, '```\n$1\n```\n\n');
  
  // Remove remaining HTML tags
  md = md.replace(/<[^>]+>/g, '');
  
  // Decode HTML entities
  md = md.replace(/&amp;/g, '&');
  md = md.replace(/&lt;/g, '<');
  md = md.replace(/&gt;/g, '>');
  md = md.replace(/&quot;/g, '"');
  md = md.replace(/&#39;/g, "'");
  md = md.replace(/&nbsp;/g, ' ');
  
  // Clean up whitespace
  md = md.replace(/\n{3,}/g, '\n\n');
  md = md.trim();
  
  return md;
}

async function fetchArticle(slug) {
  const url = `https://grokipedia.com/page/${encodeURIComponent(slug)}`;
  
  const response = await fetch(url, {
    headers: {
      'Accept': 'text/html',
      'User-Agent': 'Mozilla/5.0 (compatible; GrokipediaParser/1.0)'
    }
  });

  if (!response.ok) {
    if (response.status === 404) {
      throw new Error(`Article not found: ${slug}`);
    }
    throw new Error(`Fetch failed: ${response.status} ${response.statusText}`);
  }

  const html = await response.text();
  
  // Use JSDOM + Readability for extraction
  const dom = new JSDOM(html, { url });
  const reader = new Readability(dom.window.document);
  const article = reader.parse();
  
  if (!article) {
    throw new Error('Failed to parse article content');
  }

  const markdown = raw ? article.content : htmlToMarkdown(article.content);
  
  return {
    slug,
    title: article.title,
    url,
    content: markdown
  };
}

try {
  const result = await fetchArticle(slug);
  
  // Output just the content for LLM consumption
  console.log(`# ${result.title}\n`);
  console.log(`*Source: ${result.url}*\n`);
  console.log(result.content);
} catch (error) {
  console.error('Error:', error.message);
  process.exit(1);
}
