/**
 * Integration and adversarial tests for the full governance pipeline.
 * Covers T-007, T-008, T-009, I-001..I-006, A-001..A-005 from the spec.
 */

import { describe, it } from "node:test";
import assert from "node:assert/strict";
import { propose, promote, pipeline, resolveEscalation, verify } from "../scripts/governance.js";
import { createIntent } from "../scripts/intent.js";
import { evaluate, computeVerdictHash } from "../scripts/policy-engine.js";
import { parsePolicyFile } from "../scripts/yaml-parse.js";
import type { Verdict } from "../scripts/policy-engine.js";
import type { GovernanceConfig } from "../scripts/governance.js";
import { mkdtempSync, writeFileSync } from "node:fs";
import { join } from "node:path";
import { tmpdir } from "node:os";

// ── Helpers ──────────────────────────────────────────────────────────

function tmpDir(): string {
  return mkdtempSync(join(tmpdir(), "governance-integ-"));
}

function makeConfig(dir: string, policyContent?: string): GovernanceConfig {
  const policyPath = join(dir, "policy.yaml");
  const witnessPath = join(dir, "witness.jsonl");

  if (policyContent) {
    writeFileSync(policyPath, policyContent, "utf8");
  }

  return {
    policyPath,
    witnessPath,
    maxVerdictAge: 30,
    escalationTimeout: 300,
  };
}

const STANDARD_POLICY = `version: "0.1"
default_verdict: deny
rules:
  - name: allow-read-workspace
    match:
      action_type: read
      target_pattern: "./**"
    verdict: approve
    reason: "Workspace reads permitted"
  - name: escalate-network
    match:
      action_type: network
    verdict: escalate
    reason: "Network requires approval"
  - name: block-delete-shell
    match:
      action_type: delete
      tool_pattern: "shell.*"
    verdict: deny
    reason: "Destructive shell commands blocked"
  - name: block-profile-creation
    match:
      action_type: create
      data_scope:
        - personal
        - identity
    verdict: deny
    reason: "Profile creation not authorized"
sensitive_data:
  - category: credentials
    patterns:
      - "**/*.env"
    action: deny`;

// ── T-007: Mismatched intent_hash → PROMOTE false ────────────────────

describe("T-007: intent_hash mismatch", () => {
  it("rejects promote when intent_hash does not match verdict", () => {
    const intent = createIntent({
      skill: "test", tool: "bash", model: "claude",
      actionType: "read", target: "./file.txt",
      parameters: {}, dataScope: [],
      conversationId: "c1", messageId: "m1", userInstruction: "read",
    });

    const verdict: Verdict = {
      decision: "approve",
      intent_hash: "wrong_hash_" + "0".repeat(53),
      rule_matched: "test-rule",
      reason: "test",
      timestamp: new Date().toISOString(),
      verdict_hash: "0".repeat(64),
    };

    assert.equal(promote(intent, verdict, 30), false);
  });
});

// ── T-008: Expired verdict → PROMOTE false ───────────────────────────

describe("T-008: verdict expiry", () => {
  it("rejects verdict older than MAX_VERDICT_AGE", () => {
    const intent = createIntent({
      skill: "test", tool: "bash", model: "claude",
      actionType: "read", target: "./file.txt",
      parameters: {}, dataScope: [],
      conversationId: "c1", messageId: "m1", userInstruction: "read",
    });

    // Verdict from 60 seconds ago
    const oldTime = new Date(Date.now() - 60_000).toISOString();
    const verdict: Verdict = {
      decision: "approve",
      intent_hash: intent.intent_hash,
      rule_matched: "test-rule",
      reason: "test",
      timestamp: oldTime,
      verdict_hash: computeVerdictHash("approve", intent.intent_hash, "test-rule", oldTime),
    };

    assert.equal(promote(intent, verdict, 30), false);
  });

  it("accepts verdict within MAX_VERDICT_AGE", () => {
    const intent = createIntent({
      skill: "test", tool: "bash", model: "claude",
      actionType: "read", target: "./file.txt",
      parameters: {}, dataScope: [],
      conversationId: "c1", messageId: "m1", userInstruction: "read",
    });

    const now = new Date().toISOString();
    const verdict: Verdict = {
      decision: "approve",
      intent_hash: intent.intent_hash,
      rule_matched: "test-rule",
      reason: "test",
      timestamp: now,
      verdict_hash: computeVerdictHash("approve", intent.intent_hash, "test-rule", now),
    };

    assert.equal(promote(intent, verdict, 30), true);
  });
});

// ── I-001: Shell rm -rf blocked ──────────────────────────────────────

describe("I-001: destructive shell blocked", () => {
  it("blocks shell delete with standard policy", async () => {
    const dir = tmpDir();
    const config = makeConfig(dir, STANDARD_POLICY);

    const result = await pipeline({
      skill: "shell",
      tool: "rm",
      model: "claude",
      actionType: "delete",
      target: "/home/user/important",
      parameters: { flags: "-rf" },
      dataScope: [],
      conversationId: "c1",
      messageId: "m1",
      userInstruction: "clean up files",
    }, config);

    assert.equal(result.verdict.decision, "deny");
    assert.equal(result.promoted, false);
    assert.equal(result.record.execution_result.status, "blocked");
  });
});

// ── I-002: Local file read approved ──────────────────────────────────

describe("I-002: local read approved", () => {
  it("approves workspace file read with standard policy", async () => {
    const dir = tmpDir();
    const config = makeConfig(dir, STANDARD_POLICY);

    const result = await pipeline({
      skill: "editor",
      tool: "read",
      model: "claude",
      actionType: "read",
      target: "./src/main.ts",
      parameters: {},
      dataScope: [],
      conversationId: "c1",
      messageId: "m1",
      userInstruction: "read the main file",
    }, config);

    assert.equal(result.verdict.decision, "approve");
    assert.equal(result.promoted, true);
    assert.equal(result.record.execution_result.status, "executed");
  });
});

// ── I-003: External HTTP escalated ───────────────────────────────────

describe("I-003: network escalated", () => {
  it("escalates external network request", async () => {
    const dir = tmpDir();
    const config = makeConfig(dir, STANDARD_POLICY);

    const result = await pipeline({
      skill: "browser",
      tool: "fetch",
      model: "claude",
      actionType: "network",
      target: "https://api.example.com",
      parameters: {},
      dataScope: [],
      conversationId: "c1",
      messageId: "m1",
      userInstruction: "fetch data",
    }, config);

    assert.equal(result.verdict.decision, "escalate");
    assert.equal(result.promoted, false);
    assert.equal(result.record.execution_result.status, "escalated");
  });
});

// ── I-004: Profile creation blocked by data_scope ────────────────────

describe("I-004: profile creation blocked", () => {
  it("blocks create action with personal data scope", async () => {
    const dir = tmpDir();
    const config = makeConfig(dir, STANDARD_POLICY);

    const result = await pipeline({
      skill: "moltmatch",
      tool: "create-profile",
      model: "claude",
      actionType: "create",
      target: "api.moltmatch.com",
      parameters: { name: "Test User" },
      dataScope: ["personal", "identity"],
      conversationId: "c1",
      messageId: "m1",
      userInstruction: "create a dating profile",
    }, config);

    assert.equal(result.verdict.decision, "deny");
    assert.equal(result.promoted, false);
  });
});

// ── I-005: Policy hot-reload ─────────────────────────────────────────

describe("I-005: policy update", () => {
  it("uses updated policy after file change", async () => {
    const dir = tmpDir();
    const config = makeConfig(dir, STANDARD_POLICY);

    // First call: network is escalated
    const result1 = await pipeline({
      skill: "browser", tool: "fetch", model: "claude",
      actionType: "network", target: "https://api.example.com",
      parameters: {}, dataScope: [],
      conversationId: "c1", messageId: "m1", userInstruction: "fetch",
    }, config);
    assert.equal(result1.verdict.decision, "escalate");

    // Update policy to approve network
    const newPolicy = `version: "0.1"
default_verdict: deny
rules:
  - name: allow-network
    match:
      action_type: network
    verdict: approve
    reason: "Network now approved"`;
    writeFileSync(config.policyPath, newPolicy, "utf8");

    // Second call: network should be approved
    const result2 = await pipeline({
      skill: "browser", tool: "fetch", model: "claude",
      actionType: "network", target: "https://api.example.com",
      parameters: {}, dataScope: [],
      conversationId: "c1", messageId: "m1", userInstruction: "fetch",
    }, config);
    assert.equal(result2.verdict.decision, "approve");
  });
});

// ── I-006: Missing policy → fail-closed ──────────────────────────────

describe("I-006: fail-closed on missing policy", () => {
  it("denies everything when policy file is missing", async () => {
    const dir = tmpDir();
    const config = makeConfig(dir); // No policy file written

    const result = await pipeline({
      skill: "test", tool: "bash", model: "claude",
      actionType: "read", target: "./README.md",
      parameters: {}, dataScope: [],
      conversationId: "c1", messageId: "m1", userInstruction: "read",
    }, config);

    assert.equal(result.verdict.decision, "deny");
    assert.equal(result.promoted, false);
  });
});

// ── A-001: Spoofed intent_hash detected ──────────────────────────────

describe("A-001: spoofed intent_hash", () => {
  it("recomputes hash and detects mismatch", () => {
    const raw = {
      id: "fake-id",
      timestamp: "2026-01-01T00:00:00Z",
      source: { skill: "evil", tool: "hack", model: "gpt" },
      action: { type: "execute", target: "/bin/sh", parameters: {}, data_scope: [] },
      context: { conversation_id: "c", message_id: "m", user_instruction: "hack" },
      intent_hash: "0000000000000000000000000000000000000000000000000000000000000000",
    };

    // propose() should reject or recompute
    // When passed a full intent with wrong hash, validation should fail
    assert.throws(() => propose(raw), /Hash does not match/);
  });
});

// ── A-003: Agent attempts to modify policy file ──────────────────────

describe("A-003: policy file protection", () => {
  it("write to policy path is blocked by sensitive data rule", () => {
    // Standard policy should deny writes to governance config
    const policy = parsePolicyFile(STANDARD_POLICY);
    const intent = createIntent({
      skill: "editor", tool: "write", model: "claude",
      actionType: "write", target: "~/.openclaw/governance/policy.yaml",
      parameters: {}, dataScope: [],
      conversationId: "c1", messageId: "m1", userInstruction: "modify policy",
    });

    const verdict = evaluate(intent, policy, new Date().toISOString());
    // Should be denied by default (no write rule in standard policy matches this path)
    assert.equal(verdict.decision, "deny");
  });
});

// ── A-005: Replay attack prevention ──────────────────────────────────

describe("A-005: replay attack", () => {
  it("rejects resubmitting approved verdict for different intent", () => {
    const intent1 = createIntent({
      skill: "test", tool: "bash", model: "claude",
      actionType: "read", target: "./safe.txt",
      parameters: {}, dataScope: [],
      conversationId: "c1", messageId: "m1", userInstruction: "read",
    });

    const intent2 = createIntent({
      skill: "evil", tool: "rm", model: "claude",
      actionType: "delete", target: "/etc/passwd",
      parameters: {}, dataScope: [],
      conversationId: "c1", messageId: "m2", userInstruction: "delete",
    });

    const now = new Date().toISOString();
    const verdict: Verdict = {
      decision: "approve",
      intent_hash: intent1.intent_hash,
      rule_matched: "allow-reads",
      reason: "Safe read",
      timestamp: now,
      verdict_hash: computeVerdictHash("approve", intent1.intent_hash, "allow-reads", now),
    };

    // Verdict for intent1 should not promote intent2
    assert.equal(promote(intent1, verdict, 30), true);  // Original: works
    assert.equal(promote(intent2, verdict, 30), false);  // Replay: rejected
  });
});

// ── Escalation resolution ────────────────────────────────────────────

describe("escalation resolution", () => {
  it("resolves escalation with approve → user_approved", async () => {
    const dir = tmpDir();
    const config = makeConfig(dir, STANDARD_POLICY);

    // Create an escalated action
    const result = await pipeline({
      skill: "browser", tool: "fetch", model: "claude",
      actionType: "network", target: "https://api.example.com",
      parameters: {}, dataScope: [],
      conversationId: "c1", messageId: "m1", userInstruction: "fetch",
    }, config);

    assert.equal(result.record.execution_result.status, "escalated");

    // Resolve with approve
    const resolved = await resolveEscalation(result.intent.id, "approve", config);
    assert.equal(resolved.execution_result.status, "user_approved");
  });

  it("resolves escalation with deny → user_denied", async () => {
    const dir = tmpDir();
    const config = makeConfig(dir, STANDARD_POLICY);

    const result = await pipeline({
      skill: "browser", tool: "fetch", model: "claude",
      actionType: "network", target: "https://api.example.com",
      parameters: {}, dataScope: [],
      conversationId: "c1", messageId: "m1", userInstruction: "fetch",
    }, config);

    const resolved = await resolveEscalation(result.intent.id, "deny", config);
    assert.equal(resolved.execution_result.status, "user_denied");
  });
});

// ── Witness chain integrity across pipeline operations ───────────────

describe("witness chain integrity", () => {
  it("maintains valid chain across 10 operations", async () => {
    const dir = tmpDir();
    const config = makeConfig(dir, STANDARD_POLICY);

    for (let i = 0; i < 10; i++) {
      await pipeline({
        skill: "test", tool: "read", model: "claude",
        actionType: "read", target: `./file${i}.txt`,
        parameters: {}, dataScope: [],
        conversationId: "c1", messageId: `m${i}`, userInstruction: `read file ${i}`,
      }, config);
    }

    const result = await verify(config);
    assert.equal(result.valid, true);
    assert.equal(result.totalRecords, 10);
  });
});
