#!/bin/bash
# update-cache.sh - Fetch and cache all Gousto recipes

set -e

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
DATA_DIR="$SCRIPT_DIR/../data"
CACHE_FILE="$DATA_DIR/recipes.json"
TEMP_FILE="$DATA_DIR/.recipes-temp.jsonl"

API_URL="https://production-api.gousto.co.uk/cmsreadbroker/v1/recipes"
LIMIT=50

mkdir -p "$DATA_DIR"

# Get total count
echo "Checking API..."
first=$(curl -sS "$API_URL?limit=1" 2>/dev/null)
total=$(echo "$first" | jq -r '.data.count // 0')

if [[ "$total" -eq 0 ]]; then
    echo "Error: Could not reach Gousto API"
    exit 1
fi

echo "Fetching $total recipes..."

# Clear temp file
> "$TEMP_FILE"

offset=0
fetched=0
errors=0

while [[ $offset -lt $total ]]; do
    response=$(curl -sS "$API_URL?limit=$LIMIT&offset=$offset" 2>/dev/null)
    
    # Check valid response
    if echo "$response" | jq -e '.data.entries' &>/dev/null; then
        # Extract and append each recipe as JSONL
        echo "$response" | jq -c '.data.entries[] | {
            title,
            slug: (.url | split("/") | last),
            rating: (.rating.average // 0),
            rating_count: (.rating.count // 0),
            prep_time: (.prep_times.for_2 // .prep_times.for_4 // 0),
            uid
        }' >> "$TEMP_FILE"
        
        count=$(echo "$response" | jq '.data.entries | length')
        fetched=$((fetched + count))
        printf "\r  %d / %d recipes" "$fetched" "$total"
    else
        ((errors++))
        if [[ $errors -gt 10 ]]; then
            echo ""
            echo "Error: Too many failed requests"
            rm -f "$TEMP_FILE"
            exit 1
        fi
        sleep 1
        continue
    fi
    
    offset=$((offset + LIMIT))
    sleep 0.2
done

echo ""
echo "Deduplicating..."

# Convert JSONL to array, dedupe by slug, sort by title
jq -s 'unique_by(.slug) | sort_by(.title | ascii_downcase)' "$TEMP_FILE" > "$CACHE_FILE"
rm -f "$TEMP_FILE"

final=$(jq 'length' "$CACHE_FILE")
echo "✓ Cached $final unique recipes"
