#!/bin/bash
# recipe.sh - Get full recipe details from official Gousto API

set -e

if [[ $# -eq 0 ]]; then
    echo "Usage: $0 <recipe-slug>"
    echo "Example: $0 honey-soy-chicken-with-noodles"
    echo ""
    echo "Find slugs using: ./search.sh <term>"
    exit 1
fi

slug="$1"
API_URL="https://production-api.gousto.co.uk/cmsreadbroker/v1/recipe/${slug}"

response=$(curl -sS --fail "$API_URL" 2>/dev/null) || {
    echo "Error: Failed to fetch recipe '$slug'"
    echo "Recipe may not exist or API may be unavailable."
    exit 1
}

# Output as clean JSON with the important fields
echo "$response" | jq '{
    title: .data.entry.title,
    slug: (.data.entry.url | split("/") | last),
    rating: .data.entry.rating.average,
    prep_time: (.data.entry.prep_times.for_2 // .data.entry.prep_times.for_4),
    description: .data.entry.description,
    basic_ingredients: [.data.entry.basics[]?.title],
    ingredients: [.data.entry.ingredients[]? | .label],
    steps: [.data.entry.cooking_instructions | sort_by(.order)[]? | {
        order,
        text: (.instruction | gsub("<[^>]+>"; "") | gsub("&nbsp;"; " ") | gsub("&amp;"; "&") | gsub("&lt;"; "<") | gsub("&gt;"; ">"))
    }]
}'
