#!/bin/bash
# Refresh Google OAuth access token using refresh_token

set -euo pipefail

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
SKILL_DIR="$(cd "$SCRIPT_DIR/.." && pwd)"
TOKEN_FILE="$SKILL_DIR/token.json"
CRED_FILE="$SKILL_DIR/credentials.json"

# Check if files exist
if [ ! -f "$TOKEN_FILE" ]; then
    echo "Error: token.json not found"
    exit 1
fi

if [ ! -f "$CRED_FILE" ]; then
    echo "Error: credentials.json not found"
    exit 1
fi

# Extract tokens and credentials
REFRESH_TOKEN=$(jq -r '.refresh_token // empty' "$TOKEN_FILE")
CLIENT_ID=$(jq -r '.installed.client_id // .web.client_id // empty' "$CRED_FILE")
CLIENT_SECRET=$(jq -r '.installed.client_secret // .web.client_secret // empty' "$CRED_FILE")

if [ -z "$REFRESH_TOKEN" ] || [ -z "$CLIENT_ID" ] || [ -z "$CLIENT_SECRET" ]; then
    echo "Error: Missing required credentials"
    exit 1
fi

echo "Refreshing access token..."

# Request new access token
RESPONSE=$(curl -s -X POST "https://oauth2.googleapis.com/token" \
    -d "client_id=$CLIENT_ID" \
    -d "client_secret=$CLIENT_SECRET" \
    -d "refresh_token=$REFRESH_TOKEN" \
    -d "grant_type=refresh_token")

# Check for errors
if echo "$RESPONSE" | jq -e '.error' > /dev/null 2>&1; then
    ERROR=$(echo "$RESPONSE" | jq -r '.error_description // .error')
    echo "Error: $ERROR"
    exit 1
fi

# Extract new access token
NEW_ACCESS_TOKEN=$(echo "$RESPONSE" | jq -r '.access_token')

if [ -z "$NEW_ACCESS_TOKEN" ] || [ "$NEW_ACCESS_TOKEN" = "null" ]; then
    echo "Error: Failed to get new access token"
    exit 1
fi

# Update token.json with new access token and expiry
EXPIRY_DATE=$(($(date +%s) * 1000 + 3600000))  # Current time + 1 hour in milliseconds

jq --arg token "$NEW_ACCESS_TOKEN" --arg expiry "$EXPIRY_DATE" \
    '.access_token = $token | .expiry_date = ($expiry | tonumber)' \
    "$TOKEN_FILE" > "$TOKEN_FILE.tmp" && mv "$TOKEN_FILE.tmp" "$TOKEN_FILE"

echo "✅ Token refreshed successfully"
echo "New expiry: $(date -d "@$((EXPIRY_DATE / 1000))" 2>/dev/null || date -r $((EXPIRY_DATE / 1000)))"
