#!/bin/bash
# Fetch Google Tasks using credentials.json and token.json

set -euo pipefail

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
SKILL_DIR="$(cd "$SCRIPT_DIR/.." && pwd)"
TOKEN_FILE="$SKILL_DIR/token.json"

# Check if token exists
if [ ! -f "$TOKEN_FILE" ]; then
    echo "Error: token.json not found. Please authenticate first."
    exit 1
fi

# Extract access token
ACCESS_TOKEN=$(jq -r '.access_token // empty' "$TOKEN_FILE")

if [ -z "$ACCESS_TOKEN" ]; then
    echo "Error: Invalid token.json"
    exit 1
fi

# Fetch task lists
echo "📋 Your Google Tasks:"
echo

LISTS=$(curl -s -H "Authorization: Bearer $ACCESS_TOKEN" \
    "https://tasks.googleapis.com/tasks/v1/users/@me/lists")

# Check for auth error
if echo "$LISTS" | jq -e '.error' > /dev/null 2>&1; then
    ERROR_MSG=$(echo "$LISTS" | jq -r '.error.message')
    echo "Error: $ERROR_MSG"
    echo "Try deleting token.json and re-authenticating"
    exit 1
fi

# Process each list
echo "$LISTS" | jq -c '.items[]' | while read -r list_json; do
    LIST_ID=$(echo "$list_json" | jq -r '.id')
    LIST_TITLE=$(echo "$list_json" | jq -r '.title // "(unnamed)"')
    
    echo
    echo "📌 $LIST_TITLE"
    echo "──────────────────────────────────────────────────"
    
    # Fetch tasks for this list
    TASKS=$(curl -s -H "Authorization: Bearer $ACCESS_TOKEN" \
        "https://tasks.googleapis.com/tasks/v1/lists/$LIST_ID/tasks?showCompleted=false")
    
    # Count tasks
    TASK_COUNT=$(echo "$TASKS" | jq '.items | length')
    
    if [ "$TASK_COUNT" = "0" ] || [ "$TASK_COUNT" = "null" ]; then
        echo "  (no tasks)"
    else
        # Process each task
        INDEX=1
        echo "$TASKS" | jq -c '.items[]' | while read -r task_json; do
            TITLE=$(echo "$task_json" | jq -r '.title // "(no title)"')
            DUE=$(echo "$task_json" | jq -r '.due // empty' | cut -d'T' -f1)
            NOTES=$(echo "$task_json" | jq -r '.notes // empty')
            
            LINE="  $INDEX. ⬜ $TITLE"
            [ -n "$DUE" ] && LINE="$LINE (due: $DUE)"
            echo "$LINE"
            
            [ -n "$NOTES" ] && echo "     Note: $NOTES"
            
            INDEX=$((INDEX + 1))
        done
    fi
done

echo
