#!/bin/bash
# Delete a task from Google Tasks

set -euo pipefail

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
SKILL_DIR="$(cd "$SCRIPT_DIR/.." && pwd)"
TOKEN_FILE="$SKILL_DIR/token.json"

# Check arguments
if [ $# -lt 2 ]; then
    echo "Usage: $0 <list-name> <task-number-or-title>"
    echo ""
    echo "Examples:"
    echo "  $0 'Work' 2"
    echo "  $0 'Inbox' 'Buy groceries'"
    exit 1
fi

LIST_NAME="$1"
TASK_IDENTIFIER="$2"

# Check if token exists
if [ ! -f "$TOKEN_FILE" ]; then
    echo "Error: token.json not found. Please authenticate first."
    exit 1
fi

# Extract access token
ACCESS_TOKEN=$(jq -r '.access_token // empty' "$TOKEN_FILE")

if [ -z "$ACCESS_TOKEN" ]; then
    echo "Error: Invalid token.json"
    exit 1
fi

# Fetch task lists
LISTS=$(curl -s -H "Authorization: Bearer $ACCESS_TOKEN" \
    "https://tasks.googleapis.com/tasks/v1/users/@me/lists")

# Check for auth error
if echo "$LISTS" | jq -e '.error' > /dev/null 2>&1; then
    ERROR_MSG=$(echo "$LISTS" | jq -r '.error.message')
    echo "Error: $ERROR_MSG"
    echo "Try deleting token.json and re-authenticating"
    exit 1
fi

# Find list ID by name
LIST_ID=$(echo "$LISTS" | jq -r --arg name "$LIST_NAME" '.items[] | select(.title == $name) | .id')

if [ -z "$LIST_ID" ]; then
    echo "Error: Task list '$LIST_NAME' not found."
    echo ""
    echo "Available lists:"
    echo "$LISTS" | jq -r '.items[] | "  - \(.title)"'
    exit 1
fi

# Fetch tasks from the list
TASKS=$(curl -s -H "Authorization: Bearer $ACCESS_TOKEN" \
    "https://tasks.googleapis.com/tasks/v1/lists/$LIST_ID/tasks?showCompleted=false")

# Check if identifier is a number (task index)
if [[ "$TASK_IDENTIFIER" =~ ^[0-9]+$ ]]; then
    # Get task by index
    TASK_INDEX=$((TASK_IDENTIFIER - 1))
    TASK_ID=$(echo "$TASKS" | jq -r ".items[$TASK_INDEX].id // empty")
    TASK_TITLE=$(echo "$TASKS" | jq -r ".items[$TASK_INDEX].title // empty")
else
    # Get task by title
    TASK_ID=$(echo "$TASKS" | jq -r --arg title "$TASK_IDENTIFIER" '.items[] | select(.title == $title) | .id')
    TASK_TITLE="$TASK_IDENTIFIER"
fi

if [ -z "$TASK_ID" ]; then
    echo "Error: Task '$TASK_IDENTIFIER' not found in list '$LIST_NAME'."
    echo ""
    echo "Available tasks:"
    echo "$TASKS" | jq -r '.items[] | "  - \(.title)"'
    exit 1
fi

# Delete the task
RESULT=$(curl -s -X DELETE \
    -H "Authorization: Bearer $ACCESS_TOKEN" \
    -w "%{http_code}" \
    -o /dev/null \
    "https://tasks.googleapis.com/tasks/v1/lists/$LIST_ID/tasks/$TASK_ID")

if [ "$RESULT" != "204" ]; then
    echo "Error: Failed to delete task (HTTP $RESULT)"
    exit 1
fi

# Success
echo "✅ Task deleted successfully!"
echo ""
echo "List: $LIST_NAME"
echo "Task: $TASK_TITLE"
