#!/bin/bash
# Create a new task in Google Tasks

set -euo pipefail

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
SKILL_DIR="$(cd "$SCRIPT_DIR/.." && pwd)"
TOKEN_FILE="$SKILL_DIR/token.json"
CONFIG_FILE="$SCRIPT_DIR/../google-tasks-config.sh"

# Load default list from config
DEFAULT_LIST="Inbox"
if [ -f "$CONFIG_FILE" ]; then
    source "$CONFIG_FILE"
fi

# Parse arguments - support both formats:
# 1. create_task.sh "task title" [due-date] [notes]
# 2. create_task.sh "list name" "task title" [due-date] [notes]

if [ $# -eq 0 ]; then
    echo "Usage: $0 [list-name] <task-title> [due-date] [notes]"
    echo ""
    echo "Examples:"
    echo "  $0 'Buy groceries'"
    echo "  $0 'Buy groceries' '2026-02-10'"
    echo "  $0 'Work' 'Finish report'"
    echo "  $0 'Work' 'Finish report' '2026-02-10'"
    echo "  $0 'Personal' 'Call mom' '2026-02-05' 'Remember to ask about her health'"
    echo ""
    echo "Default list: $DEFAULT_LIST (configure in google-tasks-config.sh)"
    exit 1
fi

# Detect if first argument is a list name or task title
# If we have 2+ args and second arg doesn't look like a date, treat first as list name
if [ $# -ge 2 ] && ! [[ "$2" =~ ^[0-9]{4}-[0-9]{2}-[0-9]{2}$ ]]; then
    LIST_NAME="$1"
    TASK_TITLE="$2"
    DUE_DATE="${3:-}"
    NOTES="${4:-}"
else
    LIST_NAME="$DEFAULT_LIST"
    TASK_TITLE="$1"
    DUE_DATE="${2:-}"
    NOTES="${3:-}"
fi

# Check if token exists
if [ ! -f "$TOKEN_FILE" ]; then
    echo "Error: token.json not found. Please authenticate first."
    exit 1
fi

# Extract access token
ACCESS_TOKEN=$(jq -r '.access_token // empty' "$TOKEN_FILE")

if [ -z "$ACCESS_TOKEN" ]; then
    echo "Error: Invalid token.json"
    exit 1
fi

# Find the task list by name
LISTS=$(curl -s -H "Authorization: Bearer $ACCESS_TOKEN" \
    "https://tasks.googleapis.com/tasks/v1/users/@me/lists")

# Check for auth error
if echo "$LISTS" | jq -e '.error' > /dev/null 2>&1; then
    ERROR_MSG=$(echo "$LISTS" | jq -r '.error.message')
    echo "Error: $ERROR_MSG"
    echo "Try deleting token.json and re-authenticating"
    exit 1
fi

# Find list ID by name
LIST_ID=$(echo "$LISTS" | jq -r --arg name "$LIST_NAME" '.items[] | select(.title == $name) | .id')

if [ -z "$LIST_ID" ]; then
    echo "Error: Task list '$LIST_NAME' not found."
    echo ""
    echo "Available lists:"
    echo "$LISTS" | jq -r '.items[] | "  - \(.title)"'
    exit 1
fi

# Build task JSON
TASK_JSON=$(jq -n \
    --arg title "$TASK_TITLE" \
    --arg notes "$NOTES" \
    --arg due "$DUE_DATE" \
    '{
        title: $title,
        notes: (if $notes == "" then null else $notes end),
        due: (if $due == "" then null else ($due + "T00:00:00.000Z") end)
    }')

# Create the task
RESULT=$(curl -s -X POST \
    -H "Authorization: Bearer $ACCESS_TOKEN" \
    -H "Content-Type: application/json" \
    -d "$TASK_JSON" \
    "https://tasks.googleapis.com/tasks/v1/lists/$LIST_ID/tasks")

# Check for errors
if echo "$RESULT" | jq -e '.error' > /dev/null 2>&1; then
    ERROR_MSG=$(echo "$RESULT" | jq -r '.error.message')
    echo "Error: $ERROR_MSG"
    exit 1
fi

# Success
TASK_ID=$(echo "$RESULT" | jq -r '.id')
echo "✅ Task created successfully!"
echo ""
echo "List: $LIST_NAME"
echo "Task: $TASK_TITLE"
[ -n "$DUE_DATE" ] && echo "Due: $DUE_DATE"
[ -n "$NOTES" ] && echo "Notes: $NOTES"
echo "ID: $TASK_ID"
