#!/usr/bin/env python3
"""
Google Tasks OAuth authentication script.
Generates token.json for use with bash scripts.
"""

import os
import sys
from google.oauth2.credentials import Credentials
from google_auth_oauthlib.flow import InstalledAppFlow
from google.auth.transport.requests import Request

# Scopes required for Google Tasks
SCOPES = ['https://www.googleapis.com/auth/tasks']

def authenticate():
    """Authenticate and save credentials to token.json"""
    creds = None
    
    # Look for credentials.json and token.json in workspace root
    workspace_root = os.path.abspath(os.path.join(os.path.dirname(__file__), '../../..'))
    token_path = os.path.join(workspace_root, 'token.json')
    credentials_path = os.path.join(workspace_root, 'credentials.json')
    
    print(f"Looking for credentials in: {workspace_root}")
    
    # Check if credentials.json exists
    if not os.path.exists(credentials_path):
        print(f"❌ Error: credentials.json not found at {credentials_path}")
        print("Please place your OAuth credentials file there first.")
        sys.exit(1)
    
    # Load existing token if available
    if os.path.exists(token_path):
        print("Found existing token.json, loading...")
        creds = Credentials.from_authorized_user_file(token_path, SCOPES)
    
    # If no valid credentials, let user log in
    if not creds or not creds.valid:
        if creds and creds.expired and creds.refresh_token:
            print("Token expired, refreshing...")
            try:
                creds.refresh(Request())
                print("✅ Token refreshed successfully!")
            except Exception as e:
                print(f"⚠️  Failed to refresh token: {e}")
                print("Starting new authentication flow...")
                creds = None
        
        if not creds:
            print("Starting OAuth authentication flow...")
            print("A browser window will open. Please log in and authorize access.")
            flow = InstalledAppFlow.from_client_secrets_file(credentials_path, SCOPES)
            creds = flow.run_local_server(port=0)
            print("✅ Authentication successful!")
        
        # Save credentials for future use
        with open(token_path, 'w') as token:
            token.write(creds.to_json())
        print(f"✅ Token saved to {token_path}")
    else:
        print("✅ Valid token already exists!")
    
    return creds

if __name__ == '__main__':
    try:
        authenticate()
        print("\n✨ You can now use the bash scripts to manage your tasks!")
    except Exception as e:
        print(f"\n❌ Authentication failed: {e}")
        sys.exit(1)
