#!/usr/bin/env node
/**
 * Google Tasks OAuth authentication using Node.js
 * Generates/refreshes token.json for bash scripts
 */

import { google } from 'googleapis';
import { OAuth2Client } from 'google-auth-library';
import fs from 'fs/promises';
import path from 'path';
import { fileURLToPath } from 'url';
import http from 'http';
import url from 'url';
import open from 'open';

const __dirname = path.dirname(fileURLToPath(import.meta.url));
const WORKSPACE_ROOT = path.resolve(__dirname, '../../..');
const TOKEN_PATH = path.join(WORKSPACE_ROOT, 'token.json');
const CREDENTIALS_PATH = path.join(WORKSPACE_ROOT, 'credentials.json');
const SCOPES = ['https://www.googleapis.com/auth/tasks'];

async function loadCredentials() {
  try {
    const content = await fs.readFile(CREDENTIALS_PATH, 'utf8');
    return JSON.parse(content);
  } catch (err) {
    console.error(`❌ Error: credentials.json not found at ${CREDENTIALS_PATH}`);
    console.error('Please place your OAuth credentials file there first.');
    process.exit(1);
  }
}

async function loadToken() {
  try {
    const content = await fs.readFile(TOKEN_PATH, 'utf8');
    return JSON.parse(content);
  } catch (err) {
    return null;
  }
}

async function saveToken(token) {
  await fs.writeFile(TOKEN_PATH, JSON.stringify(token, null, 2));
  console.log(`✅ Token saved to ${TOKEN_PATH}`);
}

async function getNewToken(oauth2Client) {
  return new Promise((resolve, reject) => {
    const authUrl = oauth2Client.generateAuthUrl({
      access_type: 'offline',
      scope: SCOPES,
    });

    const server = http.createServer(async (req, res) => {
      try {
        const qs = new url.URL(req.url, 'http://localhost:3000').searchParams;
        const code = qs.get('code');
        
        if (code) {
          res.end('✅ Authentication successful! You can close this window.');
          server.close();
          
          const { tokens } = await oauth2Client.getToken(code);
          oauth2Client.setCredentials(tokens);
          resolve(tokens);
        }
      } catch (e) {
        reject(e);
      }
    }).listen(3000, () => {
      console.log('🌐 Opening browser for authentication...');
      console.log(`If browser doesn't open, visit: ${authUrl}`);
      open(authUrl);
    });
  });
}

async function authenticate() {
  console.log(`Looking for credentials in: ${WORKSPACE_ROOT}`);
  
  const credentials = await loadCredentials();
  const { client_secret, client_id, redirect_uris } = credentials.installed || credentials.web;
  
  const oauth2Client = new OAuth2Client(client_id, client_secret, redirect_uris[0]);
  
  // Try to load existing token
  const token = await loadToken();
  
  if (token) {
    console.log('Found existing token.json, checking validity...');
    oauth2Client.setCredentials(token);
    
    // Check if token is expired
    const tokenInfo = await oauth2Client.getTokenInfo(token.access_token).catch(() => null);
    
    if (!tokenInfo) {
      console.log('Token expired or invalid, refreshing...');
      try {
        const { credentials: newToken } = await oauth2Client.refreshAccessToken();
        await saveToken(newToken);
        console.log('✅ Token refreshed successfully!');
        return;
      } catch (err) {
        console.log('⚠️  Failed to refresh token, starting new authentication flow...');
      }
    } else {
      console.log('✅ Valid token already exists!');
      return;
    }
  }
  
  // Get new token
  console.log('Starting OAuth authentication flow...');
  const newToken = await getNewToken(oauth2Client);
  await saveToken(newToken);
  console.log('✅ Authentication successful!');
}

// Main
authenticate()
  .then(() => {
    console.log('\n✨ You can now use the bash scripts to manage your tasks!');
    process.exit(0);
  })
  .catch((err) => {
    console.error('\n❌ Authentication failed:', err.message);
    process.exit(1);
  });
