# Google Tasks Skill for OpenClaw

Manage your Google Tasks directly from OpenClaw using lightweight bash scripts with OAuth authentication.

## Features

- ✅ **View all tasks** across all your Google Task lists
- ✅ **Create new tasks** with optional due dates and notes
- ✅ **Delete tasks** by number or title
- ✅ **Automatic token refresh** - no re-authentication needed
- ✅ **Lightweight** - Pure bash with curl and jq
- ✅ **Configurable** - Set your default task list

## Quick Start

### View your tasks
```bash
bash scripts/get_tasks.sh
```

### Create a task
```bash
# Simple task (uses default list from config)
bash scripts/create_task.sh "Buy groceries"

# Task with due date
bash scripts/create_task.sh "Finish report" "2026-02-10"

# Task in specific list
bash scripts/create_task.sh "Work" "Finish report" "2026-02-10"

# Task with notes
bash scripts/create_task.sh "Personal" "Call mom" "2026-02-05" "Ask about her health"
```

### Delete a task
```bash
# By position in list
bash scripts/delete_task.sh "Work" 2

# By task title
bash scripts/delete_task.sh "Inbox" "Buy groceries"
```

## Setup

1. **Install dependencies:**
   - `jq` (JSON processor) - usually pre-installed
   - `curl` (HTTP client) - usually pre-installed
   - Node.js packages: `npm install`

2. **Get OAuth credentials:**
   - Follow the guide in [references/setup.md](references/setup.md)
   - Download `credentials.json` from Google Cloud Console
   - Add required scope: `https://www.googleapis.com/auth/tasks`

3. **Authenticate:**
   - Run the initial authentication to generate `token.json`
   - The bash scripts will handle token refresh automatically

4. **Configure default list (optional):**
   - Edit `google-tasks-config.sh` to set your preferred default list

## File Structure

```
google-tasks/
├── SKILL.md                    # Skill metadata and documentation
├── README.md                   # This file
├── package.json                # Node.js dependencies for auth
├── google-tasks-config.sh      # Configuration (default list)
├── .gitignore                  # Protects credentials and tokens
├── scripts/
│   ├── get_tasks.sh           # View all tasks
│   ├── create_task.sh         # Create new tasks
│   ├── delete_task.sh         # Delete tasks
│   └── refresh_token.sh       # Refresh OAuth token
└── references/
    └── setup.md               # Detailed setup guide
```

## Requirements

- **API Scopes:** `https://www.googleapis.com/auth/tasks` (read + write)
- **Credentials:** OAuth 2.0 client credentials in `credentials.json`
- **Token:** Valid `token.json` generated through initial authentication

## Output Format

```
📋 Your Google Tasks:

📌 Work
──────────────────────────────────────────────────
  1. ⬜ Finish quarterly report (due: 2026-02-10)
  2. ⬜ Team meeting preparation

📌 Personal
──────────────────────────────────────────────────
  1. ⬜ Buy groceries
     Note: Milk, bread, eggs
  2. ⬜ Call mom (due: 2026-02-05)
```

## Troubleshooting

**Token expired:**
```
Error: Invalid credentials
```
Run `bash scripts/refresh_token.sh` or delete `token.json` and re-authenticate.

**Missing jq:**
```
bash: jq: command not found
```
Install with: `apt-get install jq` or `brew install jq`

**List not found:**
The list name is case-sensitive. Use `get_tasks.sh` to see available list names.

## API Limits

- **Free quota:** 50,000 requests/day
- **Rate limit:** 600 requests per 100 seconds
- No billing required for personal use

## License

MIT

## Contributing

Contributions welcome! Please ensure all scripts follow the existing pattern:
- Use `set -euo pipefail` for safety
- Check for token validity before API calls
- Provide clear error messages
- Keep output format consistent

## Support

For detailed setup instructions, see [references/setup.md](references/setup.md).

For issues or questions, please file an issue on the repository.
