#!/home/jorge/.openclaw/workspace/skills/google-photos/venv/bin/python3
import os
import pickle
import json
import argparse
import requests
from google_auth_oauthlib.flow import InstalledAppFlow
from google.auth.transport.requests import Request
from google.oauth2.credentials import Credentials

# Scopes for Google Photos
SCOPES = ['https://www.googleapis.com/auth/photoslibrary.appendonly',
          'https://www.googleapis.com/auth/photoslibrary.readonly.appcreateddata',
          'https://www.googleapis.com/auth/photoslibrary.sharing']

# Path for credentials and token
# These are environment-specific and should be handled with care in a public skill
# Users would normally provide their own credentials.json
CREDENTIALS_FILE = 'credentials.json'
TOKEN_FILE = 'token_photos.pickle'

def get_credentials(credentials_path, token_path):
    creds = None
    if os.path.exists(token_path):
        with open(token_path, 'rb') as token:
            creds = pickle.load(token)
    if not creds or not creds.valid:
        if creds and creds.expired and creds.refresh_token:
            creds.refresh(Request())
        else:
            flow = InstalledAppFlow.from_client_secrets_file(credentials_path, SCOPES)
            # Use console flow for headless/remote environments
            auth_url, _ = flow.authorization_url(prompt='consent')
            print(f"Please visit this URL to authorize: {auth_url}")
            code = input("Enter the authorization code: ")
            flow.fetch_token(code=code)
            creds = flow.credentials
        with open(token_path, 'wb') as token:
            pickle.dump(creds, token)
    return creds

def list_albums(creds):
    headers = {
        'Authorization': f'Bearer {creds.token}',
        'Content-Type': 'application/json'
    }
    response = requests.get('https://photoslibrary.googleapis.com/v1/albums', headers=headers)
    return response.json()

def create_album(creds, title):
    headers = {
        'Authorization': f'Bearer {creds.token}',
        'Content-Type': 'application/json'
    }
    payload = {"album": {"title": title}}
    response = requests.post('https://photoslibrary.googleapis.com/v1/albums', headers=headers, json=payload)
    return response.json()

def upload_photo(creds, photo_path, album_id=None):
    # Step 1: Get upload token
    headers = {
        'Authorization': f'Bearer {creds.token}',
        'Content-type': 'application/octet-stream',
        'X-Goog-Upload-File-Name': os.path.basename(photo_path),
        'X-Goog-Upload-Protocol': 'raw',
    }
    with open(photo_path, 'rb') as f:
        response = requests.post('https://photoslibrary.googleapis.com/v1/uploads', headers=headers, data=f)
    
    upload_token = response.text
    if response.status_code != 200:
        print(f"Error uploading: {response.text}")
        return None

    # Step 2: Create media item
    headers = {
        'Authorization': f'Bearer {creds.token}',
        'Content-type': 'application/json',
    }
    payload = {
        'newMediaItems': [
            {
                'description': 'Uploaded via OpenClaw',
                'simpleMediaItem': {
                    'uploadToken': upload_token
                }
            }
        ]
    }
    if album_id:
        payload['albumId'] = album_id

    response = requests.post('https://photoslibrary.googleapis.com/v1/mediaItems:batchCreate', headers=headers, json=payload)
    return response.json()

if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='Google Photos CLI Utility')
    parser.add_argument('--action', choices=['list', 'create', 'upload'], required=True)
    parser.add_argument('--title', help='Album title for create')
    parser.add_argument('--photo', help='Photo path for upload')
    parser.add_argument('--album-id', help='Album ID for upload')
    parser.add_argument('--credentials', default=CREDENTIALS_FILE, help='Path to Google Cloud credentials.json')
    parser.add_argument('--token', default=TOKEN_FILE, help='Path to store/read auth token')
    
    args = parser.parse_args()

    if not os.path.exists(args.credentials):
        print(f"Error: Credentials file not found at {args.credentials}")
        print("Please provide a valid Google Cloud credentials.json file.")
        exit(1)

    creds = get_credentials(args.credentials, args.token)
    
    if args.action == 'list':
        print(json.dumps(list_albums(creds), indent=2))
    elif args.action == 'create':
        print(json.dumps(create_album(creds, args.title), indent=2))
    elif args.action == 'upload':
        if not args.photo:
            print("Error: --photo path is required for upload action")
            exit(1)
        print(json.dumps(upload_photo(creds, args.photo, args.album_id), indent=2))
