---
name: google-maps
description: >
  Google Maps integration for OpenClaw with Routes API. Use for: (1) Distance/travel time calculations 
  with traffic prediction, (2) Turn-by-turn directions, (3) Distance matrix between multiple points, 
  (4) Geocoding addresses to coordinates and reverse, (5) Places search and details, (6) Transit 
  planning with arrival times. Supports future departure times, traffic models (pessimistic/optimistic), 
  avoid options (tolls/highways), and multiple travel modes (driving/walking/bicycling/transit).
version: 3.1.2
author: Leo 🦁
tags: [maps, places, location, navigation, google, traffic, directions, geocoding, routes-api]
metadata: {"clawdbot":{"emoji":"🗺️","requires":{"env":["GOOGLE_API_KEY"]},"primaryEnv":"GOOGLE_API_KEY","secondaryEnv":["GOOGLE_MAPS_API_KEY"],"install":[{"id":"pip","kind":"pip","package":"requests","label":"Install requests library"}]}}
allowed-tools: [exec]
---

# Google Maps 🗺️

Google Maps integration powered by the Routes API.

## Requirements

- `GOOGLE_API_KEY` environment variable
- Enable in Google Cloud Console: Routes API, Places API, Geocoding API

## Configuration

| Env Variable | Default | Description |
|--------------|---------|-------------|
| `GOOGLE_API_KEY` | - | Required. Your Google Maps API key |
| `GOOGLE_MAPS_API_KEY` | - | Alternative to `GOOGLE_API_KEY` (fallback) |
| `GOOGLE_MAPS_LANG` | `en` | Response language (en, he, ja, etc.) |

Set in OpenClaw config:
```json
{
  "env": {
    "GOOGLE_API_KEY": "AIza...",
    "GOOGLE_MAPS_LANG": "en"
  }
}
```

## Script Location

```bash
python3 skills/google-maps/lib/map_helper.py <action> [options]
```

---

## Actions

### distance - Calculate travel time

```bash
python3 lib/map_helper.py distance "origin" "destination" [options]
```

**Options:**
| Option | Values | Description |
|--------|--------|-------------|
| `--mode` | driving, walking, bicycling, transit | Travel mode (default: driving) |
| `--depart` | now, +30m, +1h, 14:00, 2026-02-07 08:00 | Departure time |
| `--arrive` | 14:00 | Arrival time (transit only) |
| `--traffic` | best_guess, pessimistic, optimistic | Traffic model |
| `--avoid` | tolls, highways, ferries | Comma-separated |

**Examples:**
```bash
python3 lib/map_helper.py distance "New York" "Boston"
python3 lib/map_helper.py distance "Los Angeles" "San Francisco" --depart="+1h"
python3 lib/map_helper.py distance "Chicago" "Detroit" --depart="08:00" --traffic=pessimistic
python3 lib/map_helper.py distance "London" "Manchester" --mode=transit --arrive="09:00"
python3 lib/map_helper.py distance "Paris" "Lyon" --avoid=tolls,highways
```

**Response:**
```json
{
  "distance": "215.2 mi",
  "distance_meters": 346300,
  "duration": "3 hrs 45 mins",
  "duration_seconds": 13500,
  "static_duration": "3 hrs 30 mins",
  "duration_in_traffic": "3 hrs 45 mins"
}
```

---

### directions - Turn-by-turn route

```bash
python3 lib/map_helper.py directions "origin" "destination" [options]
```

**Additional options (beyond distance):**
| Option | Description |
|--------|-------------|
| `--alternatives` | Return multiple routes |
| `--waypoints` | Intermediate stops (pipe-separated) |
| `--optimize` | Optimize waypoint order (TSP) |

**Examples:**
```bash
python3 lib/map_helper.py directions "New York" "Washington DC"
python3 lib/map_helper.py directions "San Francisco" "Los Angeles" --alternatives
python3 lib/map_helper.py directions "Miami" "Orlando" --waypoints="Fort Lauderdale|West Palm Beach" --optimize
```

**Response includes:** summary, labels, duration, static_duration, warnings, steps[], optimized_waypoint_order

---

### matrix - Distance matrix

Calculate distances between multiple origins and destinations:

```bash
python3 lib/map_helper.py matrix "orig1|orig2" "dest1|dest2"
```

**Example:**
```bash
python3 lib/map_helper.py matrix "New York|Boston" "Philadelphia|Washington DC"
```

**Response:**
```json
{
  "origins": ["New York", "Boston"],
  "destinations": ["Philadelphia", "Washington DC"],
  "results": [
    {"origin_index": 0, "destination_index": 0, "distance": "97 mi", "duration": "1 hr 45 mins"},
    {"origin_index": 0, "destination_index": 1, "distance": "225 mi", "duration": "4 hrs 10 mins"}
  ]
}
```

---

### geocode - Address to coordinates

```bash
python3 lib/map_helper.py geocode "1600 Amphitheatre Parkway, Mountain View, CA"
python3 lib/map_helper.py geocode "10 Downing Street, London"
```

### reverse - Coordinates to address

```bash
python3 lib/map_helper.py reverse 40.7128 -74.0060  # New York City
python3 lib/map_helper.py reverse 51.5074 -0.1278  # London
```

---

### search - Find places

```bash
python3 lib/map_helper.py search "coffee near Times Square"
python3 lib/map_helper.py search "pharmacy in San Francisco" --open
```

### details - Place information

```bash
python3 lib/map_helper.py details "<place_id>"
```

---

## Traffic Models

| Model | Use Case |
|-------|----------|
| `best_guess` | Default balanced estimate |
| `pessimistic` | Important meetings (worst-case) |
| `optimistic` | Best-case scenario |

---

## Regional Notes

Some features may not be available in all countries:

| Feature | Availability |
|---------|--------------|
| `--fuel-efficient` | US, EU, select countries |
| `--shorter` | Limited availability |
| `--mode=two_wheeler` | Asia, select countries |

Check [Google Maps coverage](https://developers.google.com/maps/coverage) for details.

---

## Multilingual Support

Works with addresses in any language:

```bash
# Hebrew
python3 lib/map_helper.py distance "תל אביב" "ירושלים"
python3 lib/map_helper.py geocode "דיזנגוף 50, תל אביב"

# Japanese
python3 lib/map_helper.py distance "東京" "大阪"

# Arabic
python3 lib/map_helper.py distance "دبي" "أبو ظبي"
```

**Language configuration:**

1. Set default via env: `GOOGLE_MAPS_LANG=he` (persists)
2. Override per-request: `--lang=ja`

```bash
# Set Hebrew as default in OpenClaw config
GOOGLE_MAPS_LANG=he

# Override for specific request
python3 lib/map_helper.py distance "Tokyo" "Osaka" --lang=ja
```

---

## Help

```bash
python3 lib/map_helper.py help
```
