---
name: google-forms
description: |
  Google Forms API integration with managed OAuth. Create forms, add questions, and retrieve responses. Use this skill when users want to interact with Google Forms. For other third party apps, use the api-gateway skill (https://clawhub.ai/byungkyu/api-gateway).
compatibility: Requires network access and valid Maton API key
metadata:
  author: maton
  version: "1.0"
  clawdbot:
    emoji: 🧠
    requires:
      env:
        - MATON_API_KEY
---

# Google Forms

Access the Google Forms API with managed OAuth authentication. Create forms, add questions, and retrieve responses.

## Quick Start

```bash
# Get form
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://gateway.maton.ai/google-forms/v1/forms/{formId}')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

## Base URL

```
https://gateway.maton.ai/google-forms/{native-api-path}
```

Replace `{native-api-path}` with the actual Google Forms API endpoint path. The gateway proxies requests to `forms.googleapis.com` and automatically injects your OAuth token.

## Authentication

All requests require the Maton API key in the Authorization header:

```
Authorization: Bearer $MATON_API_KEY
```

**Environment Variable:** Set your API key as `MATON_API_KEY`:

```bash
export MATON_API_KEY="YOUR_API_KEY"
```

### Getting Your API Key

1. Sign in or create an account at [maton.ai](https://maton.ai)
2. Go to [maton.ai/settings](https://maton.ai/settings)
3. Copy your API key

## Connection Management

Manage your Google OAuth connections at `https://ctrl.maton.ai`.

### List Connections

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections?app=google-forms&status=ACTIVE')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Create Connection

```bash
python <<'EOF'
import urllib.request, os, json
data = json.dumps({'app': 'google-forms'}).encode()
req = urllib.request.Request('https://ctrl.maton.ai/connections', data=data, method='POST')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
req.add_header('Content-Type', 'application/json')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Get Connection

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections/{connection_id}')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

**Response:**
```json
{
  "connection": {
    "connection_id": "21fd90f9-5935-43cd-b6c8-bde9d915ca80",
    "status": "ACTIVE",
    "creation_time": "2025-12-08T07:20:53.488460Z",
    "last_updated_time": "2026-01-31T20:03:32.593153Z",
    "url": "https://connect.maton.ai/?session_token=...",
    "app": "google-forms",
    "metadata": {}
  }
}
```

Open the returned `url` in a browser to complete OAuth authorization.

### Delete Connection

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections/{connection_id}', method='DELETE')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Specifying Connection

If you have multiple Google Forms connections, specify which one to use with the `Maton-Connection` header:

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://gateway.maton.ai/google-forms/v1/forms/{formId}')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
req.add_header('Maton-Connection', '21fd90f9-5935-43cd-b6c8-bde9d915ca80')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

If omitted, the gateway uses the default (oldest) active connection.

## API Reference

### Get Form

```bash
GET /google-forms/v1/forms/{formId}
```

### Create Form

```bash
POST /google-forms/v1/forms
Content-Type: application/json

{
  "info": {
    "title": "Customer Feedback Survey"
  }
}
```

### Batch Update Form

```bash
POST /google-forms/v1/forms/{formId}:batchUpdate
Content-Type: application/json

{
  "requests": [
    {
      "createItem": {
        "item": {
          "title": "What is your name?",
          "questionItem": {
            "question": {
              "required": true,
              "textQuestion": {"paragraph": false}
            }
          }
        },
        "location": {"index": 0}
      }
    }
  ]
}
```

### List Responses

```bash
GET /google-forms/v1/forms/{formId}/responses
```

### Get Response

```bash
GET /google-forms/v1/forms/{formId}/responses/{responseId}
```

## Common batchUpdate Requests

### Create Text Question

```json
{
  "createItem": {
    "item": {
      "title": "Question text",
      "questionItem": {
        "question": {
          "required": true,
          "textQuestion": {"paragraph": false}
        }
      }
    },
    "location": {"index": 0}
  }
}
```

### Create Multiple Choice

```json
{
  "createItem": {
    "item": {
      "title": "Select an option",
      "questionItem": {
        "question": {
          "choiceQuestion": {
            "type": "RADIO",
            "options": [
              {"value": "Option A"},
              {"value": "Option B"}
            ]
          }
        }
      }
    },
    "location": {"index": 0}
  }
}
```

### Create Scale Question

```json
{
  "createItem": {
    "item": {
      "title": "Rate your experience",
      "questionItem": {
        "question": {
          "scaleQuestion": {
            "low": 1,
            "high": 5,
            "lowLabel": "Poor",
            "highLabel": "Excellent"
          }
        }
      }
    },
    "location": {"index": 0}
  }
}
```

## Question Types

- `textQuestion` - Short or paragraph text
- `choiceQuestion` - Radio, checkbox, or dropdown
- `scaleQuestion` - Linear scale
- `dateQuestion` - Date picker
- `timeQuestion` - Time picker

## Code Examples

### JavaScript

```javascript
const response = await fetch(
  'https://gateway.maton.ai/google-forms/v1/forms/FORM_ID/responses',
  {
    headers: {
      'Authorization': `Bearer ${process.env.MATON_API_KEY}`
    }
  }
);
```

### Python

```python
import os
import requests

response = requests.get(
    f'https://gateway.maton.ai/google-forms/v1/forms/FORM_ID/responses',
    headers={'Authorization': f'Bearer {os.environ["MATON_API_KEY"]}'}
)
```

## Notes

- Form IDs are found in the form URL
- Use `updateMask` to specify fields to update
- Location index is 0-based
- IMPORTANT: When using curl commands, use `curl -g` when URLs contain brackets to disable glob parsing
- IMPORTANT: When piping curl output to `jq` or other commands, environment variables like `$MATON_API_KEY` may not expand correctly in some shell environments. You may get "Invalid API key" errors when piping.

## Error Handling

| Status | Meaning |
|--------|---------|
| 400 | Missing Google Forms connection |
| 401 | Invalid or missing Maton API key |
| 429 | Rate limited (10 req/sec per account) |
| 4xx/5xx | Passthrough error from Google Forms API |

### Troubleshooting: API Key Issues

1. Check that the `MATON_API_KEY` environment variable is set:

```bash
echo $MATON_API_KEY
```

2. Verify the API key is valid by listing connections:

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Troubleshooting: Invalid App Name

1. Ensure your URL path starts with `google-forms`. For example:

- Correct: `https://gateway.maton.ai/google-forms/v1/forms/{formId}`
- Incorrect: `https://gateway.maton.ai/forms/v1/forms/{formId}`

## Resources

- [Forms API Overview](https://developers.google.com/workspace/forms/api/reference/rest)
- [Get Form](https://developers.google.com/workspace/forms/api/reference/rest/v1/forms/get)
- [Create Form](https://developers.google.com/workspace/forms/api/reference/rest/v1/forms/create)
- [Batch Update](https://developers.google.com/workspace/forms/api/reference/rest/v1/forms/batchUpdate)
- [List Responses](https://developers.google.com/workspace/forms/api/reference/rest/v1/forms.responses/list)
- [Maton Community](https://discord.com/invite/dBfFAcefs2)
- [Maton Support](mailto:support@maton.ai)
