---
name: google-analytics
description: |
  Google Analytics API integration with managed OAuth. Manage accounts, properties, and data streams (Admin API). Run reports on sessions, users, page views, and conversions (Data API). Use this skill when users want to configure or query Google Analytics. For other third party apps, use the api-gateway skill (https://clawhub.ai/byungkyu/api-gateway).
compatibility: Requires network access and valid Maton API key
metadata:
  author: maton
  version: "1.0"
---

# Google Analytics

Access Google Analytics with managed OAuth authentication. This skill covers both the Admin API (manage accounts, properties, data streams) and the Data API (run reports on metrics).

## Quick Start

```bash
# List account summaries (Admin API)
curl -s -X GET "https://gateway.maton.ai/google-analytics-admin/v1beta/accountSummaries" -H "Authorization: Bearer $MATON_API_KEY"

# Run a report (Data API)
curl -s -X POST "https://gateway.maton.ai/google-analytics-data/v1beta/properties/{propertyId}:runReport" -H "Content-Type: application/json" -H "Authorization: Bearer $MATON_API_KEY" -d '{"dateRanges": [{"startDate": "30daysAgo", "endDate": "today"}], "dimensions": [{"name": "city"}], "metrics": [{"name": "activeUsers"}]}'
```

## Base URLs

**Admin API** (manage accounts, properties, data streams):
```
https://gateway.maton.ai/google-analytics-admin/{native-api-path}
```

**Data API** (run reports):
```
https://gateway.maton.ai/google-analytics-data/{native-api-path}
```

Replace `{native-api-path}` with the actual Google Analytics API endpoint path. The gateway proxies requests to `analyticsadmin.googleapis.com` and `analyticsdata.googleapis.com` and automatically injects your OAuth token.

## Authentication

All requests require the Maton API key in the Authorization header:

```
Authorization: Bearer $MATON_API_KEY
```

**Environment Variable:** Set your API key as `MATON_API_KEY`:

```bash
export MATON_API_KEY="YOUR_API_KEY"
```

### Getting Your API Key

1. Sign in or create an account at [maton.ai](https://maton.ai)
2. Go to [maton.ai/settings](https://maton.ai/settings)
3. Copy your API key

## Connection Management

Manage your Google OAuth connections at `https://ctrl.maton.ai`.

**Important:** The Admin API and Data API use separate connections:
- `google-analytics-admin` - Required for Admin API endpoints (manage accounts, properties, data streams)
- `google-analytics-data` - Required for Data API endpoints (run reports)

Create the connection(s) you need based on which API you want to use.

### List Connections

```bash
# List Admin API connections
curl -s -X GET "https://ctrl.maton.ai/connections?app=google-analytics-admin&status=ACTIVE" -H "Authorization: Bearer $MATON_API_KEY"

# List Data API connections
curl -s -X GET "https://ctrl.maton.ai/connections?app=google-analytics-data&status=ACTIVE" -H "Authorization: Bearer $MATON_API_KEY"
```

### Create Connection

```bash
# Create Admin API connection (for managing accounts, properties, data streams)
curl -s -X POST "https://ctrl.maton.ai/connections" -H "Content-Type: application/json" -H "Authorization: Bearer $MATON_API_KEY" -d '{"app": "google-analytics-admin"}'

# Create Data API connection (for running reports)
curl -s -X POST "https://ctrl.maton.ai/connections" -H "Content-Type: application/json" -H "Authorization: Bearer $MATON_API_KEY" -d '{"app": "google-analytics-data"}'
```

### Get Connection

```bash
curl -s -X GET "https://ctrl.maton.ai/connections/{connection_id}" -H "Authorization: Bearer $MATON_API_KEY"
```

**Response:**
```json
{
  "connection": {
    "connection_id": "21fd90f9-5935-43cd-b6c8-bde9d915ca80",
    "status": "ACTIVE",
    "creation_time": "2025-12-08T07:20:53.488460Z",
    "last_updated_time": "2026-01-31T20:03:32.593153Z",
    "url": "https://connect.maton.ai/?session_token=...",
    "app": "google-analytics-admin",
    "metadata": {}
  }
}
```

Open the returned `url` in a browser to complete OAuth authorization.

### Delete Connection

```bash
curl -s -X DELETE "https://ctrl.maton.ai/connections/{connection_id}" -H "Authorization: Bearer $MATON_API_KEY"
```

### Specifying Connection

If you have multiple Google Analytics connections, specify which one to use with the `Maton-Connection` header:

```bash
curl -s -X GET "https://gateway.maton.ai/google-analytics-admin/v1beta/accountSummaries" -H "Authorization: Bearer $MATON_API_KEY" -H "Maton-Connection: 21fd90f9-5935-43cd-b6c8-bde9d915ca80"
```

If omitted, the gateway uses the default (oldest) active connection.

## Admin API Reference

### Accounts

```bash
GET /google-analytics-admin/v1beta/accounts
GET /google-analytics-admin/v1beta/accounts/{accountId}
GET /google-analytics-admin/v1beta/accountSummaries
```

### Properties

```bash
GET /google-analytics-admin/v1beta/properties?filter=parent:accounts/{accountId}
GET /google-analytics-admin/v1beta/properties/{propertyId}
```

#### Create Property

```bash
POST /google-analytics-admin/v1beta/properties
Content-Type: application/json

{
  "parent": "accounts/{accountId}",
  "displayName": "My New Property",
  "timeZone": "America/Los_Angeles",
  "currencyCode": "USD"
}
```

### Data Streams

```bash
GET /google-analytics-admin/v1beta/properties/{propertyId}/dataStreams
```

#### Create Web Data Stream

```bash
POST /google-analytics-admin/v1beta/properties/{propertyId}/dataStreams
Content-Type: application/json

{
  "type": "WEB_DATA_STREAM",
  "displayName": "My Website",
  "webStreamData": {"defaultUri": "https://example.com"}
}
```

### Custom Dimensions

```bash
GET /google-analytics-admin/v1beta/properties/{propertyId}/customDimensions
```

#### Create Custom Dimension

```bash
POST /google-analytics-admin/v1beta/properties/{propertyId}/customDimensions
Content-Type: application/json

{
  "parameterName": "user_type",
  "displayName": "User Type",
  "scope": "USER"
}
```

### Conversion Events

```bash
GET /google-analytics-admin/v1beta/properties/{propertyId}/conversionEvents
POST /google-analytics-admin/v1beta/properties/{propertyId}/conversionEvents
```

## Data API Reference

### Run Report

```bash
POST /google-analytics-data/v1beta/properties/{propertyId}:runReport
Content-Type: application/json

{
  "dateRanges": [{"startDate": "30daysAgo", "endDate": "today"}],
  "dimensions": [{"name": "city"}],
  "metrics": [{"name": "activeUsers"}]
}
```

### Run Realtime Report

```bash
POST /google-analytics-data/v1beta/properties/{propertyId}:runRealtimeReport
Content-Type: application/json

{
  "dimensions": [{"name": "country"}],
  "metrics": [{"name": "activeUsers"}]
}
```

### Batch Run Reports

```bash
POST /google-analytics-data/v1beta/properties/{propertyId}:batchRunReports
Content-Type: application/json

{
  "requests": [
    {
      "dateRanges": [{"startDate": "7daysAgo", "endDate": "today"}],
      "dimensions": [{"name": "country"}],
      "metrics": [{"name": "sessions"}]
    }
  ]
}
```

### Get Metadata

```bash
GET /google-analytics-data/v1beta/properties/{propertyId}/metadata
```

## Common Report Examples

### Page Views by Page

```json
{
  "dateRanges": [{"startDate": "30daysAgo", "endDate": "today"}],
  "dimensions": [{"name": "pagePath"}],
  "metrics": [{"name": "screenPageViews"}],
  "orderBys": [{"metric": {"metricName": "screenPageViews"}, "desc": true}],
  "limit": 10
}
```

### Users by Country

```json
{
  "dateRanges": [{"startDate": "30daysAgo", "endDate": "today"}],
  "dimensions": [{"name": "country"}],
  "metrics": [{"name": "activeUsers"}, {"name": "sessions"}]
}
```

### Traffic Sources

```json
{
  "dateRanges": [{"startDate": "30daysAgo", "endDate": "today"}],
  "dimensions": [{"name": "sessionSource"}, {"name": "sessionMedium"}],
  "metrics": [{"name": "sessions"}, {"name": "conversions"}]
}
```

## Common Dimensions

- `date`, `country`, `city`, `deviceCategory`
- `pagePath`, `pageTitle`, `landingPage`
- `sessionSource`, `sessionMedium`, `sessionCampaignName`

## Common Metrics

- `activeUsers`, `newUsers`, `sessions`
- `screenPageViews`, `bounceRate`, `averageSessionDuration`
- `conversions`, `eventCount`

## Date Formats

- Relative: `today`, `yesterday`, `7daysAgo`, `30daysAgo`
- Absolute: `2026-01-01`

## Code Examples

### JavaScript

```javascript
// List account summaries (Admin API)
const accounts = await fetch(
  'https://gateway.maton.ai/google-analytics-admin/v1beta/accountSummaries',
  {
    headers: {
      'Authorization': `Bearer ${process.env.MATON_API_KEY}`
    }
  }
);

// Run a report (Data API)
const report = await fetch(
  'https://gateway.maton.ai/google-analytics-data/v1beta/properties/123456:runReport',
  {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
      'Authorization': `Bearer ${process.env.MATON_API_KEY}`
    },
    body: JSON.stringify({
      dateRanges: [{ startDate: '30daysAgo', endDate: 'today' }],
      dimensions: [{ name: 'country' }],
      metrics: [{ name: 'activeUsers' }]
    })
  }
);
```

### Python

```python
import os
import requests

# List account summaries (Admin API)
accounts = requests.get(
    'https://gateway.maton.ai/google-analytics-admin/v1beta/accountSummaries',
    headers={'Authorization': f'Bearer {os.environ["MATON_API_KEY"]}'}
)

# Run a report (Data API)
report = requests.post(
    'https://gateway.maton.ai/google-analytics-data/v1beta/properties/123456:runReport',
    headers={'Authorization': f'Bearer {os.environ["MATON_API_KEY"]}'},
    json={
        'dateRanges': [{'startDate': '30daysAgo', 'endDate': 'today'}],
        'dimensions': [{'name': 'country'}],
        'metrics': [{'name': 'activeUsers'}]
    }
)
```

## Notes

- GA4 properties only (Universal Analytics not supported)
- Property IDs are numeric (e.g., `properties/521310447`)
- Use `accountSummaries` to quickly list all accessible properties
- Use `updateMask` for PATCH requests in Admin API
- Use metadata endpoint to discover available dimensions/metrics
- IMPORTANT: When using curl commands, use `curl -g` when URLs contain brackets (`fields[]`, `sort[]`, `records[]`) to disable glob parsing
- IMPORTANT: When piping curl output to `jq` or other commands, environment variables like `$MATON_API_KEY` may not expand correctly in some shell environments. You may get "Invalid API key" errors when piping.

## Error Handling

| Status | Meaning |
|--------|---------|
| 400 | Missing Google Analytics connection |
| 401 | Invalid or missing Maton API key |
| 429 | Rate limited (10 req/sec per account) |
| 4xx/5xx | Passthrough error from Google Analytics API |

## Resources

- [Admin API Overview](https://developers.google.com/analytics/devguides/config/admin/v1)
- [Accounts](https://developers.google.com/analytics/devguides/config/admin/v1/rest/v1beta/accounts)
- [Properties](https://developers.google.com/analytics/devguides/config/admin/v1/rest/v1beta/properties)
- [Data Streams](https://developers.google.com/analytics/devguides/config/admin/v1/rest/v1beta/properties.dataStreams)
- [Data API Overview](https://developers.google.com/analytics/devguides/reporting/data/v1)
- [Run Report](https://developers.google.com/analytics/devguides/reporting/data/v1/rest/v1beta/properties/runReport)
- [Realtime Report](https://developers.google.com/analytics/devguides/reporting/data/v1/rest/v1beta/properties/runRealtimeReport)
