---
name: go-linter-configuration
description: "Configure and troubleshoot golangci-lint for Go projects. Handle import resolution issues, type-checking problems, and optimize configurations for both local and CI environments."
metadata:
  {
    "openclaw":
      {
        "emoji": "🔍",
        "requires": { "bins": ["go", "golangci-lint"] },
        "install":
          [
            {
              "id": "golang",
              "kind": "script",
              "script": "curl -L https://golang.org/dl/go1.21.5.linux-amd64.tar.gz | tar -C /usr/local -xzf -",
              "bins": ["go"],
              "label": "Install Go",
            },
            {
              "id": "golangci",
              "kind": "script",
              "script": "curl -sSfL https://raw.githubusercontent.com/golangci/golangci-lint/master/install.sh | sh -s -- -b $(go env GOPATH)/bin v1.59.1",
              "bins": ["golangci-lint"],
              "label": "Install golangci-lint",
            },
          ],
      },
  }
---

# Go Linter Configuration Skill

Configure and troubleshoot golangci-lint for Go projects. This skill helps handle import resolution issues, type-checking problems, and optimize configurations for both local and CI environments.

## Installation

Install golangci-lint:

```bash
go install github.com/golangci/golangci-lint/cmd/golangci-lint@latest
```

Or use the official installation script:

```bash
curl -sSfL https://raw.githubusercontent.com/golangci/golangci-lint/master/install.sh | sh -s -- -b $(go env GOPATH)/bin v1.59.1
```

## Basic Usage

Run linter on entire project:

```bash
golangci-lint run ./...
```

Run with specific configuration:

```bash
golangci-lint run --config .golangci.yml ./...
```

## Configuration File (.golangci.yml)

### Minimal Configuration (for CI environments with import issues)
```yaml
run:
  timeout: 5m
  tests: false
  build-tags: []

linters:
  disable-all: true
  enable:
    - gofmt          # Format checking only

linters-settings:
  gofmt:
    simplify: true

issues:
  exclude-use-default: false
  max-issues-per-linter: 50
  max-same-issues: 3

output:
  format: tab
```

### Standard Configuration (for local development)
```yaml
run:
  timeout: 5m
  tests: true
  build-tags: []

linters:
  enable:
    - gofmt
    - govet
    - errcheck
    - staticcheck
    - unused
    - gosimple
    - ineffassign

linters-settings:
  govet:
    enable:
      - shadow
  errcheck:
    check-type-assertions: true
  staticcheck:
    checks: ["all"]

issues:
  exclude-use-default: false
  max-issues-per-linter: 50
  max-same-issues: 3

output:
  format: tab
```

## Troubleshooting Common Issues

### "undefined: package" Errors
Problem: Linter reports undefined references to imported packages
Solution: Use minimal configuration with `disable-all: true` and only enable basic linters like `gofmt`

### Import Resolution Problems
Problem: CI environment cannot resolve dependencies properly
Solution: 
1. Ensure go.mod and go.sum are up to date
2. Use `go mod download` before running linter in CI
3. Consider using simpler linters in CI environment

### Type-Checking Failures
Problem: Linter fails during type checking phase
Solution:
1. Temporarily disable complex linters that require type checking
2. Use `--fast` flag for quicker, less intensive checks
3. Verify all imports are properly declared

## CI/CD Optimization

For GitHub Actions workflow:

```yaml
name: Code Quality

on:
  push:
    branches: [ main, master ]
  pull_request:
    branches: [ main, master ]

jobs:
  lint:
    runs-on: ubuntu-latest
    steps:
    - uses: actions/checkout@v4

    - name: Set up Go
      uses: actions/setup-go@v4
      with:
        go-version: '1.21'
        cache: true

    - name: Download dependencies
      run: go mod download

    - name: Install golangci-lint
      run: |
        curl -sSfL https://raw.githubusercontent.com/golangci/golangci-lint/master/install.sh | sh -s -- -b $(go env GOPATH)/bin v1.59.1

    - name: Lint
      run: golangci-lint run --config .golangci.yml ./...
```

## Linter Selection Guidelines

- **gofmt**: For formatting consistency
- **govet**: For semantic errors
- **errcheck**: For unchecked errors
- **staticcheck**: For static analysis
- **unused**: For dead code detection
- **gosimple**: For simplification suggestions
- **ineffassign**: For ineffective assignments

Choose linters based on project needs and CI performance requirements.