#!/bin/bash
# Gmail OAuth helper for headless servers
# Usage:
#   ./gmail-auth.sh                    # Interactive setup
#   ./gmail-auth.sh --url              # Just print auth URL
#   ./gmail-auth.sh --exchange CODE    # Exchange code for tokens

set -e

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m'

# Load credentials from gog config
GOG_CREDS="${HOME}/.config/gogcli/credentials.json"

if [[ ! -f "$GOG_CREDS" ]]; then
    echo -e "${RED}Error: No gog credentials found at $GOG_CREDS${NC}"
    echo "Run: gog auth credentials /path/to/client_secret.json"
    exit 1
fi

CLIENT_ID=$(python3 -c "import json; print(json.load(open('$GOG_CREDS'))['client_id'])")
CLIENT_SECRET=$(python3 -c "import json; print(json.load(open('$GOG_CREDS'))['client_secret'])")

# Default scope - gmail.modify covers most use cases
SCOPE="https://www.googleapis.com/auth/gmail.modify"
REDIRECT_URI="http://localhost"

generate_url() {
    local encoded_scope=$(python3 -c "import urllib.parse; print(urllib.parse.quote('$SCOPE'))")
    echo "https://accounts.google.com/o/oauth2/auth?access_type=offline&client_id=${CLIENT_ID}&prompt=consent&redirect_uri=${REDIRECT_URI}&response_type=code&scope=${encoded_scope}"
}

exchange_code() {
    local code="$1"
    local email="$2"
    
    echo -e "${YELLOW}Exchanging code for tokens...${NC}"
    
    response=$(curl -s -X POST https://oauth2.googleapis.com/token \
        -d "code=${code}" \
        -d "client_id=${CLIENT_ID}" \
        -d "client_secret=${CLIENT_SECRET}" \
        -d "redirect_uri=${REDIRECT_URI}" \
        -d "grant_type=authorization_code")
    
    # Check for error
    if echo "$response" | python3 -c "import json,sys; d=json.load(sys.stdin); sys.exit(0 if 'access_token' in d else 1)" 2>/dev/null; then
        refresh_token=$(echo "$response" | python3 -c "import json,sys; print(json.load(sys.stdin)['refresh_token'])")
        
        # Create token file for gog import
        token_file=$(mktemp)
        cat > "$token_file" << EOF
{
  "email": "${email}",
  "client": "default",
  "refresh_token": "${refresh_token}",
  "scopes": ["${SCOPE}"]
}
EOF
        
        echo -e "${GREEN}Token exchange successful!${NC}"
        echo ""
        echo "Importing to gog..."
        
        if [[ -z "$GOG_KEYRING_PASSWORD" ]]; then
            echo -e "${YELLOW}Note: Set GOG_KEYRING_PASSWORD environment variable for non-interactive import${NC}"
        fi
        
        gog auth tokens import "$token_file"
        rm "$token_file"
        
        echo ""
        echo -e "${GREEN}Done! Test with:${NC}"
        echo "  gog gmail search 'is:unread' --max 5 --account ${email}"
    else
        echo -e "${RED}Token exchange failed:${NC}"
        echo "$response" | python3 -m json.tool 2>/dev/null || echo "$response"
        exit 1
    fi
}

# Parse arguments
case "${1:-}" in
    --url)
        generate_url
        ;;
    --exchange)
        if [[ -z "${2:-}" ]]; then
            echo "Usage: $0 --exchange CODE [EMAIL]"
            exit 1
        fi
        email="${3:-}"
        if [[ -z "$email" ]]; then
            read -p "Gmail address to authorize: " email
        fi
        exchange_code "$2" "$email"
        ;;
    --help|-h)
        echo "Gmail OAuth Helper"
        echo ""
        echo "Usage:"
        echo "  $0              Interactive setup"
        echo "  $0 --url        Print auth URL only"
        echo "  $0 --exchange CODE [EMAIL]  Exchange code for tokens"
        echo ""
        echo "Environment:"
        echo "  GOG_KEYRING_PASSWORD  Required for non-interactive token import"
        ;;
    *)
        # Interactive mode
        echo -e "${GREEN}Gmail OAuth Setup${NC}"
        echo ""
        
        read -p "Gmail address to authorize: " email
        
        echo ""
        echo -e "${YELLOW}Step 1: Open this URL in a browser${NC}"
        echo ""
        generate_url
        echo ""
        echo -e "${YELLOW}Step 2: Sign in with ${email} and approve access${NC}"
        echo ""
        echo -e "${YELLOW}Step 3: Copy the authorization code shown on the page${NC}"
        echo ""
        read -p "Paste the code here: " code
        
        exchange_code "$code" "$email"
        ;;
esac
