const fs = require('fs');
const path = require('path');
const crypto = require('crypto');
const http = require('http');
const https = require('https');
const url = require('url');

// ============ 配置 ============
// 优先使用环境变量指定的路径，否则使用 skill 目录
const SKILL_ROOT = process.env.GITLAB_SKILL_ROOT || path.resolve(__dirname);
const CONFIG_DIR = path.join(SKILL_ROOT, 'config');
const USERS_FILE = path.join(CONFIG_DIR, 'users.enc'); // 加密存储的用户 token
const OAUTH_FILE = path.join(CONFIG_DIR, 'oauth.json');

// 加密密钥（从配置文件读取，或环境变量）
function getEncryptKey() {
  // 优先从环境变量读取
  if (process.env.GITLAB_ENCRYPT_KEY) {
    return process.env.GITLAB_ENCRYPT_KEY;
  }
  // 其次从密钥文件读取（使用 SKILL_ROOT 确保路径正确）
  const keyFile = path.join(SKILL_ROOT, 'config', 'key.txt');
  if (fs.existsSync(keyFile)) {
    return fs.readFileSync(keyFile, 'utf8').trim();
  }
  console.error('错误：必须设置加密密钥');
  console.error('方法1: export GITLAB_ENCRYPT_KEY=你的密钥');
  console.error('方法2: echo 你的密钥 > config/key.txt');
  process.exit(1);
}
const ENCRYPT_KEY = getEncryptKey();

// ============ 工具函数 ============

// 加密
function encrypt(text) {
  const iv = crypto.randomBytes(16);
  const cipher = crypto.createCipheriv('aes-256-cbc', Buffer.from(ENCRYPT_KEY.slice(0, 32)), iv);
  let encrypted = cipher.update(text, 'utf8', 'hex');
  encrypted += cipher.final('hex');
  return iv.toString('hex') + ':' + encrypted;
}

// 解密
function decrypt(text) {
  try {
    const parts = text.split(':');
    const iv = Buffer.from(parts[0], 'hex');
    const encryptedText = parts[1];
    const decipher = crypto.createDecipheriv('aes-256-cbc', Buffer.from(ENCRYPT_KEY.slice(0, 32)), iv);
    let decrypted = decipher.update(encryptedText, 'hex', 'utf8');
    decrypted += decipher.final('utf8');
    return decrypted;
  } catch (e) {
    return null;
  }
}

// 读取用户配置
function loadUsers() {
  if (!fs.existsSync(USERS_FILE)) {
    return {};
  }
  try {
    const encrypted = fs.readFileSync(USERS_FILE, 'utf8');
    const decrypted = decrypt(encrypted);
    return decrypted ? JSON.parse(decrypted) : {};
  } catch (e) {
    return {};
  }
}

// 获取用户 Token（按优先级）
function getUserToken(userId) {
  // 1. 优先从环境变量读取（OpenClaw 调用时）
  if (process.env.GITLAB_TOKEN) {
    return process.env.GITLAB_TOKEN;
  }
  
  // 2. 其次从用户配置文件读取
  const users = loadUsers();
  if (users[userId] && users[userId].token) {
    return users[userId].token;
  }
  
  // 3. 最后从全局 token 文件读取（ ~/.gitlab-token）
  const globalTokenFile = process.env.HOME + '/.gitlab-token';
  try {
    if (fs.existsSync(globalTokenFile)) {
      return fs.readFileSync(globalTokenFile, 'utf8').trim();
    }
  } catch (e) {}
  
  return null;
}

// 保存用户配置（加密）
function saveUsers(users) {
  if (!fs.existsSync(CONFIG_DIR)) {
    fs.mkdirSync(CONFIG_DIR, { recursive: true });
  }
  const encrypted = encrypt(JSON.stringify(users));
  fs.writeFileSync(USERS_FILE, encrypted);
}

// 读取 OAuth 配置（加密存储）
function loadOauthConfig() {
  if (!fs.existsSync(OAUTH_FILE)) {
    return null;
  }
  try {
    const encrypted = fs.readFileSync(OAUTH_FILE, 'utf8');
    const decrypted = decrypt(encrypted);
    return decrypted ? JSON.parse(decrypted) : null;
  } catch (e) {
    return null;
  }
}

// 保存 OAuth 配置（加密）
function saveOauthConfig(config) {
  if (!fs.existsSync(CONFIG_DIR)) {
    fs.mkdirSync(CONFIG_DIR, { recursive: true });
  }
  const encrypted = encrypt(JSON.stringify(config));
  fs.writeFileSync(OAUTH_FILE, encrypted);
}

// API 请求
function request(token, method, endpoint, data = null) {
  return new Promise((resolve, reject) => {
    const gitlabUrl = process.env.GITLAB_URL || 'http://gitlab.example.com';
    const parsedUrl = new url.parse(gitlabUrl + endpoint);
    const isHttps = gitlabUrl.startsWith('https');
    const lib = isHttps ? https : http;

    const options = {
      hostname: parsedUrl.hostname,
      port: parsedUrl.port || (isHttps ? 443 : 80),
      path: parsedUrl.path,
      method: method,
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${token}`
      }
    };

    const req = lib.request(options, (res) => {
      let body = '';
      res.on('data', chunk => body += chunk);
      res.on('end', () => {
        try {
          resolve(JSON.parse(body));
        } catch (e) {
          resolve(body);
        }
      });
    });

    req.on('error', reject);
    if (data) {
      req.write(JSON.stringify(data));
    }
    req.end();
  });
}

// ============ 主逻辑 ============

const command = process.argv[2];
const args = process.argv.slice(3);

async function main() {
  switch (command) {
    case 'auth-url':
      // 生成授权链接
      const oauthConfig = loadOauthConfig();
      if (!oauthConfig) {
        console.log('错误：请先配置 OAuth 应用: node index.js config <Application ID> <Secret>');
        process.exit(1);
      }
      const redirectUri = 'http://localhost/callback';
      const authUrl = `${oauthConfig.gitlab_url || 'http://gitlab.example.com'}/oauth/authorize?client_id=${oauthConfig.client_id}&redirect_uri=${redirectUri}&response_type=code&scope=api`;
      console.log(authUrl);
      break;

    case 'setup':
      // 初始化设置（生成加密密钥）
      const crypto = require('crypto');
      const key = crypto.randomBytes(32).toString('base64');
      if (!fs.existsSync(CONFIG_DIR)) {
        fs.mkdirSync(CONFIG_DIR, { recursive: true });
      }
      fs.writeFileSync(path.join(CONFIG_DIR, 'key.txt'), key);
      console.log('✅ 加密密钥已生成并保存到 config/key.txt');
      console.log('注意：请妥善保管此密钥，丢失后将无法解密数据');
      break;

    case 'config':
      // 配置 OAuth 应用
      const clientId = args[0];
      const clientSecret = args[1];
      const gitlabUrl = args[2] || 'http://gitlab.example.com';
      
      if (!clientId || !clientSecret) {
        console.log('用法: node index.js config <Application ID> <Secret> [GitLab URL]');
        process.exit(1);
      }

      const configData = {
        client_id: clientId,
        client_secret: clientSecret,
        gitlab_url: gitlabUrl
      };
      
      saveOauthConfig(configData);
      console.log('✅ OAuth 配置已保存（加密存储）');
      break;

    case 'handle':
      // 处理用户发来的授权链接或 code
      const userId = args[0]; // 用户ID
      const input = args[1]; // 授权链接或 code
      
      if (!input) {
        console.log('用法: node index.js handle <用户ID> <授权链接或code>');
        console.log('');
        console.log('自动识别处理：');
        console.log('- 如果是完整授权链接，自动提取 code 并换取 token');
        console.log('- 如果是纯 code，直接换取 token');
        console.log('- token 加密保存到本地');
        process.exit(1);
      }

      // 提取 code
      let code = input;
      if (input.includes('code=')) {
        const urlMatch = input.match(/[?&]code=([a-f0-9]+)/i);
        if (urlMatch) {
          code = urlMatch[1];
          console.log('从授权链接中提取到 code');
        }
      }

      // 获取 OAuth 配置
      const oauth = loadOauthConfig();
      if (!oauth) {
        console.log('错误：请先配置 OAuth 应用');
        process.exit(1);
      }

      // 换取 token
      console.log('正在换取 token...');
      const tokenData = await new Promise((resolve, reject) => {
        const postData = new url.URLSearchParams({
          client_id: oauth.client_id,
          client_secret: oauth.client_secret,
          code: code,
          grant_type: 'authorization_code',
          redirect_uri: 'http://localhost/callback'
        }).toString();

        const gitlabUrl = oauth.gitlab_url || 'http://gitlab.example.com';
        const parsedUrl = new url.parse(gitlabUrl + '/oauth/token');
        const isHttps = gitlabUrl.startsWith('https');
        const lib = isHttps ? https : http;

        const options = {
          hostname: parsedUrl.hostname,
          port: parsedUrl.port || (isHttps ? 443 : 80),
          path: parsedUrl.path,
          method: 'POST',
          headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
            'Content-Length': Buffer.byteLength(postData)
          }
        };

        const req = lib.request(options, (res) => {
          let body = '';
          res.on('data', chunk => body += chunk);
          res.on('end', () => {
            try {
              resolve(JSON.parse(body));
            } catch (e) {
              reject(e);
            }
          });
        });

        req.on('error', reject);
        req.write(postData);
        req.end();
      });

      if (tokenData.access_token) {
        // 获取用户信息
        const userInfo = await request(tokenData.access_token, 'GET', '/api/v4/user');
        
        console.log('✅ 授权成功！');
        console.log(`用户: ${userInfo.name} (${userInfo.username})`);
        
        // 保存用户 token（加密）
        const users = loadUsers();
        users[userId] = {
          token: tokenData.access_token,
          userId: userId,
          gitlabUser: userInfo.username,
          name: userInfo.name,
          updatedAt: new Date().toISOString()
        };
        saveUsers(users);
        
        console.log(`Token 已加密保存到本地，绑定用户: ${userId}`);
      } else {
        console.log('❌ 授权失败:', tokenData.error_description || tokenData.msg);
      }
      break;

    case 'use':
      // 使用指定用户的 token 执行命令
      const useUserId = args[0];
      const targetCommand = args[1];
      const targetArgs = args.slice(2);
      
      const users = loadUsers();
      const userData = users[useUserId];
      
      if (!userData || !userData.token) {
        console.log(`错误：用户 ${useUserId} 未授权，请先运行: node index.js handle ${useUserId} <授权链接>`);
        process.exit(1);
      }

      const token = userData.token;

      switch (targetCommand) {
        case 'list':
          // 列出仓库
          const projects = await request(token, 'GET', '/api/v4/users/current/projects');
          console.log(`用户 ${userData.name} 的仓库:`);
          (Array.isArray(projects) ? projects : []).forEach(p => {
            console.log(`- ${p.name} (ID: ${p.id})`);
          });
          break;

        case 'project':
          const keyword = targetArgs[0];
          const searchResult = await request(token, 'GET', `/api/v4/projects?search=${keyword}`);
          console.log(`搜索 "${keyword}" 结果:`);
          (Array.isArray(searchResult) ? searchResult : []).forEach(p => {
            console.log(`- ${p.name_with_namespace} (ID: ${p.id})`);
          });
          break;

        case 'merge':
          const projectId = targetArgs[0];
          const branch = targetArgs[1] || 'master';
          
          const commits = await request(token, 'GET', `/api/v4/projects/${projectId}/repository/commits?ref_name=${branch}&per_page=10`);
          const mergeCommit = (Array.isArray(commits) ? commits : []).find(c => c.parent_ids && c.parent_ids.length > 1);
          
          if (!mergeCommit) {
            console.log('未找到合并提交');
            break;
          }

          console.log('='.repeat(50));
          console.log('最后一次合并信息');
          console.log('='.repeat(50));
          console.log(`时间: ${mergeCommit.created_at}`);
          console.log(`作者: ${mergeCommit.author_name}`);
          console.log(`提交: ${mergeCommit.title}`);
          
          const diff = await request(token, 'GET', `/api/v4/projects/${projectId}/repository/commits/${mergeCommit.id}/diff`);
          console.log('变更文件:');
          (Array.isArray(diff) ? diff : []).forEach(f => {
            console.log(`  - ${f.new_path || f.old_path}`);
          });
          break;

        default:
          console.log('用法: node index.js use <用户ID> <命令> [参数]');
          console.log('命令: list, project <关键词>, merge <项目ID> [分支]');
      }
      break;

    case 'users':
      // 列出已授权用户
      const allUsers = loadUsers();
      console.log('已授权用户:');
      Object.keys(allUsers).forEach(uid => {
        const u = allUsers[uid];
        console.log(`- ${uid}: ${u.name} (@${u.gitlabUser}), 更新时间: ${u.updatedAt}`);
      });
      break;

    case 'process':
      // 处理用户消息（自动检测授权链接或 GitLab 仓库链接）
      const msgUserId = args[0];
      const message = args.slice(1).join(' ');
      
      if (!msgUserId || !message) {
        console.log('用法: node index.js process <用户ID> <消息内容>');
        console.log('');
        console.log('自动处理：');
        console.log('- 如果消息包含授权链接，自动处理授权');
        console.log('- 如果消息包含 GitLab 链接，检查是否已授权');
        process.exit(1);
      }

      console.log(`处理用户 ${msgUserId} 的消息: ${message.substring(0, 50)}...`);

      // 检查是否包含授权链接
      if (message.includes('localhost/callback') || message.includes('code=')) {
        console.log('检测到授权链接，正在处理...');
        
        // 直接处理授权，不递归调用
        // 提取 code
        let code = message;
        if (message.includes('code=')) {
          const urlMatch = message.match(/[?&]code=([a-z0-9]+)/i);
          if (urlMatch) {
            code = urlMatch[1];
          }
        }
        
        // 加载 OAuth 配置
        const oauth = loadOauthConfig();
        if (!oauth) {
          console.log('错误：请先配置 OAuth 应用');
          console.log('回复: 抱歉，GitLab 授权未配置，请联系管理员');
          break;
        }

        // 换取 token
        console.log('正在换取 token...');
        const tokenData = await new Promise((resolve, reject) => {
          const postData = new url.URLSearchParams({
            client_id: oauth.client_id,
            client_secret: oauth.client_secret,
            code: code,
            grant_type: 'authorization_code',
            redirect_uri: 'http://localhost/callback'
          }).toString();

          const parsedUrl = new url.parse(oauth.gitlab_url + '/oauth/token');
          const isHttps = oauth.gitlab_url.startsWith('https');
          const lib = isHttps ? https : http;

          const options = {
            hostname: parsedUrl.hostname,
            port: parsedUrl.port || (isHttps ? 443 : 80),
            path: parsedUrl.path,
            method: 'POST',
            headers: {
              'Content-Type': 'application/x-www-form-urlencoded',
              'Content-Length': Buffer.byteLength(postData)
            }
          };

          const req = lib.request(options, (res) => {
            let body = '';
            res.on('data', chunk => body += chunk);
            res.on('end', () => {
              try {
                resolve(JSON.parse(body));
              } catch (e) {
                reject(e);
              }
            });
          });

          req.on('error', reject);
          req.write(postData);
          req.end();
        });

        if (tokenData.access_token) {
          // 获取用户信息
          const userInfo = await request(tokenData.access_token, 'GET', '/api/v4/user');
          
          console.log('✅ 授权成功！');
          console.log(`用户: ${userInfo.name} (@${userInfo.username})`);
          
          // 保存用户 token（加密）
          const users = loadUsers();
          users[msgUserId] = {
            token: tokenData.access_token,
            userId: msgUserId,
            gitlabUser: userInfo.username,
            name: userInfo.name,
            updatedAt: new Date().toISOString()
          };
          saveUsers(users);
          
          console.log(`Token 已加密保存到本地，绑定用户: ${msgUserId}`);
          console.log('回复: ✅ 授权成功！后续可直接使用 GitLab 功能。');
        } else {
          console.log('❌ 授权失败:', tokenData.error_description || tokenData.msg);
          console.log('回复: ❌ 授权失败，请重新点击授权链接。');
        }
        break;
      }

      // 检查是否包含 GitLab 仓库链接
      const gitlabLinkMatch = message.match(/gitlab[a-zA-Z0-9_.\-/]+/i);
      if (gitlabLinkMatch) {
        // 检查用户是否已授权（使用 getUserToken 支持多个来源）
        const token = getUserToken(msgUserId);
        if (token) {
          console.log('✅ 用户已授权（通过全局token），直接处理请求');
          console.log('请提供具体要执行的操作');
        } else {
          // 生成授权链接
          const oauth = loadOauthConfig();
          if (!oauth) {
            console.log('错误：请先配置 OAuth 应用');
            console.log('回复: 抱歉，GitLab 授权未配置，请联系管理员');
            break;
          }
          const authUrl = `${oauth.gitlab_url}/oauth/authorize?client_id=${oauth.client_id}&redirect_uri=http://localhost/callback&response_type=code&scope=api`;
          console.log('用户未授权，需要引导授权');
          console.log('回复: 请按以下步骤授权：\\n1. 打开浏览器访问：' + authUrl + '\\n2. 授权后把跳转的完整链接发给我');
        }
      }
      break;

    default:
      console.log(`
GitLab Private v0.5.0 - 支持多用户授权

用法:
  node index.js config <Application ID> <Secret> [GitLab URL]    - 配置 OAuth 应用
  node index.js auth-url                                          - 生成授权链接
  node index.js handle <用户ID> <授权链接或code>                  - 处理授权（核心）
  node index.js use <用户ID> <命令> [参数]                        - 使用用户 token 执行命令
  node index.js users                                             - 列出已授权用户

命令示例:
  node index.js use <用户ID> merge <项目ID> [分支]
  node index.js use <用户ID> project <关键词>
  node index.js use <用户ID> list
`);
  }
}

main().catch(console.error);
