#!/usr/bin/env node

const fs = require('fs');
const path = require('path');

const GITLAB_API = 'https://gitlab.com/api/v4';
const TOKEN = process.env.GITLAB_TOKEN;

if (!TOKEN) {
  console.error('Error: GITLAB_TOKEN environment variable is required.');
  process.exit(1);
}

const args = process.argv.slice(2);
const command = args[0];

async function request(endpoint, method = 'GET', body = null) {
  const headers = {
    'Private-Token': TOKEN,
    'Content-Type': 'application/json',
  };
  
  const options = { method, headers };
  if (body) options.body = JSON.stringify(body);

  const res = await fetch(`${GITLAB_API}${endpoint}`, options);
  
  if (!res.ok) {
    const err = await res.text();
    throw new Error(`GitLab API Error (${res.status}): ${err}`);
  }
  
  return res.json();
}

async function createRepo(name, description, visibility = 'private') {
  console.log(`Creating repository: ${name}...`);
  const data = await request('/projects', 'POST', {
    name,
    description,
    visibility,
    initialize_with_readme: true
  });
  console.log(`Repository created: ${data.web_url}`);
  return data;
}

async function listMergeRequests(projectId, state = 'opened') {
  console.log(`Listing ${state} MRs for project ${projectId}...`);
  // If projectId is 'jorgermp/repo', we need to URL encode it or use ID
  const id = encodeURIComponent(projectId);
  const data = await request(`/projects/${id}/merge_requests?state=${state}`);
  console.log(JSON.stringify(data, null, 2));
}

async function commentOnMergeRequest(projectId, mrIid, body) {
  console.log(`Commenting on MR #${mrIid} in ${projectId}...`);
  const id = encodeURIComponent(projectId);
  const data = await request(`/projects/${id}/merge_requests/${mrIid}/notes`, 'POST', { body });
  console.log(`Comment added: ${data.web_url}`);
}

async function createIssue(projectId, title, description) {
  console.log(`Creating issue in ${projectId}...`);
  const id = encodeURIComponent(projectId);
  const data = await request(`/projects/${id}/issues`, 'POST', { title, description });
  console.log(`Issue created: ${data.web_url}`);
}

async function main() {
  try {
    switch (command) {
      case 'create_repo':
        await createRepo(args[1], args[2], args[3]);
        break;
      case 'list_mrs':
        await listMergeRequests(args[1], args[2]);
        break;
      case 'comment_mr':
        await commentOnMergeRequest(args[1], args[2], args[3]);
        break;
      case 'create_issue':
        await createIssue(args[1], args[2], args[3]);
        break;
      default:
        console.log(`
Usage:
  gitlab_api.js create_repo <name> [description] [visibility]
  gitlab_api.js list_mrs <project_id_or_path> [state]
  gitlab_api.js comment_mr <project_id_or_path> <mr_iid> <body>
  gitlab_api.js create_issue <project_id_or_path> <title> [description]
        `);
    }
  } catch (err) {
    console.error(err.message);
    process.exit(1);
  }
}

main();
