---
name: Git (Essentials + Workflows + Advanced)
slug: git
version: 1.0.7
description: Full version control coverage with essential commands, team workflows, branching strategies, and recovery techniques.
homepage: https://clawic.com/skills/git
changelog: Translated all auxiliary files to English
metadata: {"clawdbot":{"emoji":"📚","requires":{"bins":["git"]},"os":["linux","darwin","win32"]}}
---

## Setup

On first use, read `setup.md`. Default: best practices mode (no config needed).

## When to Use

User needs Git expertise — from basic operations to complex workflows. Agent handles branching, merging, rebasing, conflict resolution, and team collaboration patterns.

## Architecture

Memory in `~/git/`. See `memory-template.md` for structure.

```
~/git/
└── memory.md    # User preferences (optional)
```

## Quick Reference

| Topic | File |
|-------|------|
| Essential commands | `commands.md` |
| Advanced operations | `advanced.md` |
| Branch strategies | `branching.md` |
| Conflict resolution | `conflicts.md` |
| History and recovery | `history.md` |
| Team workflows | `collaboration.md` |
| Setup | `setup.md` |
| Memory | `memory-template.md` |

## Core Rules

1. **Never force push to shared branches** — Use `--force-with-lease` on feature branches only
2. **Commit early, commit often** — Small commits are easier to review, revert, and bisect
3. **Write meaningful commit messages** — First line under 72 chars, imperative mood
4. **Pull before push** — Always `git pull --rebase` before pushing to avoid merge commits
5. **Clean up before merging** — Use `git rebase -i` to squash fixup commits

## Team Workflows

**Feature Branch Flow:**
1. `git checkout -b feature/name` from main
2. Make commits, push regularly
3. Open PR, get review
4. Squash and merge to main
5. Delete feature branch

**Hotfix Flow:**
1. `git checkout -b hotfix/issue` from main
2. Fix, test, commit
3. Merge to main AND develop (if exists)
4. Tag the release

**Daily Sync:**
```bash
git fetch --all --prune
git rebase origin/main  # or merge if team prefers
```

## Commit Messages

- Use conventional commit format: `type(scope): description`
- Keep first line under 72 characters
- Types: `feat`, `fix`, `docs`, `style`, `refactor`, `test`, `chore`

## Push Safety

- Use `git push --force-with-lease` instead of `--force` — prevents overwriting others' work
- If push rejected, run `git pull --rebase` before retrying
- Never force push to main/master branch

## Conflict Resolution

- After editing conflicted files, verify no markers remain: `grep -r "<<<\|>>>\|===" .`
- Test that code builds before completing merge
- If merge becomes complex, abort with `git merge --abort` and try `git rebase` instead

## Branch Hygiene

- Delete merged branches locally: `git branch -d branch-name`
- Clean remote tracking: `git fetch --prune`
- Before creating PR, rebase feature branch onto latest main
- Use `git rebase -i` to squash messy commits before pushing

## Safety Checklist

Before destructive operations (`reset --hard`, `rebase`, `force push`):

- [ ] Is this a shared branch? → Don't rewrite history
- [ ] Do I have uncommitted changes? → Stash or commit first
- [ ] Am I on the right branch? → `git branch` to verify
- [ ] Is remote up to date? → `git fetch` first

## Common Traps

- **git user.email wrong** — Verify with `git config user.email` before important commits
- **Empty directories** — Git doesn't track them, add `.gitkeep`
- **Submodules** — Always clone with `--recurse-submodules`
- **Detached HEAD** — Use `git switch -` to return to previous branch
- **Push rejected** — Usually needs `git pull --rebase` first
- **stash pop on conflict** — Stash disappears. Use `stash apply` instead
- **Large files** — Use Git LFS for files >50MB, never commit secrets
- **Case sensitivity** — Mac/Windows ignore case, Linux doesn't — causes CI failures

## Recovery Commands

- Undo last commit keeping changes: `git reset --soft HEAD~1`
- Discard unstaged changes: `git restore filename`
- Find lost commits: `git reflog` (keeps ~90 days of history)
- Recover deleted branch: `git checkout -b branch-name <sha-from-reflog>`
- Use `git add -p` for partial staging when commit mixes multiple changes

## Debugging with Bisect

Find the commit that introduced a bug:
```bash
git bisect start
git bisect bad                    # current commit is broken
git bisect good v1.0.0            # this version worked
# Git checks out middle commit, test it, then:
git bisect good                   # or git bisect bad
# Repeat until Git finds the culprit
git bisect reset                  # return to original branch
```

## Quick Summary

```bash
git status -sb                    # short status with branch
git log --oneline -5              # last 5 commits
git shortlog -sn                  # contributors by commit count
git diff --stat HEAD~5            # changes summary last 5 commits
git branch -vv                    # branches with tracking info
git stash list                    # pending stashes
```

## Related Skills
Install with `clawhub install <slug>` if user confirms:
- `gitlab` — GitLab CI/CD and merge requests
- `docker` — Containerization workflows
- `code` — Code quality and best practices

## Feedback

- If useful: `clawhub star git`
- Stay updated: `clawhub sync`
