#!/usr/bin/env python3
import os
import zipfile
import json
import yaml
import sys

def validate_skill(skill_dir):
    """Validate the skill before packaging"""
    skill_path = os.path.join(skill_dir, "SKILL.md")
    
    if not os.path.exists(skill_path):
        print(f"Error: SKILL.md not found in {skill_dir}")
        return False
    
    with open(skill_path, 'r') as f:
        content = f.read()
    
    # Extract frontmatter
    if not content.startswith('---'):
        print("Error: SKILL.md must start with YAML frontmatter (---)")
        return False
    
    parts = content.split('---', 2)
    if len(parts) < 3:
        print("Error: Invalid YAML frontmatter format")
        return False
    
    try:
        frontmatter = yaml.safe_load(parts[1])
    except yaml.YAMLError as e:
        print(f"Error: Invalid YAML in frontmatter: {e}")
        return False
    
    # Check required fields
    if 'name' not in frontmatter:
        print("Error: Missing required 'name' field in frontmatter")
        return False
    
    if 'description' not in frontmatter:
        print("Error: Missing required 'description' field in frontmatter")
        return False
    
    # Validate name format
    name = frontmatter['name']
    if not name.replace('-', '').replace('_', '').isalnum():
        print("Error: Name must contain only letters, digits, hyphens, and underscores")
        return False
    
    print(f"✓ Validation passed: {name}")
    print(f"  Description: {frontmatter['description'][:60]}...")
    return True

def package_skill(skill_dir, output_dir=None):
    """Package the skill into a .skill file"""
    skill_name = os.path.basename(skill_dir)
    
    if output_dir:
        os.makedirs(output_dir, exist_ok=True)
        output_path = os.path.join(output_dir, f"{skill_name}.skill")
    else:
        output_path = f"{skill_name}.skill"
    
    with zipfile.ZipFile(output_path, 'w', zipfile.ZIP_DEFLATED) as zipf:
        for root, dirs, files in os.walk(skill_dir):
            for file in files:
                file_path = os.path.join(root, file)
                arcname = os.path.relpath(file_path, skill_dir)
                zipf.write(file_path, arcname)
    
    print(f"✓ Packaged skill: {output_path}")
    return output_path

if __name__ == "__main__":
    if len(sys.argv) < 2:
        print("Usage: package_skill.py <skill-directory> [output-directory]")
        sys.exit(1)
    
    skill_dir = sys.argv[1]
    output_dir = sys.argv[2] if len(sys.argv) > 2 else None
    
    if not os.path.exists(skill_dir):
        print(f"Error: Directory not found: {skill_dir}")
        sys.exit(1)
    
    print(f"Packaging skill: {skill_dir}")
    
    if validate_skill(skill_dir):
        package_skill(skill_dir, output_dir)
    else:
        sys.exit(1)
