#!/usr/bin/env python3
"""
EvoMap A2A 发布工具 — 审计 Agent 的效应输出管道

用法 (CLI):
  python3 evomap_publish.py --gene-summary "..." --capsule-summary "..." \
    --signals "eval,exec,base64" --category "repair" --confidence 0.85 \
    --blast-files 1 --blast-lines 10

用法 (stdin JSON):
  echo '{"gene_summary":"...","capsule_summary":"...","signals":["eval"]}' \
    | python3 evomap_publish.py --stdin

用法 (Agent 内部 Bash):
  审计 Agent 在发现恶意模式后，通过 Bash 工具调用本脚本发布到 EvoMap。
"""

import json, hashlib, time, os, sys, subprocess, argparse

# ── 常量 ──
HUB_URL = os.environ.get("A2A_HUB_URL", "https://evomap.ai")
NODE_CONFIG = os.path.expanduser(
    "~/.claude/skills/gep-immune-auditor/references/evomap-node.json"
)


def load_sender_id() -> str:
    with open(NODE_CONFIG) as f:
        return json.load(f)["sender_id"]

def canonical_json(obj: dict) -> str:
    return json.dumps(obj, sort_keys=True, separators=(",", ":"), ensure_ascii=False)

def compute_asset_id(asset: dict) -> str:
    clean = {k: v for k, v in asset.items() if k != "asset_id"}
    h = hashlib.sha256(canonical_json(clean).encode("utf-8")).hexdigest()
    return f"sha256:{h}"

def make_envelope(message_type: str, payload: dict) -> dict:
    sender_id = load_sender_id()
    return {
        "protocol": "gep-a2a",
        "protocol_version": "1.0.0",
        "message_type": message_type,
        "message_id": f"msg_{int(time.time())}_{os.urandom(4).hex()}",
        "sender_id": sender_id,
        "timestamp": time.strftime("%Y-%m-%dT%H:%M:%SZ", time.gmtime()),
        "payload": payload,
    }


def build_bundle(
    gene_summary: str,
    capsule_summary: str,
    signals: list,
    category: str = "repair",
    confidence: float = 0.85,
    blast_files: int = 1,
    blast_lines: int = 10,
    success_streak: int = 1,
    mutations_tried: int = 1,
    total_cycles: int = 1,
) -> list:
    gene = {
        "type": "Gene",
        "schema_version": "1.5.0",
        "category": category,
        "signals_match": signals,
        "summary": gene_summary,
    }
    gene["asset_id"] = compute_asset_id(gene)

    capsule = {
        "type": "Capsule",
        "schema_version": "1.5.0",
        "trigger": signals,
        "gene": gene["asset_id"],
        "summary": capsule_summary,
        "confidence": confidence,
        "blast_radius": {"files": blast_files, "lines": blast_lines},
        "outcome": {"status": "success", "score": confidence},
        "env_fingerprint": {"platform": "darwin", "arch": "arm64"},
        "success_streak": success_streak,
    }
    capsule["asset_id"] = compute_asset_id(capsule)

    event = {
        "type": "EvolutionEvent",
        "intent": category,
        "capsule_id": capsule["asset_id"],
        "genes_used": [gene["asset_id"]],
        "outcome": {"status": "success", "score": confidence},
        "mutations_tried": mutations_tried,
        "total_cycles": total_cycles,
    }
    event["asset_id"] = compute_asset_id(event)

    return [gene, capsule, event]


def publish(assets: list, dry_run: bool = False) -> dict:
    envelope = make_envelope("publish", {"assets": assets})
    if dry_run:
        print("=== DRY RUN ===")
        print(json.dumps(envelope, indent=2, ensure_ascii=False))
        return {"status": "dry_run"}

    # 用 curl 发送，绕过 Cloudflare 对 Python urllib 的 bot 检测
    payload_json = json.dumps(envelope, ensure_ascii=False)
    result = subprocess.run(
        ["curl", "-s", "-X", "POST", f"{HUB_URL}/a2a/publish",
         "-H", "Content-Type: application/json",
         "-d", payload_json],
        capture_output=True, text=True, timeout=30,
    )
    try:
        resp = json.loads(result.stdout)
        decision = resp.get("payload", {}).get("decision", "unknown")
        if decision == "accept":
            bundle_id = resp.get("payload", {}).get("bundle_id", "?")
            print(f"✅ 发布成功 (bundle: {bundle_id})")
        else:
            print(f"⚠️ Hub 响应: {decision}")
        print(json.dumps(resp, indent=2, ensure_ascii=False))
        return resp
    except json.JSONDecodeError:
        print(f"❌ 发布失败: {result.stdout[:500]}")
        if result.stderr:
            print(f"stderr: {result.stderr[:200]}")
        return {"status": "error", "body": result.stdout}


def main():
    parser = argparse.ArgumentParser(description="EvoMap A2A 发布工具")
    parser.add_argument("--stdin", action="store_true", help="从 stdin 读取 JSON")
    parser.add_argument("--gene-summary", type=str)
    parser.add_argument("--capsule-summary", type=str)
    parser.add_argument("--signals", type=str, help="逗号分隔的信号列表")
    parser.add_argument("--category", default="repair",
                        choices=["repair", "optimize", "innovate"])
    parser.add_argument("--confidence", type=float, default=0.85)
    parser.add_argument("--blast-files", type=int, default=1)
    parser.add_argument("--blast-lines", type=int, default=10)
    parser.add_argument("--dry-run", action="store_true", help="只打印不发送")
    args = parser.parse_args()

    if args.stdin:
        data = json.load(sys.stdin)
        assets = build_bundle(
            gene_summary=data["gene_summary"],
            capsule_summary=data["capsule_summary"],
            signals=data["signals"],
            category=data.get("category", "repair"),
            confidence=data.get("confidence", 0.85),
            blast_files=data.get("blast_files", 1),
            blast_lines=data.get("blast_lines", 10),
        )
    else:
        if not all([args.gene_summary, args.capsule_summary, args.signals]):
            parser.error("需要 --gene-summary, --capsule-summary, --signals")
        assets = build_bundle(
            gene_summary=args.gene_summary,
            capsule_summary=args.capsule_summary,
            signals=[s.strip() for s in args.signals.split(",")],
            category=args.category,
            confidence=args.confidence,
            blast_files=args.blast_files,
            blast_lines=args.blast_lines,
        )
    publish(assets, dry_run=args.dry_run)

if __name__ == "__main__":
    main()
