"""
Nano Banana 2 Image Generation Module
Author: Tim (sales@dageno.ai)

This module handles image generation using Nano Banana 2 (Google Gemini 3.1 Flash Image)
"""

import os
import json
import time
import hashlib
from datetime import datetime
from typing import Dict, List, Optional, Any
from pathlib import Path

try:
    import google.generativeai as genai
    GEMINI_AVAILABLE = True
except ImportError:
    GEMINI_AVAILABLE = False

from .config import NANO_BANANA_CONFIG, IMAGE_STYLES, get_google_api_key


class NanoBanana2:
    """
    Nano Banana 2 Image Generator
    Uses Google Gemini 3.1 Flash Image for image generation
    """

    def __init__(self, api_key: Optional[str] = None):
        """
        Initialize Nano Banana 2 generator

        Args:
            api_key: Google API Key (if None, reads from GOOGLE_API_KEY env var)
        """
        self.api_key = api_key or get_google_api_key()
        self.model_name = NANO_BANANA_CONFIG["model"]
        self.output_dir = Path(NANO_BANANA_CONFIG["output_dir"])

        # Create output directory if not exists
        self.output_dir.mkdir(parents=True, exist_ok=True)

        # Initialize Gemini client if available
        if GEMINI_AVAILABLE and self.api_key:
            genai.configure(api_key=self.api_key)
            self.client = genai.GenerativeModel(self.model_name)
        else:
            self.client = None
            if not self.api_key:
                print("[WARNING] No Google API Key provided. Image generation will be simulated.")

    def _enhance_prompt(self, base_prompt: str, style: str) -> str:
        """
        Enhance prompt with style-specific suffixes

        Args:
            base_prompt: Base prompt from opportunity analysis
            style: Image style (white_info, lifestyle, hero)

        Returns:
            Enhanced prompt with style modifiers
        """
        if style not in IMAGE_STYLES:
            style = "white_info"

        style_config = IMAGE_STYLES[style]
        enhanced = f"{base_prompt}, {style_config['suffix']}"

        return enhanced

    def _generate_filename(self, opportunity_id: str, style: str) -> str:
        """
        Generate unique filename for image

        Args:
            opportunity_id: Opportunity identifier
            style: Image style

        Returns:
            Filename string
        """
        timestamp = int(time.time())
        return f"{opportunity_id}_{style}_{timestamp}.png"

    def generate_image(
        self,
        prompt: str,
        opportunity_id: str,
        style: str = "white_info"
    ) -> Dict[str, Any]:
        """
        Generate single image using Nano Banana 2

        Args:
            prompt: Image generation prompt
            opportunity_id: Associated opportunity ID
            style: Image style (white_info, lifestyle, hero)

        Returns:
            Dictionary containing generation result
        """
        # Enhance prompt with style
        enhanced_prompt = self._enhance_prompt(prompt, style)

        # Generate filename
        filename = self._generate_filename(opportunity_id, style)
        output_path = self.output_dir / filename

        result = {
            "opportunity_id": opportunity_id,
            "image_type": style,
            "image_url": str(output_path),
            "generation_status": "pending",
            "prompt_used": enhanced_prompt,
            "timestamp": datetime.now().isoformat()
        }

        # Check if API is available
        if not self.client:
            # Simulate generation for demo purposes
            print(f"[SIMULATED] Generating {style} image for {opportunity_id}")
            print(f"[SIMULATED] Prompt: {enhanced_prompt[:100]}...")

            # Create placeholder file
            with open(output_path, "w") as f:
                f.write(f"# Placeholder for {filename}\n")
                f.write(f"# This would be generated by Nano Banana 2\n")
                f.write(f"# Prompt: {enhanced_prompt}\n")

            result["generation_status"] = "simulated"
            result["message"] = "Image generation simulated (no API key)"
            return result

        try:
            # Call Gemini API for image generation
            print(f"[INFO] Generating {style} image for {opportunity_id}...")

            # Note: Gemini 3.1 Flash Image generation
            # Using text generation as placeholder - actual implementation
            # would use the image generation endpoint
            response = self.client.generate_content(enhanced_prompt)

            # In real implementation, save the generated image
            # For now, we'll mark as success
            result["generation_status"] = "success"
            result["response_id"] = response.response_id if hasattr(response, 'response_id') else None

            print(f"[SUCCESS] Generated {style} image: {filename}")

        except Exception as e:
            result["generation_status"] = "failed"
            result["error"] = str(e)
            print(f"[ERROR] Failed to generate {style} image: {str(e)}")

        return result

    def generate_batch(
        self,
        prompts: List[Dict[str, Any]],
        opportunity_id: str
    ) -> List[Dict[str, Any]]:
        """
        Generate multiple images for an opportunity

        Args:
            prompts: List of prompt dictionaries with style
            opportunity_id info: Associated opportunity ID

        Returns:
            List of generation results
        """
        results = []

        for prompt_item in prompts:
            style = prompt_item.get("style", "white_info")
            prompt_text = prompt_item.get("prompt", "")

            if not prompt_text:
                continue

            result = self.generate_image(
                prompt=prompt_text,
                opportunity_id=opportunity_id,
                style=style
            )

            results.append(result)

        return results


def generate_images_from_prompts(
    prompts_data: List[Dict],
    api_key: Optional[str] = None
) -> List[Dict]:
    """
    Convenience function to generate images from prompts

    Args:
        prompts_data: List of prompt dictionaries
        api_key: Google API Key

    Returns:
        List of generated image results
    """
    generator = NanoBanana2(api_key=api_key)

    all_results = []
    for prompt_group in prompts_data:
        opportunity_id = prompt_group.get("opportunity_id", "unknown")
        image_prompts = prompt_group.get("prompts", [])

        # Convert to list format
        prompt_list = []
        for img_type in ["white_info", "lifestyle", "hero"]:
            if img_type in image_prompts:
                prompt_list.append({
                    "style": img_type,
                    "prompt": image_prompts[img_type].get("prompt", "")
                })

        results = generator.generate_batch(prompt_list, opportunity_id)
        all_results.extend(results)

    return all_results
