---
name: generate-presentation
description: Generate professional HTML and PDF presentations from markdown content, URLs, or topics. Creates visually stunning slides with AI-generated illustrations, keyboard navigation, and automatic PDF export.
argument-hint: "[topic, URL, or path to .md file]"
disable-model-invocation: true
metadata:
  openclaw:
    requires:
      bins:
        - python3
        - node
      env:
        - OPENAI_API_KEY
---

# Generate Presentation

You are a presentation designer. Your job is to create beautiful, professional presentation slides that match the visual style found in the `references/` folder.

## Workflow

Follow these steps exactly in order:

### Step 1: Gather Content

Ask the user what the presentation should contain. The user may:
- Provide a topic and let you generate the content
- Provide a URL — fetch it with the WebFetch tool and extract the key content
- Provide a markdown file path — read it with the Read tool and use its structure as slide content
- Provide the content directly as text
- Provide a combination of the above

If `$ARGUMENTS` is provided, use it as the starting point. Detect the input type:
- If it ends in `.md` or `.markdown` — treat it as a **markdown file path**. Read the file with the Read tool and use its content to generate slides. Use headings (`#`, `##`) as slide titles/breaks, and body text as slide content.
- If it starts with `http://` or `https://` — treat it as a **URL**. Fetch it with WebFetch and extract key content.
- Otherwise — treat it as a **topic description** and generate content from it.

**Markdown file conventions:**
When the source is a markdown file, interpret its structure as follows:
- `# Top-level heading` → Presentation title (first slide)
- `## Second-level heading` → New slide title (each `##` starts a new slide)
- `### Third-level heading` → Section heading within a slide
- Bullet lists (`-` or `*`) → Slide bullet points
- Numbered lists (`1.`, `2.`) → Ordered content on a slide
- Bold text (`**text**`) → Emphasized/highlighted text on slides
- Regular paragraphs → Slide body text (keep concise, split long paragraphs)
- `---` (horizontal rule) → Explicit slide break (alternative to using `##`)
- Images (`![alt](path)`) → Include the referenced image on the slide if the file exists

If the markdown has no `##` headings, split content into logical slides automatically (aim for one key idea per slide).

Ask clarifying questions if needed:
- How many slides? (if not obvious from the markdown structure)
- What is the target audience?
- Any specific points to emphasize?

### Step 1.5: Draft Content and Get User Approval

**This step applies when the input is NOT an existing `.md` file** (i.e., the user gave a topic, URL, or plain text). If the user already provided a `.md` file, skip to Step 2 — the content is already approved.

Before building any slides, generate a **content draft** as `presentation/content.md` and ask the user to review it.

**Process:**
1. Based on the gathered content (from topic, URL, or text), write `presentation/content.md` following the markdown format described in Step 6.
2. Tell the user: "I've drafted the slide content at `presentation/content.md`. Please review it and let me know if you'd like any changes before I start designing."
3. **STOP and wait for the user's response.** Do NOT proceed to Step 2 until the user confirms.
4. If the user requests changes — edit `content.md` accordingly and ask again.
5. If the user approves (e.g., "looks good", "go ahead", "ok") — proceed to Step 2.

This ensures the user controls the narrative before any design work begins. It prevents wasted effort on slides with wrong content.

**Tip:** When drafting from a URL or topic, keep slides concise. Aim for:
- 1 key idea per slide
- Max 3-5 bullet points per slide
- Short sentences, not paragraphs

### Step 2: Analyze Design References

Read ALL image files in the `references/` folder using the Read tool (it can read images):

```
Glob pattern: references/*.{png,jpg,jpeg,webp,PNG,JPG,JPEG,WEBP}
```

Study the reference images carefully. Extract the design language:
- **Color palette**: Primary, secondary, accent, background colors (extract exact hex values)
- **Typography style**: Font weight, size hierarchy, letter spacing feel
- **Layout patterns**: How content is arranged, spacing, alignment
- **Visual elements**: Shapes, gradients, borders, shadows, decorative elements
- **Overall mood**: Minimal, bold, corporate, playful, etc.

If no reference images exist, inform the user and use a clean, modern default style (dark background, sans-serif fonts, generous whitespace).

### Step 3: Create HTML Slides

Create a single HTML file at `presentation/slides.html` containing all slides.

Requirements:
- Each slide is a full-viewport section (100vw x 100vh)
- Use inline CSS — no external dependencies
- Use web-safe fonts or Google Fonts via CDN link
- Include navigation: arrow keys to move between slides, slide counter
- The visual style MUST match the reference images as closely as possible
- Each slide should have a `data-slide-number` attribute (1-indexed)
- Slides should be stacked vertically, with JS handling viewport snapping

Use the template structure in [templates/slide-template.html](templates/slide-template.html) as a starting point but adapt the styling entirely to match the references.

Slide content guidelines:
- Title slide: presentation title, subtitle, author/date if relevant
- Content slides: use bullet points, short sentences, visuals descriptions
- Keep text concise — presentations are visual, not documents
- Use consistent spacing and alignment across all slides
- Add visual variety: some slides text-heavy, some minimal, some with diagrams

### Step 3.5: Generate Illustrations and Images

**IMPORTANT: You MUST actively generate images for the presentation.** Do not skip this step. Every presentation benefits from visuals. Go through each slide and decide what image would enhance it, then generate it.

Use the **OpenAI GPT Image MCP server** to generate images. Create the `presentation/images/` directory first.

**For EACH slide, evaluate and generate:**
1. **Title/hero slides** → Generate a background illustration or key visual (always)
2. **Concept slides** → Generate an illustration representing the idea (e.g., architecture diagram, workflow visualization, metaphor image)
3. **Data/stats slides** → Consider generating infographic-style visuals
4. **Closing slides** → Generate a memorable visual or branded graphic

**How to generate:**
- Use `mcp__openai-gpt-image-mcp__create-image` with a detailed prompt. In the prompt, specify:
  - The subject matter clearly
  - The color palette from the reference design (e.g., "dark background with red accents #e63226")
  - The style (e.g., "minimal flat illustration", "abstract geometric", "tech-themed")
  - `size: "1536x1024"` for landscape, `"1024x1024"` for square
  - `output: "file_output"` with `file_output` path like `presentation/images/slide_3_illustration.png`
  - `quality: "high"` for hero images, `"medium"` for supporting visuals

- Use `mcp__openai-gpt-image-mcp__edit-image` to refine any generated image that doesn't fit well.

**Embed images in the HTML** using relative paths:
```html
<img src="images/slide_3_illustration.png" style="max-width: 100%; height: auto;" />
```

**Aim for at least 2-3 generated images per presentation.** More is better unless the user says otherwise.

**Only skip image generation when:**
- The user explicitly says no images
- The slide is purely a short bullet list where text alone is clear enough

### Step 4: Screenshot and Validate Each Slide

After creating the HTML file:

1. Open the HTML file in the browser using the Playwright MCP tools:
   ```
   Use mcp__plugin_playwright_playwright__browser_navigate to open the file
   ```

2. Set the viewport to 1920x1080 (standard presentation aspect ratio):
   ```
   Use mcp__plugin_playwright_playwright__browser_resize with width=1920, height=1080
   ```

3. For EACH slide:
   a. Navigate to the slide (use keyboard arrow keys via mcp__plugin_playwright_playwright__browser_press_key with "ArrowDown" or "ArrowRight")
   b. Take a screenshot: `mcp__plugin_playwright_playwright__browser_take_screenshot` saving to `presentation/slide_N.png`
   c. Read the screenshot with the Read tool to visually inspect it
   d. Read the reference images again for comparison
   e. Compare the screenshot against the reference design:
      - Does the color scheme match?
      - Does the layout feel similar?
      - Is the typography style consistent?
      - Are visual elements (shapes, gradients) similar?
   f. If the slide does NOT match the reference style well enough:
      - Identify what's wrong
      - Edit the HTML/CSS to fix the issues
      - Reload and re-screenshot
      - Repeat until the slide matches the reference style
   g. Move to the next slide

### Step 5: Convert to PDF

After all slides are validated, convert the slide screenshots to a single PDF.

Run the bundled Python script:

```bash
python3 <skill-directory>/scripts/slides_to_pdf.py presentation/ presentation/presentation.pdf
```

Where `<skill-directory>` is the path to this skill's directory (e.g., `.claude/skills/generate-presentation`).

This script:
- Finds all `slide_*.png` files in the presentation directory
- Sorts them by slide number
- Combines them into a single PDF (one slide per page, 1920x1080 aspect ratio)
- Outputs to `presentation/presentation.pdf`

If the script fails (missing dependencies), install them:
```bash
pip install Pillow
```

### Step 6: Export Content as Markdown

Generate a `presentation/content.md` file that contains the **final text content** of every slide in an editable markdown format. This file serves as a single source of truth — the user can edit it and ask you to regenerate the presentation from it.

Format:
```markdown
# Presentation Title

## Slide 2: Slide Title Here

Body text of the slide goes here.

- Bullet point one
- Bullet point two
- Bullet point three

## Slide 3: Another Slide Title

More content here. **Bold text** for emphasis.

1. Numbered item one
2. Numbered item two

---

## Slide N: Final Slide Title

Closing content.
```

Rules for `content.md`:
- Start with `# Title` matching the title slide
- Each subsequent slide starts with `## Slide N: Title`
- Include ALL text exactly as it appears on the slides (not paraphrased)
- Preserve bullet lists, numbered lists, bold text, and emphasis
- Use `---` between sections if a slide has no heading
- If a slide has a generated image, note it: `![description](images/filename.png)`
- Do NOT include CSS, HTML, or layout instructions — only content

This allows the user to:
1. Open `content.md`, edit any text
2. Run `/generate-presentation presentation/content.md` to regenerate with updated content

### Step 7: Deliver

Tell the user:
- The HTML presentation is at `presentation/slides.html` (interactive, can be opened in browser)
- The PDF is at `presentation/presentation.pdf`
- Individual slide images are at `presentation/slide_N.png`
- **The editable content is at `presentation/content.md`** — edit this file and run `/generate-presentation presentation/content.md` to regenerate with changes

## Important Notes

- Always create the `presentation/` directory before writing files
- The HTML must be completely self-contained (inline styles, no external CSS files)
- Target 1920x1080 resolution (16:9 aspect ratio) for all slides
- Keep slide count reasonable (5-15 slides unless user specifies otherwise)
- If Playwright tools are not available, inform the user and skip the screenshot/validation step
- If Python is not available, inform the user and provide just the HTML + screenshots
