#!/usr/bin/env npx tsx
/**
 * Check position status in Moonwell Flagship USDC vault
 * Usage: npx tsx status.ts
 */

import {
  loadConfig,
  getClients,
  VAULT_ADDRESS,
  USDC_ADDRESS,
  VAULT_ABI,
  ERC20_ABI,
  formatUSDC,
  fetchVaultAPY,
  handleError,
} from './config.js';

async function main() {
  const config = loadConfig();
  const { publicClient, account } = getClients(config);

  console.log('🦎 Gekko Yield — Moonwell USDC Vault Status\n');
  console.log(`Wallet: ${account.address}`);
  console.log(`Vault:  ${VAULT_ADDRESS}`);
  console.log(`Chain:  Base (8453)\n`);

  // Fetch all data in parallel
  const [
    usdcBalance,
    vaultShares,
    totalAssets,
    totalSupply,
    vaultName,
    ethBalance,
    vaultAPY,
  ] = await Promise.all([
    publicClient.readContract({
      address: USDC_ADDRESS,
      abi: ERC20_ABI,
      functionName: 'balanceOf',
      args: [account.address],
    }),
    publicClient.readContract({
      address: VAULT_ADDRESS,
      abi: VAULT_ABI,
      functionName: 'balanceOf',
      args: [account.address],
    }),
    publicClient.readContract({
      address: VAULT_ADDRESS,
      abi: VAULT_ABI,
      functionName: 'totalAssets',
    }),
    publicClient.readContract({
      address: VAULT_ADDRESS,
      abi: VAULT_ABI,
      functionName: 'totalSupply',
    }),
    publicClient.readContract({
      address: VAULT_ADDRESS,
      abi: VAULT_ABI,
      functionName: 'name',
    }),
    publicClient.getBalance({ address: account.address }),
    fetchVaultAPY(),
  ]);

  // Calculate position value if user has shares
  let positionValue = 0n;
  if (vaultShares > 0n) {
    positionValue = await publicClient.readContract({
      address: VAULT_ADDRESS,
      abi: VAULT_ABI,
      functionName: 'convertToAssets',
      args: [vaultShares],
    });
  }

  // Calculate share price
  const sharePrice = totalSupply > 0n
    ? (totalAssets * BigInt(1e18)) / totalSupply
    : BigInt(1e18);

  const sharePriceFloat = Number(sharePrice) / 1e18;

  console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
  console.log('💰 Wallet Balances');
  console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
  console.log(`USDC (available):  ${formatUSDC(usdcBalance)} USDC`);
  console.log(`ETH (for gas):     ${(Number(ethBalance) / 1e18).toFixed(6)} ETH`);

  console.log('\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
  console.log('🏦 Vault Position');
  console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
  console.log(`Vault:             ${vaultName}`);
  console.log(`Your shares:       ${formatUSDC(vaultShares)} mwUSDC`);
  console.log(`Position value:    ${formatUSDC(positionValue)} USDC`);
  console.log(`Share price:       ${sharePriceFloat.toFixed(6)} USDC/share`);

  console.log('\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
  console.log('📈 Vault Stats');
  console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
  console.log(`Total TVL:         ${formatUSDC(totalAssets)} USDC`);
  console.log(`Total shares:      ${formatUSDC(totalSupply)} mwUSDC`);

  if (vaultAPY > 0) {
    console.log(`Current APY:       ${vaultAPY.toFixed(2)}%`);
    
    if (positionValue > 0n) {
      const monthlyEarnings = Number(positionValue) / 1e6 * vaultAPY / 100 / 12;
      const yearlyEarnings = Number(positionValue) / 1e6 * vaultAPY / 100;
      console.log(`Est. Monthly:      $${monthlyEarnings.toFixed(2)}`);
      console.log(`Est. Yearly:       $${yearlyEarnings.toFixed(2)}`);
    }
  } else {
    console.log(`Current APY:       (see app.morpho.org)`);
  }

  console.log('\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');

  // Warnings and tips
  if (Number(ethBalance) < 1e14) {
    console.log('\n⚠️  Warning: Low ETH balance for gas. Add ETH to continue transacting.');
  }

  if (vaultShares === 0n && usdcBalance > 0n) {
    console.log('\n💡 You have USDC available. Run deposit.ts to earn yield!');
  } else if (vaultShares > 0n) {
    console.log('\n✨ Your USDC is earning yield in the Moonwell vault!');
  }
  
  // Explicitly exit to close any open connections
  process.exit(0);
}

main().catch((err) => handleError(err, 'Status check failed'));
