#!/usr/bin/env python3
"""Parse any GEDCOM file and generate an interactive Family Tree Explorer dashboard.

Usage:
  python3 build_explorer.py <input.ged> [output.html] [--title "My Family Tree"] [--subtitle "Explore your ancestry"]

If output is omitted, writes to family-explorer.html in the current directory.
Presidents are auto-detected from OCCU fields (works with any GEDCOM, not just presidential data).
"""

import re
import json
import os
import sys
from collections import defaultdict, Counter
from datetime import datetime

SCRIPT_DIR = os.path.dirname(os.path.abspath(__file__))
GEDCOM_PATH = None  # Set via CLI
OUTPUT_PATH = None  # Set via CLI
TITLE = "Family Explorer"
SUBTITLE = "Explore your family history"

MONTH_MAP = {
    'JAN': 1, 'FEB': 2, 'MAR': 3, 'APR': 4, 'MAY': 5, 'JUN': 6,
    'JUL': 7, 'AUG': 8, 'SEP': 9, 'OCT': 10, 'NOV': 11, 'DEC': 12
}

def parse_date(date_str):
    """Parse GEDCOM date string into components."""
    if not date_str:
        return None
    # Remove prefixes like ABT, BEF, AFT, EST, CAL
    clean = re.sub(r'^(ABT|BEF|AFT|EST|CAL|FROM|TO|BET|AND)\s+', '', date_str.strip(), flags=re.IGNORECASE)
    
    # Try full date: 19 AUG 1946
    m = re.match(r'(\d{1,2})\s+([A-Z]{3})\s+(\d{4})', clean, re.IGNORECASE)
    if m:
        day, mon, year = int(m.group(1)), MONTH_MAP.get(m.group(2).upper(), 0), int(m.group(3))
        return {'year': year, 'month': mon, 'day': day, 'raw': date_str}
    
    # Month Year: AUG 1946
    m = re.match(r'([A-Z]{3})\s+(\d{4})', clean, re.IGNORECASE)
    if m:
        mon, year = MONTH_MAP.get(m.group(1).upper(), 0), int(m.group(2))
        return {'year': year, 'month': mon, 'day': 0, 'raw': date_str}
    
    # Year only: 1946
    m = re.match(r'(\d{4})', clean)
    if m:
        return {'year': int(m.group(1)), 'month': 0, 'day': 0, 'raw': date_str}
    
    return {'year': 0, 'month': 0, 'day': 0, 'raw': date_str}


def parse_president_info(occu_value, occu_date=None, occu_place=None):
    """Parse OCCU field to extract president number and party."""
    if not occu_value:
        return None
    m = re.search(r'President\s+No\.\s*(\d+)', occu_value, re.IGNORECASE)
    if not m:
        m = re.search(r'(\d+)\w*\s+President', occu_value, re.IGNORECASE)
    if not m:
        if 'president' in occu_value.lower():
            m = re.search(r'(\d+)', occu_value)
    if not m:
        return None
    
    number = int(m.group(1))
    
    # Extract party
    party = 'Unknown'
    val_lower = occu_value.lower()
    if 'democrat' in val_lower:
        party = 'Democratic'
    elif 'republican' in val_lower:
        party = 'Republican'
    elif 'whig' in val_lower:
        party = 'Whig'
    elif 'federalist' in val_lower:
        party = 'Federalist'
    elif 'democrat-republican' in val_lower or 'democratic-republican' in val_lower:
        party = 'Democratic-Republican'
    elif 'national union' in val_lower:
        party = 'National Union'
    elif 'no party' in val_lower or 'independent' in val_lower:
        party = 'Independent'
    
    return {'number': number, 'party': party, 'raw': occu_value}


def parse_gedcom(filepath):
    """Parse GEDCOM file into structured data."""
    with open(filepath, 'r', encoding='utf-8') as f:
        lines = f.readlines()
    
    individuals = {}
    families = {}
    current_record = None
    current_id = None
    current_event = None
    
    for line in lines:
        line = line.rstrip('\r\n')
        if not line:
            continue
        
        match = re.match(r'^(\d+)\s+(.+)$', line)
        if not match:
            continue
        
        level = int(match.group(1))
        rest = match.group(2)
        
        if level == 0:
            current_event = None
            id_match = re.match(r'@(\S+)@\s+(\S+)', rest)
            if id_match:
                current_id = id_match.group(1)
                record_type = id_match.group(2)
                if record_type == 'INDI':
                    current_record = 'INDI'
                    individuals[current_id] = {
                        'id': current_id, 'name': '', 'givenName': '', 'surname': '',
                        'sex': '', 'birth': {}, 'death': {}, 'burial': {},
                        'occupation': '', 'occuDate': '', 'occuPlace': '',
                        'president': None,
                        'familySpouse': [], 'familyChild': None,
                        'events': [], 'notes': []
                    }
                elif record_type == 'FAM':
                    current_record = 'FAM'
                    families[current_id] = {
                        'id': current_id, 'husband': None, 'wife': None,
                        'children': [], 'marriage': {}, 'divorce': {}
                    }
                else:
                    current_record = None
            else:
                current_record = None
            continue
        
        if current_record == 'INDI' and current_id in individuals:
            indi = individuals[current_id]
            
            if level == 1:
                current_event = None
                tag_match = re.match(r'(\S+)\s*(.*)', rest)
                if tag_match:
                    tag, value = tag_match.group(1), tag_match.group(2).strip()
                    
                    if tag == 'NAME':
                        # Extract surname from /slashes/
                        name_clean = value.replace('/', '').strip()
                        indi['name'] = name_clean
                        sm = re.search(r'/([^/]+)/', value)
                        if sm:
                            indi['surname'] = sm.group(1).strip()
                            indi['givenName'] = value[:value.index('/')].strip()
                    elif tag == 'SEX':
                        indi['sex'] = value
                    elif tag == 'BIRT':
                        current_event = 'birth'
                    elif tag == 'DEAT':
                        current_event = 'death'
                    elif tag == 'BURI':
                        current_event = 'burial'
                    elif tag == 'OCCU':
                        current_event = 'occu'
                        indi['occupation'] = value
                    elif tag == 'FAMC':
                        indi['familyChild'] = value.replace('@', '')
                    elif tag == 'FAMS':
                        indi['familySpouse'].append(value.replace('@', ''))
            
            elif level == 2:
                tag_match = re.match(r'(\S+)\s*(.*)', rest)
                if tag_match:
                    tag, value = tag_match.group(1), tag_match.group(2).strip()
                    
                    if tag == 'GIVN' and current_event is None:
                        indi['givenName'] = value
                    elif tag == 'SURN' and current_event is None:
                        indi['surname'] = value
                    
                    target = None
                    if current_event == 'birth':
                        target = indi['birth']
                    elif current_event == 'death':
                        target = indi['death']
                    elif current_event == 'burial':
                        target = indi['burial']
                    elif current_event == 'occu':
                        if tag == 'DATE':
                            indi['occuDate'] = value
                        elif tag == 'PLAC':
                            indi['occuPlace'] = value
                    
                    if target is not None:
                        if tag == 'DATE':
                            target['date'] = value
                        elif tag == 'PLAC':
                            target['place'] = value
        
        elif current_record == 'FAM' and current_id in families:
            fam = families[current_id]
            
            if level == 1:
                current_event = None
                tag_match = re.match(r'(\S+)\s*(.*)', rest)
                if tag_match:
                    tag, value = tag_match.group(1), tag_match.group(2).strip()
                    
                    if tag == 'HUSB':
                        fam['husband'] = value.replace('@', '')
                    elif tag == 'WIFE':
                        fam['wife'] = value.replace('@', '')
                    elif tag == 'CHIL':
                        fam['children'].append(value.replace('@', ''))
                    elif tag == 'MARR':
                        current_event = 'marriage'
                    elif tag == 'DIV':
                        current_event = 'divorce'
            
            elif level == 2:
                tag_match = re.match(r'(\S+)\s*(.*)', rest)
                if tag_match:
                    tag, value = tag_match.group(1), tag_match.group(2).strip()
                    
                    if current_event == 'marriage':
                        if tag == 'DATE':
                            fam['marriage']['date'] = value
                        elif tag == 'PLAC':
                            fam['marriage']['place'] = value
                    elif current_event == 'divorce':
                        if tag == 'DATE':
                            fam['divorce']['date'] = value
    
    # Post-process: detect presidents
    for indi in individuals.values():
        if indi['occupation']:
            pres = parse_president_info(indi['occupation'], indi.get('occuDate'), indi.get('occuPlace'))
            if pres:
                indi['president'] = pres
    
    return individuals, families


def build_people_json(individuals, families):
    """Build JSON-ready people list."""
    people = []
    for indi in individuals.values():
        birth_parsed = parse_date(indi['birth'].get('date', ''))
        death_parsed = parse_date(indi['death'].get('date', ''))
        
        birth_year = birth_parsed['year'] if birth_parsed else 0
        death_year = death_parsed['year'] if death_parsed else 0
        birth_month = birth_parsed['month'] if birth_parsed else 0
        birth_day = birth_parsed['day'] if birth_parsed else 0
        death_month = death_parsed['month'] if death_parsed else 0
        death_day = death_parsed['day'] if death_parsed else 0
        
        # Get spouse info
        spouses = []
        for fam_id in indi['familySpouse']:
            fam = families.get(fam_id, {})
            spouse_id = fam.get('wife') if indi['sex'] == 'M' else fam.get('husband')
            if spouse_id and spouse_id in individuals:
                spouses.append({
                    'id': spouse_id,
                    'name': individuals[spouse_id]['name'],
                    'marriage': fam.get('marriage', {})
                })
        
        # Get parents
        parents = {}
        if indi['familyChild'] and indi['familyChild'] in families:
            fam = families[indi['familyChild']]
            if fam['husband'] and fam['husband'] in individuals:
                parents['father'] = {'id': fam['husband'], 'name': individuals[fam['husband']]['name']}
            if fam['wife'] and fam['wife'] in individuals:
                parents['mother'] = {'id': fam['wife'], 'name': individuals[fam['wife']]['name']}
        
        # Get children
        children = []
        for fam_id in indi['familySpouse']:
            fam = families.get(fam_id, {})
            for child_id in fam.get('children', []):
                if child_id in individuals:
                    children.append({'id': child_id, 'name': individuals[child_id]['name']})
        
        person = {
            'id': indi['id'],
            'name': indi['name'],
            'givenName': indi['givenName'],
            'surname': indi['surname'],
            'sex': indi['sex'],
            'birthYear': birth_year,
            'birthMonth': birth_month,
            'birthDay': birth_day,
            'birthDate': indi['birth'].get('date', ''),
            'birthPlace': indi['birth'].get('place', ''),
            'deathYear': death_year,
            'deathMonth': death_month,
            'deathDay': death_day,
            'deathDate': indi['death'].get('date', ''),
            'deathPlace': indi['death'].get('place', ''),
            'burialDate': indi['burial'].get('date', ''),
            'burialPlace': indi['burial'].get('place', ''),
            'occupation': indi['occupation'],
            'president': indi['president'],
            'spouses': spouses,
            'parents': parents,
            'children': children,
            'familySpouse': indi['familySpouse'],
            'familyChild': indi['familyChild'],
        }
        people.append(person)
    
    return people


def build_families_json(families, individuals):
    """Build JSON-ready families list."""
    fams = []
    for fam in families.values():
        marriage_parsed = parse_date(fam['marriage'].get('date', ''))
        f = {
            'id': fam['id'],
            'husband': fam['husband'],
            'wife': fam['wife'],
            'husbandName': individuals[fam['husband']]['name'] if fam['husband'] and fam['husband'] in individuals else '',
            'wifeName': individuals[fam['wife']]['name'] if fam['wife'] and fam['wife'] in individuals else '',
            'children': fam['children'],
            'marriageDate': fam['marriage'].get('date', ''),
            'marriagePlace': fam['marriage'].get('place', ''),
            'marriageYear': marriage_parsed['year'] if marriage_parsed else 0,
            'marriageMonth': marriage_parsed['month'] if marriage_parsed else 0,
            'marriageDay': marriage_parsed['day'] if marriage_parsed else 0,
        }
        fams.append(f)
    return fams


def compute_stats(people, fam_list):
    """Compute dashboard statistics."""
    total = len(people)
    males = sum(1 for p in people if p['sex'] == 'M')
    females = sum(1 for p in people if p['sex'] == 'F')
    presidents = sorted([p for p in people if p['president']], key=lambda x: x['president']['number'])
    num_presidents = len(presidents)
    num_families = len(fam_list)
    
    # Earliest/latest
    birth_years = [p['birthYear'] for p in people if p['birthYear'] > 0]
    earliest_year = min(birth_years) if birth_years else 0
    latest_year = max(birth_years) if birth_years else 0
    earliest_person = next((p for p in people if p['birthYear'] == earliest_year), None)
    
    # Unique places
    places = set()
    for p in people:
        if p['birthPlace']:
            places.add(p['birthPlace'])
        if p['deathPlace']:
            places.add(p['deathPlace'])
    
    # Surnames
    surnames = Counter()
    for p in people:
        if p['surname']:
            surnames[p['surname'].upper()] += 1
    top_surnames = surnames.most_common(15)
    
    # Geographic origins (extract state/country from birth place)
    geo = Counter()
    for p in people:
        bp = p['birthPlace']
        if bp:
            parts = [x.strip() for x in bp.split(',')]
            if len(parts) >= 2:
                geo[parts[-1] if len(parts[-1]) > 2 else ', '.join(parts[-2:])] += 1
    top_geo = geo.most_common(12)
    
    # People by century
    century_counts = Counter()
    for p in people:
        if p['birthYear'] > 0:
            c = (p['birthYear'] - 1) // 100 + 1
            century_counts[c] += 1
    centuries = sorted(century_counts.items())
    
    # Generations (rough estimate)
    generations = 0
    if earliest_year > 0 and latest_year > 0:
        generations = max(1, (latest_year - earliest_year) // 25)
    
    return {
        'total': total,
        'males': males,
        'females': females,
        'numPresidents': num_presidents,
        'numFamilies': num_families,
        'earliestYear': earliest_year,
        'latestYear': latest_year,
        'earliestPerson': earliest_person['name'] if earliest_person else '',
        'uniquePlaces': len(places),
        'topSurnames': top_surnames,
        'topGeo': top_geo,
        'centuries': centuries,
        'generations': generations,
        'presidents': [{'name': p['name'], 'number': p['president']['number'], 'party': p['president']['party'],
                        'birthYear': p['birthYear'], 'deathYear': p['deathYear'], 'id': p['id']} for p in presidents]
    }


def generate_html(people, fam_list, stats):
    """Generate the full HTML dashboard."""
    
    people_json = json.dumps(people, separators=(',', ':'))
    families_json = json.dumps(fam_list, separators=(',', ':'))
    stats_json = json.dumps(stats, separators=(',', ':'))
    
    html = f'''<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>{TITLE}</title>
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
<style>
:root {{
  --bg-primary: #0a0e17;
  --bg-secondary: #111827;
  --bg-card: rgba(17, 24, 39, 0.8);
  --bg-glass: rgba(255,255,255,0.04);
  --border: rgba(255,255,255,0.08);
  --border-hover: rgba(255,255,255,0.15);
  --text-primary: #f1f5f9;
  --text-secondary: #94a3b8;
  --text-muted: #64748b;
  --accent-blue: #3b82f6;
  --accent-gold: #f59e0b;
  --accent-green: #10b981;
  --accent-red: #ef4444;
  --accent-purple: #8b5cf6;
  --accent-pink: #ec4899;
  --accent-cyan: #06b6d4;
  --glow-gold: rgba(245, 158, 11, 0.3);
  --glow-blue: rgba(59, 130, 246, 0.2);
  --gradient-gold: linear-gradient(135deg, #f59e0b, #d97706);
  --gradient-blue: linear-gradient(135deg, #3b82f6, #2563eb);
  --gradient-green: linear-gradient(135deg, #10b981, #059669);
  --gradient-red: linear-gradient(135deg, #ef4444, #dc2626);
  --gradient-purple: linear-gradient(135deg, #8b5cf6, #7c3aed);
  --shadow-lg: 0 10px 40px rgba(0,0,0,0.4);
  --shadow-glow: 0 0 30px rgba(59,130,246,0.1);
  --radius: 16px;
  --radius-sm: 10px;
}}

* {{ margin: 0; padding: 0; box-sizing: border-box; }}

body {{
  font-family: 'Inter', -apple-system, sans-serif;
  background: var(--bg-primary);
  color: var(--text-primary);
  min-height: 100vh;
  overflow-x: hidden;
}}

/* Animated background */
body::before {{
  content: '';
  position: fixed;
  top: 0; left: 0; right: 0; bottom: 0;
  background: 
    radial-gradient(ellipse at 20% 50%, rgba(59,130,246,0.08) 0%, transparent 50%),
    radial-gradient(ellipse at 80% 20%, rgba(245,158,11,0.06) 0%, transparent 50%),
    radial-gradient(ellipse at 50% 80%, rgba(139,92,246,0.05) 0%, transparent 50%);
  z-index: 0;
  pointer-events: none;
}}

/* ===== HEADER ===== */
.header {{
  position: sticky;
  top: 0;
  z-index: 100;
  background: rgba(10, 14, 23, 0.85);
  backdrop-filter: blur(20px);
  border-bottom: 1px solid var(--border);
  padding: 0 32px;
  height: 64px;
  display: flex;
  align-items: center;
  justify-content: space-between;
}}

.header-left {{
  display: flex;
  align-items: center;
  gap: 16px;
}}

.header-logo {{
  font-size: 28px;
}}

.header-title {{
  font-size: 18px;
  font-weight: 700;
  background: linear-gradient(135deg, var(--accent-gold), #fbbf24);
  -webkit-background-clip: text;
  -webkit-text-fill-color: transparent;
}}

.header-subtitle {{
  font-size: 12px;
  color: var(--text-muted);
  margin-top: 2px;
}}

.search-bar {{
  position: relative;
  width: 300px;
}}

.search-bar input {{
  width: 100%;
  padding: 8px 16px 8px 36px;
  background: var(--bg-glass);
  border: 1px solid var(--border);
  border-radius: 24px;
  color: var(--text-primary);
  font-size: 13px;
  font-family: inherit;
  outline: none;
  transition: all 0.3s;
}}

.search-bar input:focus {{
  border-color: var(--accent-blue);
  box-shadow: 0 0 0 3px rgba(59,130,246,0.15);
}}

.search-bar::before {{
  content: '🔍';
  position: absolute;
  left: 12px;
  top: 50%;
  transform: translateY(-50%);
  font-size: 14px;
}}

/* ===== NAV TABS ===== */
nav {{
  display: flex;
  gap: 4px;
}}

nav button {{
  padding: 8px 20px;
  background: transparent;
  border: 1px solid transparent;
  border-radius: 24px;
  color: var(--text-secondary);
  font-size: 13px;
  font-weight: 500;
  font-family: inherit;
  cursor: pointer;
  transition: all 0.3s;
  white-space: nowrap;
}}

nav button:hover {{
  color: var(--text-primary);
  background: var(--bg-glass);
}}

nav button.active {{
  background: var(--gradient-blue);
  color: white;
  border-color: transparent;
  box-shadow: 0 2px 12px rgba(59,130,246,0.3);
}}

/* ===== MAIN CONTENT ===== */
.content {{
  position: relative;
  z-index: 1;
  max-width: 1400px;
  margin: 0 auto;
  padding: 32px;
}}

.tab-content {{
  display: none;
  animation: fadeIn 0.4s ease;
}}

.tab-content.active {{
  display: block;
}}

@keyframes fadeIn {{
  from {{ opacity: 0; transform: translateY(10px); }}
  to {{ opacity: 1; transform: translateY(0); }}
}}

/* ===== CARDS ===== */
.card {{
  background: var(--bg-card);
  border: 1px solid var(--border);
  border-radius: var(--radius);
  backdrop-filter: blur(10px);
  transition: all 0.3s;
}}

.card:hover {{
  border-color: var(--border-hover);
  box-shadow: var(--shadow-glow);
}}

.card-header {{
  padding: 20px 24px 12px;
  font-size: 16px;
  font-weight: 600;
  color: var(--text-primary);
  display: flex;
  align-items: center;
  gap: 8px;
}}

.card-body {{
  padding: 0 24px 24px;
}}

/* ===== HERO ===== */
.hero {{
  text-align: center;
  padding: 48px 0 32px;
}}

.hero h2 {{
  font-size: 42px;
  font-weight: 800;
  background: linear-gradient(135deg, #f1f5f9, #cbd5e1);
  -webkit-background-clip: text;
  -webkit-text-fill-color: transparent;
  margin-bottom: 8px;
  letter-spacing: -1px;
}}

.hero p {{
  color: var(--text-muted);
  font-size: 16px;
}}

/* ===== STAT CARDS ===== */
.stats-grid {{
  display: grid;
  grid-template-columns: repeat(auto-fit, minmax(180px, 1fr));
  gap: 16px;
  margin: 32px 0;
}}

.stat-card {{
  background: var(--bg-card);
  border: 1px solid var(--border);
  border-radius: var(--radius);
  padding: 24px;
  text-align: center;
  transition: all 0.3s;
  position: relative;
  overflow: hidden;
}}

.stat-card::before {{
  content: '';
  position: absolute;
  top: 0;
  left: 0;
  right: 0;
  height: 3px;
}}

.stat-card:nth-child(1)::before {{ background: var(--gradient-blue); }}
.stat-card:nth-child(2)::before {{ background: var(--gradient-gold); }}
.stat-card:nth-child(3)::before {{ background: var(--gradient-green); }}
.stat-card:nth-child(4)::before {{ background: var(--gradient-purple); }}
.stat-card:nth-child(5)::before {{ background: var(--gradient-red); }}
.stat-card:nth-child(6)::before {{ background: linear-gradient(135deg, var(--accent-cyan), #0891b2); }}

.stat-card:hover {{
  transform: translateY(-4px);
  box-shadow: var(--shadow-lg);
  border-color: var(--border-hover);
}}

.stat-icon {{
  font-size: 28px;
  margin-bottom: 8px;
}}

.stat-value {{
  font-size: 36px;
  font-weight: 800;
  margin-bottom: 4px;
}}

.stat-card:nth-child(1) .stat-value {{ color: var(--accent-blue); }}
.stat-card:nth-child(2) .stat-value {{ color: var(--accent-gold); }}
.stat-card:nth-child(3) .stat-value {{ color: var(--accent-green); }}
.stat-card:nth-child(4) .stat-value {{ color: var(--accent-purple); }}
.stat-card:nth-child(5) .stat-value {{ color: var(--accent-red); }}
.stat-card:nth-child(6) .stat-value {{ color: var(--accent-cyan); }}

.stat-label {{
  font-size: 11px;
  font-weight: 600;
  text-transform: uppercase;
  letter-spacing: 1px;
  color: var(--text-muted);
  margin-bottom: 4px;
}}

.stat-detail {{
  font-size: 12px;
  color: var(--text-muted);
}}

/* ===== ON THIS DAY ===== */
.otd-section {{
  margin: 32px 0;
}}

.otd-header {{
  display: flex;
  align-items: center;
  justify-content: space-between;
  margin-bottom: 16px;
}}

.otd-header h3 {{
  font-size: 22px;
  font-weight: 700;
  display: flex;
  align-items: center;
  gap: 10px;
}}

.otd-count {{
  background: var(--bg-glass);
  padding: 4px 12px;
  border-radius: 20px;
  font-size: 13px;
  color: var(--text-secondary);
}}

.otd-grid {{
  display: grid;
  grid-template-columns: repeat(auto-fill, minmax(320px, 1fr));
  gap: 12px;
}}

.otd-card {{
  display: flex;
  align-items: center;
  gap: 16px;
  padding: 16px 20px;
  background: var(--bg-card);
  border: 1px solid var(--border);
  border-radius: var(--radius-sm);
  cursor: pointer;
  transition: all 0.3s;
}}

.otd-card:hover {{
  border-color: var(--accent-gold);
  background: rgba(245, 158, 11, 0.05);
  transform: translateX(4px);
}}

.otd-year {{
  font-size: 28px;
  font-weight: 800;
  color: var(--accent-gold);
  min-width: 70px;
}}

.otd-info {{
  flex: 1;
}}

.otd-badge {{
  display: inline-block;
  padding: 2px 8px;
  border-radius: 4px;
  font-size: 10px;
  font-weight: 700;
  text-transform: uppercase;
  letter-spacing: 0.5px;
  margin-bottom: 4px;
}}

.otd-badge.birth {{ background: rgba(16,185,129,0.2); color: var(--accent-green); }}
.otd-badge.death {{ background: rgba(239,68,68,0.2); color: var(--accent-red); }}
.otd-badge.marriage {{ background: rgba(245,158,11,0.2); color: var(--accent-gold); }}
.otd-badge.burial {{ background: rgba(139,92,246,0.2); color: var(--accent-purple); }}

.otd-name {{
  font-weight: 600;
  font-size: 15px;
}}

.otd-name .pres-badge {{
  display: inline-block;
  background: var(--gradient-gold);
  color: #000;
  font-size: 9px;
  font-weight: 700;
  padding: 1px 6px;
  border-radius: 3px;
  margin-left: 6px;
  vertical-align: middle;
}}

.otd-place {{
  font-size: 12px;
  color: var(--text-muted);
  margin-top: 2px;
}}

.otd-years-ago {{
  font-size: 11px;
  color: var(--accent-cyan);
  font-weight: 500;
}}

/* ===== PRESIDENTS ROW ===== */
.pres-section {{
  margin: 32px 0;
}}

.pres-section h3 {{
  font-size: 22px;
  font-weight: 700;
  margin-bottom: 16px;
  display: flex;
  align-items: center;
  gap: 10px;
}}

.pres-scroll {{
  display: flex;
  gap: 12px;
  overflow-x: auto;
  padding: 8px 0 16px;
  scrollbar-width: thin;
  scrollbar-color: var(--border) transparent;
}}

.pres-scroll::-webkit-scrollbar {{ height: 6px; }}
.pres-scroll::-webkit-scrollbar-thumb {{ background: var(--border); border-radius: 3px; }}

.pres-card {{
  min-width: 140px;
  padding: 16px;
  background: var(--bg-card);
  border: 1px solid var(--border);
  border-radius: var(--radius-sm);
  text-align: center;
  cursor: pointer;
  transition: all 0.3s;
  flex-shrink: 0;
}}

.pres-card:hover {{
  border-color: var(--accent-gold);
  box-shadow: 0 0 20px var(--glow-gold);
  transform: translateY(-4px);
}}

.pres-number {{
  font-size: 24px;
  font-weight: 800;
  color: var(--accent-gold);
}}

.pres-name {{
  font-size: 13px;
  font-weight: 600;
  margin: 4px 0;
  line-height: 1.3;
}}

.pres-party {{
  font-size: 10px;
  color: var(--text-muted);
  padding: 2px 8px;
  border-radius: 10px;
  display: inline-block;
}}

.pres-party.Democratic {{ background: rgba(59,130,246,0.15); color: var(--accent-blue); }}
.pres-party.Republican {{ background: rgba(239,68,68,0.15); color: var(--accent-red); }}
.pres-party.Whig {{ background: rgba(245,158,11,0.15); color: var(--accent-gold); }}
.pres-party.Federalist {{ background: rgba(139,92,246,0.15); color: var(--accent-purple); }}
.pres-party.Democratic-Republican {{ background: rgba(16,185,129,0.15); color: var(--accent-green); }}
.pres-party.Independent {{ background: rgba(148,163,184,0.15); color: var(--text-secondary); }}

.pres-dates {{
  font-size: 11px;
  color: var(--text-muted);
  margin-top: 4px;
}}

/* ===== CHARTS GRID ===== */
.charts-grid {{
  display: grid;
  grid-template-columns: repeat(2, 1fr);
  gap: 20px;
  margin: 32px 0;
}}

@media (max-width: 900px) {{
  .charts-grid {{ grid-template-columns: 1fr; }}
}}

.bar-row {{
  display: flex;
  align-items: center;
  gap: 12px;
  margin-bottom: 8px;
}}

.bar-label {{
  font-size: 12px;
  color: var(--text-secondary);
  min-width: 120px;
  text-align: right;
  white-space: nowrap;
  overflow: hidden;
  text-overflow: ellipsis;
}}

.bar-track {{
  flex: 1;
  height: 24px;
  background: var(--bg-glass);
  border-radius: 4px;
  overflow: hidden;
  position: relative;
}}

.bar-fill {{
  height: 100%;
  border-radius: 4px;
  display: flex;
  align-items: center;
  padding-left: 8px;
  font-size: 11px;
  font-weight: 600;
  color: white;
  transition: width 1s ease;
}}

/* ===== PEOPLE TAB ===== */
.people-controls {{
  display: flex;
  gap: 12px;
  margin-bottom: 24px;
  flex-wrap: wrap;
  align-items: center;
}}

.people-controls input {{
  flex: 1;
  min-width: 200px;
  padding: 10px 16px;
  background: var(--bg-glass);
  border: 1px solid var(--border);
  border-radius: var(--radius-sm);
  color: var(--text-primary);
  font-size: 14px;
  font-family: inherit;
  outline: none;
}}

.people-controls input:focus {{
  border-color: var(--accent-blue);
}}

.filter-btn {{
  padding: 8px 16px;
  background: var(--bg-glass);
  border: 1px solid var(--border);
  border-radius: 20px;
  color: var(--text-secondary);
  font-size: 13px;
  font-family: inherit;
  cursor: pointer;
  transition: all 0.3s;
}}

.filter-btn:hover {{ border-color: var(--border-hover); color: var(--text-primary); }}
.filter-btn.active {{ background: var(--gradient-blue); color: white; border-color: transparent; }}
.filter-btn.pres-filter.active {{ background: var(--gradient-gold); }}

.people-count {{
  font-size: 13px;
  color: var(--text-muted);
  margin-bottom: 16px;
}}

.people-grid {{
  display: grid;
  grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
  gap: 12px;
}}

.person-card {{
  display: flex;
  align-items: center;
  gap: 14px;
  padding: 14px 18px;
  background: var(--bg-card);
  border: 1px solid var(--border);
  border-radius: var(--radius-sm);
  cursor: pointer;
  transition: all 0.3s;
}}

.person-card:hover {{
  border-color: var(--accent-blue);
  background: rgba(59,130,246,0.05);
}}

.person-card.is-president {{
  border-color: rgba(245,158,11,0.3);
}}

.person-card.is-president:hover {{
  border-color: var(--accent-gold);
  background: rgba(245,158,11,0.05);
  box-shadow: 0 0 20px var(--glow-gold);
}}

.person-avatar {{
  width: 44px;
  height: 44px;
  border-radius: 50%;
  display: flex;
  align-items: center;
  justify-content: center;
  font-weight: 700;
  font-size: 16px;
  flex-shrink: 0;
}}

.person-avatar.male {{ background: rgba(59,130,246,0.2); color: var(--accent-blue); }}
.person-avatar.female {{ background: rgba(236,72,153,0.2); color: var(--accent-pink); }}
.person-avatar.president {{ background: var(--gradient-gold); color: #000; font-size: 13px; }}

.person-info {{
  flex: 1;
  min-width: 0;
}}

.person-name {{
  font-weight: 600;
  font-size: 14px;
  white-space: nowrap;
  overflow: hidden;
  text-overflow: ellipsis;
}}

.person-dates {{
  font-size: 12px;
  color: var(--text-muted);
}}

.person-place {{
  font-size: 11px;
  color: var(--text-muted);
  white-space: nowrap;
  overflow: hidden;
  text-overflow: ellipsis;
}}

.pagination {{
  display: flex;
  justify-content: center;
  gap: 8px;
  margin-top: 24px;
}}

.pagination button {{
  padding: 8px 14px;
  background: var(--bg-glass);
  border: 1px solid var(--border);
  border-radius: 8px;
  color: var(--text-secondary);
  font-family: inherit;
  cursor: pointer;
  transition: all 0.2s;
}}

.pagination button:hover {{ border-color: var(--accent-blue); color: var(--text-primary); }}
.pagination button.active {{ background: var(--gradient-blue); color: white; border-color: transparent; }}
.pagination button:disabled {{ opacity: 0.3; cursor: default; }}

/* ===== TIMELINE TAB ===== */
.timeline-controls {{
  display: flex;
  gap: 12px;
  margin-bottom: 24px;
  flex-wrap: wrap;
}}

.timeline {{
  position: relative;
  padding-left: 60px;
}}

.timeline::before {{
  content: '';
  position: absolute;
  left: 28px;
  top: 0;
  bottom: 0;
  width: 2px;
  background: var(--border);
}}

.timeline-item {{
  position: relative;
  margin-bottom: 16px;
  padding: 14px 20px;
  background: var(--bg-card);
  border: 1px solid var(--border);
  border-radius: var(--radius-sm);
  transition: all 0.3s;
  cursor: pointer;
}}

.timeline-item:hover {{
  border-color: var(--border-hover);
  transform: translateX(4px);
}}

.timeline-dot {{
  position: absolute;
  left: -40px;
  top: 18px;
  width: 14px;
  height: 14px;
  border-radius: 50%;
  border: 2px solid var(--bg-primary);
}}

.timeline-dot.birth {{ background: var(--accent-green); }}
.timeline-dot.death {{ background: var(--accent-red); }}
.timeline-dot.marriage {{ background: var(--accent-gold); }}

.timeline-year {{
  font-size: 13px;
  font-weight: 600;
  color: var(--accent-gold);
}}

.timeline-name {{
  font-weight: 600;
  font-size: 15px;
  margin: 4px 0;
}}

.timeline-detail {{
  font-size: 12px;
  color: var(--text-muted);
}}

/* ===== DAILY ALERTS TAB ===== */
.alerts-hero {{
  text-align: center;
  padding: 40px 0;
}}

.alerts-hero h2 {{
  font-size: 36px;
  font-weight: 800;
  margin-bottom: 8px;
}}

.alerts-hero .date-display {{
  font-size: 20px;
  color: var(--accent-gold);
  font-weight: 600;
}}

.spotlight-card {{
  background: var(--bg-card);
  border: 2px solid rgba(245,158,11,0.3);
  border-radius: var(--radius);
  padding: 32px;
  margin: 24px 0;
  text-align: center;
  position: relative;
  overflow: hidden;
}}

.spotlight-card::before {{
  content: '';
  position: absolute;
  top: 0; left: 0; right: 0;
  height: 4px;
  background: var(--gradient-gold);
}}

.spotlight-label {{
  font-size: 12px;
  font-weight: 600;
  text-transform: uppercase;
  letter-spacing: 2px;
  color: var(--accent-gold);
  margin-bottom: 16px;
}}

.spotlight-name {{
  font-size: 32px;
  font-weight: 800;
  margin-bottom: 8px;
}}

.spotlight-dates {{
  font-size: 18px;
  color: var(--text-secondary);
  margin-bottom: 16px;
}}

.spotlight-details {{
  display: flex;
  justify-content: center;
  gap: 32px;
  flex-wrap: wrap;
}}

.spotlight-detail {{
  text-align: center;
}}

.spotlight-detail-label {{
  font-size: 11px;
  color: var(--text-muted);
  text-transform: uppercase;
  letter-spacing: 1px;
}}

.spotlight-detail-value {{
  font-size: 14px;
  color: var(--text-primary);
  margin-top: 4px;
}}

.shuffle-btn {{
  margin-top: 20px;
  padding: 10px 24px;
  background: var(--gradient-gold);
  border: none;
  border-radius: 24px;
  color: #000;
  font-size: 14px;
  font-weight: 600;
  font-family: inherit;
  cursor: pointer;
  transition: all 0.3s;
}}

.shuffle-btn:hover {{
  transform: scale(1.05);
  box-shadow: 0 4px 20px var(--glow-gold);
}}

/* ===== TREE TAB ===== */
.tree-controls {{
  display: flex;
  gap: 8px;
  margin-bottom: 16px;
  flex-wrap: wrap;
}}

.tree-controls button {{
  padding: 8px 16px;
  background: var(--bg-glass);
  border: 1px solid var(--border);
  border-radius: 8px;
  color: var(--text-secondary);
  font-family: inherit;
  font-size: 13px;
  cursor: pointer;
  transition: all 0.2s;
}}

.tree-controls button:hover {{
  border-color: var(--accent-blue);
  color: var(--text-primary);
}}

.tree-controls select {{
  padding: 8px 16px;
  background: var(--bg-secondary);
  border: 1px solid var(--border);
  border-radius: 8px;
  color: var(--text-primary);
  font-family: inherit;
  font-size: 13px;
  cursor: pointer;
}}

#tree-canvas {{
  width: 100%;
  height: 600px;
  background: var(--bg-card);
  border: 1px solid var(--border);
  border-radius: var(--radius);
  overflow: hidden;
  position: relative;
}}

.tree-node {{
  position: absolute;
  padding: 10px 16px;
  border-radius: 8px;
  border: 2px solid;
  cursor: pointer;
  transition: all 0.2s;
  white-space: nowrap;
  font-size: 13px;
  z-index: 2;
}}

.tree-node:hover {{
  transform: scale(1.05);
  z-index: 10;
}}

.tree-node.male {{
  background: rgba(59,130,246,0.15);
  border-color: var(--accent-blue);
  color: var(--text-primary);
}}

.tree-node.female {{
  background: rgba(236,72,153,0.15);
  border-color: var(--accent-pink);
  color: var(--text-primary);
}}

.tree-node.president {{
  background: rgba(245,158,11,0.2);
  border-color: var(--accent-gold);
  color: var(--text-primary);
  box-shadow: 0 0 15px var(--glow-gold);
}}

.tree-node-name {{
  font-weight: 600;
}}

.tree-node-dates {{
  font-size: 11px;
  color: var(--text-muted);
}}

/* ===== PERSON MODAL ===== */
.modal-overlay {{
  display: none;
  position: fixed;
  top: 0; left: 0; right: 0; bottom: 0;
  background: rgba(0,0,0,0.7);
  backdrop-filter: blur(8px);
  z-index: 200;
  align-items: center;
  justify-content: center;
}}

.modal-overlay.show {{
  display: flex;
}}

.modal {{
  background: var(--bg-secondary);
  border: 1px solid var(--border);
  border-radius: var(--radius);
  width: 90%;
  max-width: 600px;
  max-height: 80vh;
  overflow-y: auto;
  padding: 32px;
  position: relative;
  animation: modalIn 0.3s ease;
}}

@keyframes modalIn {{
  from {{ opacity: 0; transform: scale(0.9) translateY(20px); }}
  to {{ opacity: 1; transform: scale(1) translateY(0); }}
}}

.modal-close {{
  position: absolute;
  top: 16px;
  right: 16px;
  background: var(--bg-glass);
  border: 1px solid var(--border);
  border-radius: 50%;
  width: 32px;
  height: 32px;
  display: flex;
  align-items: center;
  justify-content: center;
  cursor: pointer;
  color: var(--text-secondary);
  font-size: 18px;
  transition: all 0.2s;
}}

.modal-close:hover {{
  background: rgba(239,68,68,0.2);
  color: var(--accent-red);
}}

.modal h2 {{
  font-size: 24px;
  font-weight: 700;
  margin-bottom: 4px;
}}

.modal .pres-tag {{
  display: inline-block;
  background: var(--gradient-gold);
  color: #000;
  padding: 4px 12px;
  border-radius: 6px;
  font-size: 12px;
  font-weight: 700;
  margin: 8px 0;
}}

.modal-section {{
  margin-top: 20px;
}}

.modal-section h4 {{
  font-size: 13px;
  font-weight: 600;
  color: var(--text-muted);
  text-transform: uppercase;
  letter-spacing: 1px;
  margin-bottom: 8px;
}}

.modal-row {{
  display: flex;
  gap: 8px;
  padding: 6px 0;
  font-size: 14px;
}}

.modal-row .label {{
  color: var(--text-muted);
  min-width: 80px;
}}

.modal-link {{
  color: var(--accent-blue);
  cursor: pointer;
  text-decoration: underline;
}}

.modal-link:hover {{
  color: var(--accent-cyan);
}}

/* ===== LOADING ===== */
.loading {{
  display: flex;
  align-items: center;
  justify-content: center;
  height: 200px;
  color: var(--text-muted);
}}

/* ===== SCROLLBAR ===== */
::-webkit-scrollbar {{ width: 8px; }}
::-webkit-scrollbar-track {{ background: transparent; }}
::-webkit-scrollbar-thumb {{ background: var(--border); border-radius: 4px; }}
::-webkit-scrollbar-thumb:hover {{ background: var(--border-hover); }}

/* ===== RESPONSIVE ===== */
@media (max-width: 768px) {{
  .header {{ padding: 0 16px; }}
  .search-bar {{ display: none; }}
  .content {{ padding: 16px; }}
  .hero h2 {{ font-size: 28px; }}
  .stats-grid {{ grid-template-columns: repeat(2, 1fr); }}
  .charts-grid {{ grid-template-columns: 1fr; }}
  nav button {{ padding: 6px 12px; font-size: 12px; }}
}}
</style>
</head>
<body>

<div class="header">
  <div class="header-left">
    <div class="header-logo">🏛️</div>
    <div>
      <div class="header-title">{TITLE}</div>
      <div class="header-subtitle">{SUBTITLE}</div>
    </div>
  </div>
  <div class="search-bar">
    <input type="text" id="globalSearch" placeholder="Search people, places, dates..." />
  </div>
  <nav>
    <button class="active" onclick="switchTab('dashboard')">📊 Dashboard</button>
    <button onclick="switchTab('tree')">🌲 Family Tree</button>
    <button onclick="switchTab('people')">👥 People</button>
    <button onclick="switchTab('timeline')">📅 Timeline</button>
    <button onclick="switchTab('alerts')">🔔 Daily Alerts</button>
  </nav>
</div>

<div class="content">
  <!-- DASHBOARD TAB -->
  <div id="tab-dashboard" class="tab-content active">
    <div class="hero">
      <h2>{TITLE}</h2>
      <p>{SUBTITLE}</p>
    </div>
    
    <div class="stats-grid" id="statsGrid"></div>
    
    <div class="otd-section">
      <div class="otd-header">
        <h3>✨ On This Day</h3>
        <div class="otd-count" id="otdCount"></div>
      </div>
      <div class="otd-grid" id="otdGrid"></div>
    </div>
    
    <div class="pres-section">
      <h3>🏛️ The Presidents</h3>
      <div class="pres-scroll" id="presScroll"></div>
    </div>
    
    <div class="charts-grid">
      <div class="card">
        <div class="card-header">👨‍👩‍👧‍👦 Top Surnames</div>
        <div class="card-body" id="surnameChart"></div>
      </div>
      <div class="card">
        <div class="card-header">🗺️ Geographic Origins</div>
        <div class="card-body" id="geoChart"></div>
      </div>
      <div class="card">
        <div class="card-header">⏳ People by Century</div>
        <div class="card-body" id="centuryChart"></div>
      </div>
      <div class="card">
        <div class="card-header">🎉 Party Breakdown</div>
        <div class="card-body" id="partyChart"></div>
      </div>
    </div>
  </div>
  
  <!-- TREE TAB -->
  <div id="tab-tree" class="tab-content">
    <div class="tree-controls">
      <select id="treeRoot" onchange="renderTree()"></select>
      <button onclick="treeZoom(1.2)">＋ Zoom In</button>
      <button onclick="treeZoom(0.8)">－ Zoom Out</button>
      <button onclick="treeReset()">↺ Reset</button>
    </div>
    <div id="tree-canvas"></div>
  </div>
  
  <!-- PEOPLE TAB -->
  <div id="tab-people" class="tab-content">
    <div class="people-controls">
      <input type="text" id="peopleSearch" placeholder="Search by name, place, or year..." oninput="filterPeople()" />
      <button class="filter-btn active" onclick="setGender('all', this)">All</button>
      <button class="filter-btn" onclick="setGender('M', this)">♂ Male</button>
      <button class="filter-btn" onclick="setGender('F', this)">♀ Female</button>
      <button class="filter-btn pres-filter" onclick="togglePresFilter(this)">🏛️ Presidents</button>
    </div>
    <div class="people-count" id="peopleCount"></div>
    <div class="people-grid" id="peopleGrid"></div>
    <div class="pagination" id="peoplePagination"></div>
  </div>
  
  <!-- TIMELINE TAB -->
  <div id="tab-timeline" class="tab-content">
    <div class="timeline-controls">
      <input type="text" id="timelineSearch" placeholder="Filter by name, year, or place..." oninput="filterTimeline()" style="flex:1;min-width:200px;padding:10px 16px;background:var(--bg-glass);border:1px solid var(--border);border-radius:10px;color:var(--text-primary);font-family:inherit;font-size:14px;outline:none;" />
      <button class="filter-btn active" onclick="setTimelineType('all', this)">All</button>
      <button class="filter-btn" onclick="setTimelineType('birth', this)">🟢 Births</button>
      <button class="filter-btn" onclick="setTimelineType('death', this)">🔴 Deaths</button>
      <button class="filter-btn" onclick="setTimelineType('marriage', this)">💛 Marriages</button>
    </div>
    <div class="timeline" id="timelineContainer"></div>
    <div class="pagination" id="timelinePagination"></div>
  </div>
  
  <!-- DAILY ALERTS TAB -->
  <div id="tab-alerts" class="tab-content">
    <div class="alerts-hero">
      <h2>🔔 This Day in Presidential History</h2>
      <div class="date-display" id="alertsDate"></div>
    </div>
    
    <div class="spotlight-card" id="spotlightCard"></div>
    
    <div class="otd-section">
      <h3 style="margin-bottom:16px;">📅 All Events Today</h3>
      <div class="otd-grid" id="alertsOtdGrid"></div>
    </div>
    
    <div class="card" style="margin-top:32px;">
      <div class="card-header">🎲 Fun Facts</div>
      <div class="card-body" id="funFacts"></div>
    </div>
  </div>
</div>

<!-- PERSON MODAL -->
<div class="modal-overlay" id="personModal" onclick="if(event.target===this)closeModal()">
  <div class="modal" id="modalContent"></div>
</div>

<script>
// ===== DATA =====
const PEOPLE = {people_json};
const FAMILIES = {families_json};
const STATS = {stats_json};

// Index people by ID
const PEOPLE_MAP = {{}};
PEOPLE.forEach(p => PEOPLE_MAP[p.id] = p);

// ===== STATE =====
let currentTab = 'dashboard';
let peopleGenderFilter = 'all';
let peoplePresFilter = false;
let peoplePage = 1;
const PEOPLE_PER_PAGE = 60;
let timelineTypeFilter = 'all';
let timelinePage = 1;
const TIMELINE_PER_PAGE = 100;
let treeScale = 1;
let treeOffsetX = 0, treeOffsetY = 0;

// ===== TAB SWITCHING =====
function switchTab(tab) {{
  currentTab = tab;
  document.querySelectorAll('.tab-content').forEach(el => el.classList.remove('active'));
  document.getElementById('tab-' + tab).classList.add('active');
  document.querySelectorAll('nav button').forEach(btn => {{
    btn.classList.toggle('active', btn.textContent.toLowerCase().includes(tab === 'alerts' ? 'daily' : tab));
  }});
  
  if (tab === 'tree') renderTree();
  if (tab === 'alerts') renderAlerts();
}}

// ===== DASHBOARD =====
function renderDashboard() {{
  // Stats grid
  const grid = document.getElementById('statsGrid');
  grid.innerHTML = `
    <div class="stat-card">
      <div class="stat-icon">👥</div>
      <div class="stat-value">${{STATS.total.toLocaleString()}}</div>
      <div class="stat-label">People</div>
      <div class="stat-detail">${{STATS.males}} males · ${{STATS.females}} females</div>
    </div>
    <div class="stat-card">
      <div class="stat-icon">🏛️</div>
      <div class="stat-value">${{STATS.numPresidents}}</div>
      <div class="stat-label">Presidents</div>
      <div class="stat-detail">Washington to Trump</div>
    </div>
    <div class="stat-card">
      <div class="stat-icon">💑</div>
      <div class="stat-value">${{STATS.numFamilies.toLocaleString()}}</div>
      <div class="stat-label">Families</div>
      <div class="stat-detail">Linked family units</div>
    </div>
    <div class="stat-card">
      <div class="stat-icon">📜</div>
      <div class="stat-value">${{STATS.earliestYear}}</div>
      <div class="stat-label">Earliest Record</div>
      <div class="stat-detail">${{STATS.earliestPerson}}</div>
    </div>
    <div class="stat-card">
      <div class="stat-icon">📍</div>
      <div class="stat-value">${{STATS.uniquePlaces.toLocaleString()}}</div>
      <div class="stat-label">Unique Places</div>
      <div class="stat-detail">Locations mentioned</div>
    </div>
    <div class="stat-card">
      <div class="stat-icon">🌲</div>
      <div class="stat-value">${{STATS.generations}}</div>
      <div class="stat-label">Generations</div>
      <div class="stat-detail">~${{STATS.latestYear - STATS.earliestYear}} years of history</div>
    </div>
  `;
  
  // On This Day
  renderOnThisDay();
  
  // Presidents row
  const scroll = document.getElementById('presScroll');
  scroll.innerHTML = STATS.presidents.map(p => `
    <div class="pres-card" onclick="showPerson('${{p.id}}')">
      <div class="pres-number">#${{p.number}}</div>
      <div class="pres-name">${{p.name}}</div>
      <span class="pres-party ${{p.party.replace(/[^a-zA-Z]/g,'')}}">${{p.party}}</span>
      <div class="pres-dates">${{p.birthYear}}${{p.deathYear ? ' – ' + p.deathYear : ''}}</div>
    </div>
  `).join('');
  
  // Charts
  renderBarChart('surnameChart', STATS.topSurnames.map(s => [s[0], s[1]]), 'var(--accent-blue)');
  renderBarChart('geoChart', STATS.topGeo.map(g => [g[0], g[1]]), 'var(--accent-green)');
  renderBarChart('centuryChart', STATS.centuries.map(c => [ordinal(c[0]) + ' C', c[1]]), 'var(--accent-purple)');
  
  // Party chart
  const parties = {{}};
  STATS.presidents.forEach(p => {{ parties[p.party] = (parties[p.party] || 0) + 1; }});
  const partyData = Object.entries(parties).sort((a,b) => b[1] - a[1]);
  const partyColors = {{
    'Republican': 'var(--accent-red)', 'Democratic': 'var(--accent-blue)',
    'Whig': 'var(--accent-gold)', 'Federalist': 'var(--accent-purple)',
    'Democratic-Republican': 'var(--accent-green)', 'Independent': 'var(--text-secondary)',
    'National Union': 'var(--accent-cyan)', 'Unknown': 'var(--text-muted)'
  }};
  const partyEl = document.getElementById('partyChart');
  const maxParty = Math.max(...partyData.map(p => p[1]));
  partyEl.innerHTML = partyData.map(([name, count]) => `
    <div class="bar-row">
      <div class="bar-label">${{name}}</div>
      <div class="bar-track">
        <div class="bar-fill" style="width:${{(count/maxParty*100)}}%;background:${{partyColors[name]||'var(--accent-blue)'}}">${{count}}</div>
      </div>
    </div>
  `).join('');
}}

function renderOnThisDay() {{
  const now = new Date();
  const month = now.getMonth() + 1;
  const day = now.getDate();
  const year = now.getFullYear();
  
  const events = [];
  
  PEOPLE.forEach(p => {{
    if (p.birthMonth === month && p.birthDay === day && p.birthYear > 0) {{
      events.push({{ type: 'birth', year: p.birthYear, person: p, place: p.birthPlace }});
    }}
    if (p.deathMonth === month && p.deathDay === day && p.deathYear > 0) {{
      events.push({{ type: 'death', year: p.deathYear, person: p, place: p.deathPlace }});
    }}
  }});
  
  FAMILIES.forEach(f => {{
    if (f.marriageMonth === month && f.marriageDay === day && f.marriageYear > 0) {{
      const h = PEOPLE_MAP[f.husband];
      const w = PEOPLE_MAP[f.wife];
      if (h && w) {{
        events.push({{ type: 'marriage', year: f.marriageYear, person: {{ name: h.name + ' & ' + w.name, id: h.id, president: h.president || w.president }}, place: f.marriagePlace }});
      }}
    }}
  }});
  
  events.sort((a, b) => a.year - b.year);
  
  const monthNames = ['', 'January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December'];
  
  document.getElementById('otdCount').textContent = events.length + ' events on ' + monthNames[month] + ' ' + day;
  
  const grid = document.getElementById('otdGrid');
  if (events.length === 0) {{
    grid.innerHTML = '<div style="padding:24px;color:var(--text-muted);text-align:center;">No events found for today in this family tree.</div>';
    return events;
  }}
  
  grid.innerHTML = events.map(e => `
    <div class="otd-card" onclick="showPerson('${{e.person.id}}')">
      <div class="otd-year">${{e.year}}</div>
      <div class="otd-info">
        <span class="otd-badge ${{e.type}}">${{e.type}}</span>
        <div class="otd-name">${{e.person.name}}${{e.person.president ? '<span class="pres-badge">#' + e.person.president.number + '</span>' : ''}}</div>
        ${{e.place ? '<div class="otd-place">📍 ' + e.place + '</div>' : ''}}
        <div class="otd-years-ago">${{year - e.year}} years ago</div>
      </div>
    </div>
  `).join('');
  
  return events;
}}

function renderBarChart(containerId, data, color) {{
  const el = document.getElementById(containerId);
  const max = Math.max(...data.map(d => d[1]));
  el.innerHTML = data.map(([label, value]) => `
    <div class="bar-row">
      <div class="bar-label">${{label}}</div>
      <div class="bar-track">
        <div class="bar-fill" style="width:${{(value/max*100)}}%;background:${{color}}">${{value}}</div>
      </div>
    </div>
  `).join('');
}}

function ordinal(n) {{
  const s = ['th','st','nd','rd'];
  const v = n % 100;
  return n + (s[(v-20)%10]||s[v]||s[0]);
}}

// ===== PEOPLE TAB =====
function filterPeople() {{
  peoplePage = 1;
  renderPeople();
}}

function setGender(g, btn) {{
  peopleGenderFilter = g;
  peoplePage = 1;
  document.querySelectorAll('#tab-people .filter-btn:not(.pres-filter)').forEach(b => b.classList.remove('active'));
  btn.classList.add('active');
  renderPeople();
}}

function togglePresFilter(btn) {{
  peoplePresFilter = !peoplePresFilter;
  btn.classList.toggle('active', peoplePresFilter);
  peoplePage = 1;
  renderPeople();
}}

function renderPeople() {{
  const query = (document.getElementById('peopleSearch')?.value || '').toLowerCase();
  
  let filtered = PEOPLE.filter(p => {{
    if (peopleGenderFilter !== 'all' && p.sex !== peopleGenderFilter) return false;
    if (peoplePresFilter && !p.president) return false;
    if (query) {{
      const searchStr = (p.name + ' ' + p.birthPlace + ' ' + p.deathPlace + ' ' + p.birthYear + ' ' + p.surname).toLowerCase();
      return searchStr.includes(query);
    }}
    return true;
  }});
  
  // Sort: presidents first, then by birth year
  filtered.sort((a, b) => {{
    if (a.president && !b.president) return -1;
    if (!a.president && b.president) return 1;
    if (a.president && b.president) return a.president.number - b.president.number;
    return (a.birthYear || 9999) - (b.birthYear || 9999);
  }});
  
  const total = filtered.length;
  const totalPages = Math.ceil(total / PEOPLE_PER_PAGE);
  const start = (peoplePage - 1) * PEOPLE_PER_PAGE;
  const page = filtered.slice(start, start + PEOPLE_PER_PAGE);
  
  document.getElementById('peopleCount').textContent = `Showing ${{start+1}}-${{Math.min(start+PEOPLE_PER_PAGE, total)}} of ${{total}} people`;
  
  const grid = document.getElementById('peopleGrid');
  grid.innerHTML = page.map(p => {{
    const isPres = !!p.president;
    const avatarClass = isPres ? 'president' : (p.sex === 'M' ? 'male' : 'female');
    const initial = isPres ? '#' + p.president.number : (p.name[0] || '?');
    const dates = [p.birthYear || '?', p.deathYear || (p.birthYear > 1900 ? '' : '?')].filter(Boolean).join(' – ');
    return `
      <div class="person-card ${{isPres ? 'is-president' : ''}}" onclick="showPerson('${{p.id}}')">
        <div class="person-avatar ${{avatarClass}}">${{initial}}</div>
        <div class="person-info">
          <div class="person-name">${{p.name}}</div>
          <div class="person-dates">${{dates}}</div>
          ${{p.birthPlace ? '<div class="person-place">📍 ' + p.birthPlace + '</div>' : ''}}
        </div>
      </div>
    `;
  }}).join('');
  
  // Pagination
  const pagEl = document.getElementById('peoplePagination');
  if (totalPages <= 1) {{ pagEl.innerHTML = ''; return; }}
  let pagHtml = `<button ${{peoplePage<=1?'disabled':''}} onclick="peoplePage--;renderPeople()">← Prev</button>`;
  const range = getPageRange(peoplePage, totalPages);
  range.forEach(pg => {{
    if (pg === '...') pagHtml += `<button disabled>…</button>`;
    else pagHtml += `<button class="${{pg===peoplePage?'active':''}}" onclick="peoplePage=${{pg}};renderPeople()">${{pg}}</button>`;
  }});
  pagHtml += `<button ${{peoplePage>=totalPages?'disabled':''}} onclick="peoplePage++;renderPeople()">Next →</button>`;
  pagEl.innerHTML = pagHtml;
}}

function getPageRange(current, total) {{
  if (total <= 7) return Array.from({{length:total}}, (_,i) => i+1);
  const pages = [1];
  if (current > 3) pages.push('...');
  for (let i = Math.max(2, current-1); i <= Math.min(total-1, current+1); i++) pages.push(i);
  if (current < total-2) pages.push('...');
  pages.push(total);
  return pages;
}}

// ===== TIMELINE TAB =====
function buildTimelineEvents() {{
  const events = [];
  PEOPLE.forEach(p => {{
    if (p.birthYear > 0) events.push({{ type: 'birth', year: p.birthYear, date: p.birthDate, name: p.name, place: p.birthPlace, id: p.id, president: p.president }});
    if (p.deathYear > 0) events.push({{ type: 'death', year: p.deathYear, date: p.deathDate, name: p.name, place: p.deathPlace, id: p.id, president: p.president }});
  }});
  FAMILIES.forEach(f => {{
    if (f.marriageYear > 0) {{
      const h = PEOPLE_MAP[f.husband];
      const w = PEOPLE_MAP[f.wife];
      if (h && w) events.push({{ type: 'marriage', year: f.marriageYear, date: f.marriageDate, name: h.name + ' & ' + w.name, place: f.marriagePlace, id: h.id, president: h.president || w.president }});
    }}
  }});
  return events.sort((a,b) => a.year - b.year);
}}

let allTimelineEvents = [];

function filterTimeline() {{
  timelinePage = 1;
  renderTimeline();
}}

function setTimelineType(t, btn) {{
  timelineTypeFilter = t;
  timelinePage = 1;
  document.querySelectorAll('#tab-timeline .filter-btn').forEach(b => b.classList.remove('active'));
  btn.classList.add('active');
  renderTimeline();
}}

function renderTimeline() {{
  if (!allTimelineEvents.length) allTimelineEvents = buildTimelineEvents();
  
  const query = (document.getElementById('timelineSearch')?.value || '').toLowerCase();
  
  let filtered = allTimelineEvents.filter(e => {{
    if (timelineTypeFilter !== 'all' && e.type !== timelineTypeFilter) return false;
    if (query) {{
      return (e.name + ' ' + e.year + ' ' + (e.place||'')).toLowerCase().includes(query);
    }}
    return true;
  }});
  
  const total = filtered.length;
  const totalPages = Math.ceil(total / TIMELINE_PER_PAGE);
  const start = (timelinePage - 1) * TIMELINE_PER_PAGE;
  const page = filtered.slice(start, start + TIMELINE_PER_PAGE);
  
  const container = document.getElementById('timelineContainer');
  container.innerHTML = page.map(e => `
    <div class="timeline-item" onclick="showPerson('${{e.id}}')">
      <div class="timeline-dot ${{e.type}}"></div>
      <div class="timeline-year">${{e.year}} — ${{e.type === 'birth' ? '🟢 Born' : e.type === 'death' ? '🔴 Died' : '💛 Married'}}</div>
      <div class="timeline-name">${{e.name}}${{e.president ? ' 🏛️' : ''}}</div>
      ${{e.place ? '<div class="timeline-detail">📍 ' + e.place + '</div>' : ''}}
    </div>
  `).join('');
  
  // Pagination
  const pagEl = document.getElementById('timelinePagination');
  if (totalPages <= 1) {{ pagEl.innerHTML = ''; return; }}
  let html = `<button ${{timelinePage<=1?'disabled':''}} onclick="timelinePage--;renderTimeline()">← Prev</button>`;
  const range = getPageRange(timelinePage, totalPages);
  range.forEach(pg => {{
    if (pg === '...') html += `<button disabled>…</button>`;
    else html += `<button class="${{pg===timelinePage?'active':''}}" onclick="timelinePage=${{pg}};renderTimeline()">${{pg}}</button>`;
  }});
  html += `<button ${{timelinePage>=totalPages?'disabled':''}} onclick="timelinePage++;renderTimeline()">Next →</button>`;
  pagEl.innerHTML = html;
}}

// ===== TREE TAB =====
function renderTree() {{
  const select = document.getElementById('treeRoot');
  if (!select.options.length) {{
    // Populate with presidents
    STATS.presidents.forEach(p => {{
      const opt = document.createElement('option');
      opt.value = p.id;
      opt.textContent = '#' + p.number + ' ' + p.name;
      select.appendChild(opt);
    }});
  }}
  
  const rootId = select.value;
  const root = PEOPLE_MAP[rootId];
  if (!root) return;
  
  const canvas = document.getElementById('tree-canvas');
  canvas.innerHTML = '';
  
  // Build ancestor tree (up to 4 generations)
  const nodes = [];
  const lines = [];
  
  function addNode(personId, x, y, gen) {{
    const p = PEOPLE_MAP[personId];
    if (!p || gen > 4) return;
    
    const cls = p.president ? 'president' : (p.sex === 'M' ? 'male' : 'female');
    const dates = [p.birthYear || '?', p.deathYear || ''].filter(Boolean).join('–');
    
    nodes.push({{ id: p.id, x, y, name: p.name, dates, cls }});
    
    // Find parents
    if (p.familyChild && gen < 4) {{
      const fam = FAMILIES.find(f => f.id === p.familyChild);
      if (fam) {{
        const spacing = 280 / Math.pow(2, gen);
        if (fam.husband && PEOPLE_MAP[fam.husband]) {{
          lines.push({{ x1: x, y1: y, x2: x - spacing, y2: y + 100 }});
          addNode(fam.husband, x - spacing, y + 100, gen + 1);
        }}
        if (fam.wife && PEOPLE_MAP[fam.wife]) {{
          lines.push({{ x1: x, y1: y, x2: x + spacing, y2: y + 100 }});
          addNode(fam.wife, x + spacing, y + 100, gen + 1);
        }}
      }}
    }}
  }}
  
  addNode(rootId, 500, 30, 0);
  
  // Draw SVG lines
  const svg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
  svg.style.cssText = 'position:absolute;top:0;left:0;width:100%;height:100%;z-index:1;';
  lines.forEach(l => {{
    const line = document.createElementNS('http://www.w3.org/2000/svg', 'path');
    const midY = (l.y1 + l.y2) / 2;
    line.setAttribute('d', `M${{l.x1+80}},${{l.y1+40}} C${{l.x1+80}},${{midY+20}} ${{l.x2+80}},${{midY}} ${{l.x2+80}},${{l.y2}}`);
    line.setAttribute('stroke', 'rgba(255,255,255,0.15)');
    line.setAttribute('stroke-width', '2');
    line.setAttribute('fill', 'none');
    svg.appendChild(line);
  }});
  canvas.appendChild(svg);
  
  // Draw nodes
  nodes.forEach(n => {{
    const div = document.createElement('div');
    div.className = `tree-node ${{n.cls}}`;
    div.style.left = n.x + 'px';
    div.style.top = n.y + 'px';
    div.innerHTML = `<div class="tree-node-name">${{n.name}}</div><div class="tree-node-dates">${{n.dates}}</div>`;
    div.onclick = () => showPerson(n.id);
    canvas.appendChild(div);
  }});
}}

function treeZoom(factor) {{
  treeScale *= factor;
  const canvas = document.getElementById('tree-canvas');
  canvas.style.transform = `scale(${{treeScale}})`;
  canvas.style.transformOrigin = 'center top';
}}

function treeReset() {{
  treeScale = 1;
  document.getElementById('tree-canvas').style.transform = '';
  renderTree();
}}

// ===== DAILY ALERTS TAB =====
function renderAlerts() {{
  const now = new Date();
  const monthNames = ['January','February','March','April','May','June','July','August','September','October','November','December'];
  document.getElementById('alertsDate').textContent = monthNames[now.getMonth()] + ' ' + now.getDate() + ', ' + now.getFullYear();
  
  // Random ancestor spotlight
  randomSpotlight();
  
  // OTD events for alerts tab
  const month = now.getMonth() + 1;
  const day = now.getDate();
  const year = now.getFullYear();
  
  const events = [];
  PEOPLE.forEach(p => {{
    if (p.birthMonth === month && p.birthDay === day && p.birthYear > 0)
      events.push({{ type: 'birth', year: p.birthYear, person: p, place: p.birthPlace }});
    if (p.deathMonth === month && p.deathDay === day && p.deathYear > 0)
      events.push({{ type: 'death', year: p.deathYear, person: p, place: p.deathPlace }});
  }});
  FAMILIES.forEach(f => {{
    if (f.marriageMonth === month && f.marriageDay === day && f.marriageYear > 0) {{
      const h = PEOPLE_MAP[f.husband]; const w = PEOPLE_MAP[f.wife];
      if (h && w) events.push({{ type: 'marriage', year: f.marriageYear, person: {{ name: h.name + ' & ' + w.name, id: h.id, president: h.president }}, place: f.marriagePlace }});
    }}
  }});
  events.sort((a,b) => a.year - b.year);
  
  const grid = document.getElementById('alertsOtdGrid');
  if (events.length === 0) {{
    grid.innerHTML = '<div style="padding:24px;color:var(--text-muted);text-align:center;">No events for today in this presidential family tree. Try the spotlight below!</div>';
  }} else {{
    grid.innerHTML = events.map(e => `
      <div class="otd-card" onclick="showPerson('${{e.person.id}}')">
        <div class="otd-year">${{e.year}}</div>
        <div class="otd-info">
          <span class="otd-badge ${{e.type}}">${{e.type}}</span>
          <div class="otd-name">${{e.person.name}}${{e.person.president ? '<span class="pres-badge">#' + e.person.president.number + '</span>' : ''}}</div>
          ${{e.place ? '<div class="otd-place">📍 ' + e.place + '</div>' : ''}}
          <div class="otd-years-ago">${{year - e.year}} years ago today</div>
        </div>
      </div>
    `).join('');
  }}
  
  // Fun facts
  renderFunFacts();
}}

function randomSpotlight() {{
  const candidates = PEOPLE.filter(p => p.birthYear > 0 && p.name);
  const person = candidates[Math.floor(Math.random() * candidates.length)];
  if (!person) return;
  
  const card = document.getElementById('spotlightCard');
  const lifespan = person.deathYear ? (person.deathYear - person.birthYear) + ' years' : 'Living or unknown';
  
  card.innerHTML = `
    <div class="spotlight-label">🎲 Random Ancestor Spotlight</div>
    <div class="spotlight-name">${{person.name}}${{person.president ? ' 🏛️' : ''}}</div>
    <div class="spotlight-dates">${{person.birthDate || person.birthYear}} ${{person.deathDate ? '— ' + person.deathDate : ''}}</div>
    ${{person.president ? '<div style="margin-bottom:16px;"><span style="background:var(--gradient-gold);color:#000;padding:4px 12px;border-radius:6px;font-size:13px;font-weight:700;">President #' + person.president.number + ' · ' + person.president.party + '</span></div>' : ''}}
    <div class="spotlight-details">
      ${{person.birthPlace ? '<div class="spotlight-detail"><div class="spotlight-detail-label">Born</div><div class="spotlight-detail-value">📍 ' + person.birthPlace + '</div></div>' : ''}}
      ${{person.deathPlace ? '<div class="spotlight-detail"><div class="spotlight-detail-label">Died</div><div class="spotlight-detail-value">📍 ' + person.deathPlace + '</div></div>' : ''}}
      <div class="spotlight-detail"><div class="spotlight-detail-label">Lifespan</div><div class="spotlight-detail-value">${{lifespan}}</div></div>
      ${{person.children.length ? '<div class="spotlight-detail"><div class="spotlight-detail-label">Children</div><div class="spotlight-detail-value">' + person.children.length + '</div></div>' : ''}}
    </div>
    <button class="shuffle-btn" onclick="randomSpotlight()">🎲 Shuffle — New Spotlight</button>
  `;
}}

function renderFunFacts() {{
  const facts = [];
  
  // Oldest person
  const withAge = PEOPLE.filter(p => p.birthYear > 0 && p.deathYear > 0).map(p => ({{ ...p, age: p.deathYear - p.birthYear }}));
  withAge.sort((a,b) => b.age - a.age);
  if (withAge.length) facts.push(`🏆 <strong>Longest-lived:</strong> ${{withAge[0].name}} lived to ${{withAge[0].age}} years (${{withAge[0].birthYear}}–${{withAge[0].deathYear}})`);
  
  // Most children
  const byChildren = [...PEOPLE].sort((a,b) => b.children.length - a.children.length);
  if (byChildren.length && byChildren[0].children.length > 0) facts.push(`👶 <strong>Most children:</strong> ${{byChildren[0].name}} had ${{byChildren[0].children.length}} children`);
  
  // Presidents count
  const presCount = PEOPLE.filter(p => p.president).length;
  facts.push(`🏛️ <strong>${{presCount}} US Presidents</strong> are represented in this family tree with their ancestors and descendants`);
  
  // Earliest ancestor
  const earliest = PEOPLE.filter(p => p.birthYear > 0).sort((a,b) => a.birthYear - b.birthYear)[0];
  if (earliest) facts.push(`📜 <strong>Earliest ancestor:</strong> ${{earliest.name}} (born ${{earliest.birthYear}}) — that's ${{new Date().getFullYear() - earliest.birthYear}} years ago!`);
  
  // Most common surname
  if (STATS.topSurnames.length) facts.push(`📛 <strong>Most common surname:</strong> ${{STATS.topSurnames[0][0]}} (${{STATS.topSurnames[0][1]}} people)`);
  
  document.getElementById('funFacts').innerHTML = facts.map(f => `<div style="padding:8px 0;border-bottom:1px solid var(--border);font-size:14px;">${{f}}</div>`).join('');
}}

// ===== PERSON MODAL =====
function showPerson(id) {{
  const p = PEOPLE_MAP[id];
  if (!p) return;
  
  const modal = document.getElementById('personModal');
  const content = document.getElementById('modalContent');
  
  let html = `<div class="modal-close" onclick="closeModal()">✕</div>`;
  html += `<h2>${{p.name}}</h2>`;
  if (p.president) {{
    html += `<div class="pres-tag">🏛️ President #${{p.president.number}} · ${{p.president.party}}</div>`;
  }}
  
  html += `<div class="modal-section"><h4>Vital Records</h4>`;
  if (p.birthDate || p.birthPlace) html += `<div class="modal-row"><span class="label">🟢 Born:</span><span>${{p.birthDate || ''}} ${{p.birthPlace ? '— ' + p.birthPlace : ''}}</span></div>`;
  if (p.deathDate || p.deathPlace) html += `<div class="modal-row"><span class="label">🔴 Died:</span><span>${{p.deathDate || ''}} ${{p.deathPlace ? '— ' + p.deathPlace : ''}}</span></div>`;
  if (p.burialPlace) html += `<div class="modal-row"><span class="label">⚰️ Buried:</span><span>${{p.burialDate || ''}} ${{p.burialPlace ? '— ' + p.burialPlace : ''}}</span></div>`;
  if (p.occupation) html += `<div class="modal-row"><span class="label">💼 Occupation:</span><span>${{p.occupation}}</span></div>`;
  if (p.birthYear && p.deathYear) html += `<div class="modal-row"><span class="label">⏳ Lifespan:</span><span>${{p.deathYear - p.birthYear}} years</span></div>`;
  html += `</div>`;
  
  // Parents
  if (p.parents.father || p.parents.mother) {{
    html += `<div class="modal-section"><h4>Parents</h4>`;
    if (p.parents.father) html += `<div class="modal-row"><span class="label">👨 Father:</span><span class="modal-link" onclick="showPerson('${{p.parents.father.id}}')">${{p.parents.father.name}}</span></div>`;
    if (p.parents.mother) html += `<div class="modal-row"><span class="label">👩 Mother:</span><span class="modal-link" onclick="showPerson('${{p.parents.mother.id}}')">${{p.parents.mother.name}}</span></div>`;
    html += `</div>`;
  }}
  
  // Spouses
  if (p.spouses.length) {{
    html += `<div class="modal-section"><h4>Spouse${{p.spouses.length > 1 ? 's' : ''}}</h4>`;
    p.spouses.forEach(s => {{
      html += `<div class="modal-row"><span class="label">💍</span><span class="modal-link" onclick="showPerson('${{s.id}}')">${{s.name}}</span>${{s.marriage && s.marriage.date ? ' <span style="color:var(--text-muted);font-size:12px;">(' + s.marriage.date + ')</span>' : ''}}</div>`;
    }});
    html += `</div>`;
  }}
  
  // Children
  if (p.children.length) {{
    html += `<div class="modal-section"><h4>Children (${{p.children.length}})</h4>`;
    p.children.forEach(c => {{
      html += `<div class="modal-row"><span class="modal-link" onclick="showPerson('${{c.id}}')">${{c.name}}</span></div>`;
    }});
    html += `</div>`;
  }}
  
  content.innerHTML = html;
  modal.classList.add('show');
}}

function closeModal() {{
  document.getElementById('personModal').classList.remove('show');
}}

// ===== GLOBAL SEARCH =====
document.getElementById('globalSearch').addEventListener('input', function(e) {{
  const q = e.target.value.trim().toLowerCase();
  if (!q) return;
  
  // Switch to people tab and search
  switchTab('people');
  document.getElementById('peopleSearch').value = q;
  peoplePage = 1;
  renderPeople();
}});

// ===== KEYBOARD =====
document.addEventListener('keydown', e => {{
  if (e.key === 'Escape') closeModal();
}});

// ===== INIT =====
renderDashboard();
renderPeople();
renderTimeline();
</script>
</body>
</html>'''
    
    return html


def main():
    global GEDCOM_PATH, OUTPUT_PATH, TITLE, SUBTITLE
    
    args = sys.argv[1:]
    if not args or args[0] in ('-h', '--help'):
        print(__doc__)
        sys.exit(0 if args else 1)
    
    GEDCOM_PATH = args[0]
    if not os.path.exists(GEDCOM_PATH):
        print(f"Error: GEDCOM file not found: {GEDCOM_PATH}")
        sys.exit(1)
    
    # Parse remaining args
    positional = []
    i = 1
    while i < len(args):
        if args[i] == '--title' and i + 1 < len(args):
            TITLE = args[i + 1]; i += 2
        elif args[i] == '--subtitle' and i + 1 < len(args):
            SUBTITLE = args[i + 1]; i += 2
        else:
            positional.append(args[i]); i += 1
    
    OUTPUT_PATH = positional[0] if positional else 'family-explorer.html'
    
    print(f"Parsing GEDCOM file: {GEDCOM_PATH}")
    individuals, families = parse_gedcom(GEDCOM_PATH)
    print(f"  Found {len(individuals)} individuals and {len(families)} families")
    
    print("Building data structures...")
    people = build_people_json(individuals, families)
    fam_list = build_families_json(families, individuals)
    
    # Count presidents
    presidents = [p for p in people if p['president']]
    if presidents:
        print(f"  Detected {len(presidents)} presidents")
    
    print("Computing statistics...")
    stats = compute_stats(people, fam_list)
    
    print("Generating HTML...")
    html = generate_html(people, fam_list, stats)
    
    with open(OUTPUT_PATH, 'w', encoding='utf-8') as f:
        f.write(html)
    
    size_kb = os.path.getsize(OUTPUT_PATH) / 1024
    print(f"  Output: {OUTPUT_PATH} ({size_kb:.0f} KB)")
    print("Done!")


if __name__ == '__main__':
    main()
