#!/bin/bash
# Gateway Monitor and Auto-Restart Script

# Configuration
LOG_DIR="$HOME/.openclaw/logs"
LOG_FILE="$LOG_DIR/gateway_monitor.log"
MAX_LOG_AGE_DAYS=7

# Create log directory if it doesn't exist
mkdir -p "$LOG_DIR"

# Function to log messages
log_message() {
    echo "$(date '+%Y-%m-%d %H:%M:%S') - $1" >> "$LOG_FILE"
}

# Function to rotate logs older than MAX_LOG_AGE_DAYS
rotate_logs() {
    find "$LOG_DIR" -name "gateway_monitor*.log" -mtime +$MAX_LOG_AGE_DAYS -delete 2>/dev/null || true
}

# Function to check if gateway is running
check_gateway_status() {
    # Try to check if the gateway is responding
    if openclaw gateway status 2>/dev/null | grep -q "running"; then
        return 0  # Gateway is running
    else
        return 1  # Gateway is not running
    fi
}

# Function to restart the gateway
restart_gateway() {
    log_message "Attempting to restart gateway..."
    
    # Try to restart the gateway
    restart_output=$(openclaw gateway restart 2>&1)
    restart_exit_code=$?
    
    if [ $restart_exit_code -eq 0 ]; then
        log_message "Gateway restart successful"
        return 0
    else
        log_message "Gateway restart failed: $restart_output"
        
        # Try alternative restart method using launchctl
        if launchctl list | grep -q "ai.openclaw.gateway"; then
            log_message "Attempting restart via launchctl..."
            launchctl bootout gui/$UID/ai.openclaw.gateway 2>/dev/null
            sleep 2
            if openclaw gateway install && openclaw gateway start; then
                log_message "Gateway restarted successfully via launchctl method"
                return 0
            else
                log_message "Gateway restart via launchctl also failed"
                return 1
            fi
        else
            log_message "Gateway service not found in launchctl"
            return 1
        fi
    fi
}

# Function to diagnose gateway issues
diagnose_issues() {
    log_message "Diagnosing gateway issues..."
    
    # Check for common issues
    if pgrep -f "openclaw-gateway" >/dev/null; then
        log_message "Found existing gateway processes - killing them before restart"
        pkill -f "openclaw-gateway" 2>/dev/null || true
        sleep 2
    fi
    
    # Check if port is in use
    if lsof -i :18789 >/dev/null 2>&1; then
        log_message "Port 18789 is in use by another process"
        lsof -i :18789 >> "$LOG_FILE" 2>&1
    fi
}

# Main execution
log_message "Starting gateway health check"

# Rotate old logs
rotate_logs

# Check gateway status
if check_gateway_status; then
    log_message "Gateway is running normally"
else
    log_message "Gateway is not running - attempting restart"
    
    # Diagnose potential issues first
    diagnose_issues
    
    # Attempt to restart the gateway
    if restart_gateway; then
        log_message "Gateway successfully restarted"
    else
        log_message "Failed to restart gateway after diagnosis"
    fi
fi

log_message "Gateway health check completed"