#!/usr/bin/env python3
"""
Garmin Connect CLI for OpenClaw Skill - PRO Edition

Features:
- Natural language queries ("how did I sleep?")
- Full activity data with HR zones, training effect
- Sleep analysis (Deep, Light, REM, Awake)
- Body composition & VO2 max
- ASCII charts for trends
- FIT/GPX file download
- Race predictions
- Week-over-week comparison

Usage:
    python3 garmin.py <command> [options]

Commands:
    ask             Natural language query (e.g., "how did I sleep?")
    summary         Full daily summary
    activities      Recent activities
    activity        Specific activity details
    stats           Daily stats
    sleep           Sleep data
    hr              Heart rate data
    hrv             HRV data
    stress          Stress data
    body-battery    Body battery
    body            Body composition (weight, muscle, fat)
    vo2max          VO2 max & cardio fitness
    training        Training readiness & status
    race            Race predictions
    week            Weekly overview
    compare         Week over week comparison
    trends          7-day trends with ASCII charts
    chart           ASCII chart for any metric
    download        Download FIT/GPX files
    devices         Connected devices
    workouts        Saved workouts
    badges          Earned badges
    export          Full JSON export
"""

import argparse
import json
import os
import sys
import re
from datetime import date, timedelta
from pathlib import Path

try:
    from garminconnect import Garmin
except ImportError:
    print("Error: garminconnect not installed. Run: pip3 install garminconnect")
    sys.exit(1)

TOKEN_DIR = Path.home() / ".config" / "garmin-connect" / "tokens"
CREDENTIALS_FILE = Path.home() / ".config" / "garmin-connect" / "credentials.json"

# Emoji indicators
EMOJI = {
    "steps": "👣", "distance": "📏", "calories": "🔥", "hr": "❤️",
    "stress": "😤", "sleep": "😴", "battery": "🔋", "activity": "🏃",
    "training": "💪", "weight": "⚖️", "water": "💧", "floors": "🏢",
    "good": "✅", "warning": "⚠️", "bad": "❌", "up": "📈", "down": "📉",
    "same": "➡️", "chart": "📊", "vo2max": "🫁", "body": "🏋️",
}


def get_credentials():
    email = os.environ.get("GARMIN_EMAIL")
    password = os.environ.get("GARMIN_PASSWORD")
    if email and password:
        return email, password
    if CREDENTIALS_FILE.exists():
        with open(CREDENTIALS_FILE) as f:
            creds = json.load(f)
            return creds.get("email"), creds.get("password")
    return None, None


def get_client():
    email, password = get_credentials()
    if not email or not password:
        print("Error: No credentials. Run: python3 garmin.py login")
        return None
    try:
        client = Garmin(email=email, password=password)
        if TOKEN_DIR.exists():
            client.login(tokenstore=str(TOKEN_DIR))
        else:
            client.login()
            TOKEN_DIR.mkdir(parents=True, exist_ok=True)
            client.garth.dump(str(TOKEN_DIR))
        return client
    except Exception as e:
        print(f"Error: {e}\nRun: python3 garmin.py login")
        return None


def format_time(seconds):
    if not seconds:
        return "N/A"
    h = seconds // 3600
    m = (seconds % 3600) // 60
    s = seconds % 60
    if h > 0:
        return f"{h}:{m:02d}:{s:02d}"
    return f"{m}:{s:02d}"


def format_duration(seconds):
    if not seconds:
        return "N/A"
    m = seconds // 60
    if m >= 60:
        h = m // 60
        m = m % 60
        return f"{h}h {m}m"
    return f"{m}m"


def ascii_chart(data, width=40, height=10, title=None):
    """Generate ASCII chart from data points."""
    if not data or len(data) < 2:
        return "Not enough data for chart"
    
    # Filter out None values
    data = [(k, v) for k, v in data if v is not None]
    if not data:
        return "No valid data points"
    
    values = [v for _, v in data]
    min_val = min(values)
    max_val = max(values)
    range_val = max_val - min_val if max_val != min_val else 1
    
    lines = []
    if title:
        lines.append(f"\n{title}")
        lines.append("─" * (width + 10))
    
    # Create chart grid
    for row in range(height, 0, -1):
        threshold = min_val + (range_val * row / height)
        line = f"{int(threshold):>5} │"
        for _, v in data[-width:]:
            if v >= threshold:
                line += "█"
            else:
                line += " "
        lines.append(line)
    
    # X-axis
    lines.append("     └" + "─" * min(len(data), width))
    
    # Labels
    if len(data) >= 2:
        lines.append(f"      {data[0][0][:8]}" + " " * (width - 16) + f"{data[-1][0][:8]}")
    
    return "\n".join(lines)


def natural_language_query(query, client):
    """Parse natural language query and return relevant data."""
    query = query.lower()
    result = []
    
    # Sleep-related
    if any(word in query for word in ["sleep", "geschlaf", "night", "nacht", "müde", "tired"]):
        sleep = client.get_sleep_data(date.today().isoformat())
        daily = sleep.get('dailySleepDTO', {}) if sleep else {}
        if daily.get('sleepTimeSeconds'):
            total = daily.get('sleepTimeSeconds', 0) / 3600
            deep = (daily.get('deepSleepSeconds') or 0) / 3600
            light = (daily.get('lightSleepSeconds') or 0) / 3600
            rem = (daily.get('remSleepSeconds') or 0) / 3600
            result.append(f"😴 **Letzte Nacht:**")
            result.append(f"   Gesamt: {total:.1f}h (Deep: {deep:.1f}h, Light: {light:.1f}h, REM: {rem:.1f}h)")
            if total < 7:
                result.append(f"   ⚠️ Unter 7 Stunden - du könntest müde sein")
            elif deep < 1:
                result.append(f"   ⚠️ Wenig Tiefschlaf")
            else:
                result.append(f"   ✅ Guter Schlaf!")
        else:
            result.append("Keine Schlafdaten für heute verfügbar")
    
    # Activity-related
    elif any(word in query for word in ["activity", "activities", "aktivität", "workout", "training", "exercise", "exercise"]):
        activities = client.get_activities(0, 5)
        if activities:
            act = activities[0]
            result.append(f"🏃 **Letzte Aktivität:** {act.get('activityName')}")
            result.append(f"   Typ: {act.get('activityType', {}).get('typeKey')}")
            result.append(f"   Dauer: {format_duration(act.get('duration', 0))}")
            result.append(f"   Kalorien: {act.get('calories')} kcal")
            result.append(f"   Ø HR: {act.get('averageHR')} bpm")
        else:
            result.append("Keine aktuellen Aktivitäten gefunden")
    
    # Heart rate
    elif any(word in query for word in ["heart", "herz", "hr", "puls", "pulse"]):
        stats = client.get_stats(date.today().isoformat())
        hr = client.get_heart_rates(date.today().isoformat())
        result.append(f"❤️ **Herzfrequenz heute:**")
        result.append(f"   Ruhe-HF: {stats.get('restingHeartRate', 'N/A')} bpm")
        result.append(f"   Minimum: {stats.get('minHeartRate', 'N/A')} bpm")
        result.append(f"   Maximum: {stats.get('maxHeartRate', 'N/A')} bpm")
        rhr = stats.get('restingHeartRate')
        if rhr:
            if rhr < 60:
                result.append(f"   ✅ Exzellent! (Athleten-Bereich)")
            elif rhr < 70:
                result.append(f"   ✅ Gut!")
            elif rhr < 80:
                result.append(f"   ⚠️ Durchschnittlich")
            else:
                result.append(f"   ⚠️ Etwas hoch")
    
    # Steps / Movement
    elif any(word in query for word in ["step", "schritt", "walk", "lauf", "move", "beweg"]):
        stats = client.get_stats(date.today().isoformat())
        steps = stats.get('totalSteps', 0)
        goal = 10000
        result.append(f"👣 **Bewegung heute:**")
        result.append(f"   Schritte: {steps:,}")
        result.append(f"   Distanz: {round((stats.get('totalDistanceMeters') or 0) / 1000, 2)} km")
        result.append(f"   Stockwerke: {stats.get('floorsAscended', 'N/A')}")
        pct = (steps / goal) * 100 if steps else 0
        bar = "█" * int(pct / 5) + "░" * (20 - int(pct / 5))
        result.append(f"   Ziel: [{bar}] {pct:.0f}%")
        if steps >= goal:
            result.append(f"   ✅ Tagesziel erreicht!")
        elif steps >= goal * 0.7:
            result.append(f"   📈 Fast da! Noch {goal - steps:,} Schritte")
    
    # Body Battery
    elif any(word in query for word in ["battery", "energie", "energy", "kraft", "müde"]):
        stats = client.get_stats(date.today().isoformat())
        bb = stats.get('bodyBatteryMostRecentValue')
        result.append(f"🔋 **Body Battery:**")
        result.append(f"   Aktuell: {bb}%")
        if bb:
            if bb >= 80:
                result.append(f"   ✅ Voll geladen! Bereit für alles!")
            elif bb >= 50:
                result.append(f"   ✅ Gut! Genug Energie für ein Workout")
            elif bb >= 30:
                result.append(f"   ⚠️ Niedrig - entspannte Aktivität empfohlen")
            else:
                result.append(f"   ❌ Erschöpft - Ruhe empfohlen!")
    
    # Stress
    elif any(word in query for word in ["stress", "entspann", "relax"]):
        stress = client.get_all_day_stress(date.today().isoformat())
        if stress:
            avg = stress.get('avgStressLevel')
            max_s = stress.get('maxStressLevel')
            result.append(f"😤 **Stress heute:**")
            result.append(f"   Durchschnitt: {avg}")
            result.append(f"   Maximum: {max_s}")
            if avg:
                if avg < 25:
                    result.append(f"   ✅ Sehr entspannt!")
                elif avg < 50:
                    result.append(f"   ⚠️ Mäßiger Stress")
                elif avg < 75:
                    result.append(f"   ⚠️ Hoher Stress - Pausen einplanen!")
                else:
                    result.append(f"   ❌ Sehr hoher Stress - Entspannung nötig!")
    
    # Training / Readiness
    elif any(word in query for word in ["train", "workout", "readiness", "bereit", "fitness"]):
        result.append(f"💪 **Training heute:**")
        im = client.get_intensity_minutes_data(date.today().isoformat())
        result.append(f"   Moderate: {im.get('moderateMinutes', 0)} min")
        result.append(f"   Vigorous: {im.get('vigorousMinutes', 0)} min")
        total = im.get('weeklyTotal', 0)
        goal = im.get('weekGoal', 300)
        result.append(f"   Wochenziel: {total}/{goal} min ({(total/goal*100):.0f}%)")
        
        try:
            tr = client.get_training_readiness(date.today().isoformat())
            if isinstance(tr, list) and tr:
                tr = tr[0]
            if tr and tr.get('trainingReadinessScore'):
                result.append(f"\n   Training Readiness: {tr.get('trainingReadinessScore')}")
                result.append(f"   Level: {tr.get('trainingReadinessLevel', 'N/A')}")
        except:
            pass
    
    # Weight / Body
    elif any(word in query for word in ["weight", "gewicht", "body", "körper", "fat", "fett", "muscle", "muskel"]):
        try:
            body = client.get_body_composition(date.today().isoformat())
            if body:
                result.append(f"⚖️ **Body Composition:**")
                # Parse body composition data
                result.append(f"   Daten verfügbar - nutze 'body' command für Details")
        except:
            result.append("Keine Body Composition Daten verfügbar")
    
    # Race predictions
    elif any(word in query for word in ["race", "lauf", "run", "marathon", "5k", "10k", "predict"]):
        rp = client.get_race_predictions()
        result.append(f"🏁 **Race Predictions:**")
        result.append(f"   5K: {format_time(rp.get('time5K'))}")
        result.append(f"   10K: {format_time(rp.get('time10K'))}")
        result.append(f"   Halbmarathon: {format_time(rp.get('timeHalfMarathon'))}")
        result.append(f"   Marathon: {format_time(rp.get('timeMarathon'))}")
    
    # Default: summary
    else:
        stats = client.get_stats(date.today().isoformat())
        result.append(f"📊 **Summary heute:**")
        result.append(f"   Schritte: {stats.get('totalSteps', 'N/A'):,}")
        result.append(f"   Ruhe-HF: {stats.get('restingHeartRate', 'N/A')} bpm")
        result.append(f"   Stress: {stats.get('averageStressLevel', 'N/A')}")
        result.append(f"   Body Battery: {stats.get('bodyBatteryMostRecentValue', 'N/A')}%")
    
    return "\n".join(result)


# ... [Keep all the other functions from the previous version, just add the new ones below]
# [For brevity, I'll show just the new functions and the updated main()]


def cmd_ask(args):
    """Natural language query."""
    client = get_client()
    if not client:
        return 1
    
    query = " ".join(args.query) if args.query else args.question
    if not query:
        print("Error: Please provide a question")
        print("Example: python3 garmin.py ask 'how did I sleep?'")
        return 1
    
    result = natural_language_query(query, client)
    print(result)
    return 0


def cmd_body(args):
    """Body composition data."""
    client = get_client()
    if not client:
        return 1
    
    target_date = args.date or date.today().isoformat()
    
    try:
        body = client.get_body_composition(target_date)
        
        if args.json:
            print(json.dumps(body, indent=2, default=str))
            return 0
        
        print(f"\n{EMOJI['body']} BODY COMPOSITION - {target_date}")
        print("=" * 50)
        
        if body:
            # Extract weight and composition data
            weight = body.get('totalWeight')
            if weight:
                print(f"Weight:         {weight} kg")
            
            # Look for muscle mass, fat percentage, etc.
            for key, value in body.items():
                if 'muscle' in key.lower() or 'fat' in key.lower() or 'bmi' in key.lower():
                    print(f"{key}:          {value}")
        else:
            print("Keine Body Composition Daten verfügbar")
            print("Tipp: Stelle sicher dass du eine Garmin Waage (Index S2) hast")
            
    except Exception as e:
        print(f"Fehler beim Abrufen der Body Composition: {e}")
        print("Möglicherweise keine Daten oder keine Index Waage vorhanden")
    
    return 0


def cmd_vo2max(args):
    """VO2 max and cardio fitness."""
    client = get_client()
    if not client:
        return 1
    
    try:
        # Get VO2 max from activities
        activities = client.get_activities(0, 10)
        
        if args.json:
            data = {"activities": activities}
            print(json.dumps(data, indent=2, default=str))
            return 0
        
        print(f"\n{EMOJI['vo2max']} VO2 MAX & CARDIO FITNESS")
        print("=" * 50)
        
        vo2max_values = []
        for act in activities:
            if act.get('vO2MaxValue'):
                vo2max_values.append({
                    "date": act.get('startTimeLocal', '')[:10],
                    "type": act.get('activityType', {}).get('typeKey'),
                    "vo2max": act.get('vO2MaxValue')
                })
        
        if vo2max_values:
            print("\nVO2 Max aus Aktivitäten:")
            for v in vo2max_values[-5:]:
                print(f"  {v['date']}: {v['vo2max']} ({v['type']})")
            
            avg = sum(v['vo2max'] for v in vo2max_values) / len(vo2max_values)
            print(f"\nDurchschnitt: {avg:.1f}")
            
            # Rating
            if avg >= 50:
                print("Bewertung: ✅ Exzellent (Athleten-Niveau)")
            elif avg >= 40:
                print("Bewertung: ✅ Gut")
            elif avg >= 30:
                print("Bewertung: ⚠️ Durchschnittlich")
            else:
                print("Bewertung: ⚠️ Verbesserungswürdig")
        else:
            print("Keine VO2 Max Daten in den letzten Aktivitäten gefunden")
            print("Tipp: VO2 Max wird bei Laufen und Radfahren berechnet")
        
        # Race predictions
        rp = client.get_race_predictions()
        print(f"\n🏁 Race Predictions:")
        print(f"   5K:         {format_time(rp.get('time5K'))}")
        print(f"   10K:        {format_time(rp.get('time10K'))}")
        print(f"   Halbmarathon: {format_time(rp.get('timeHalfMarathon'))}")
        print(f"   Marathon:   {format_time(rp.get('timeMarathon'))}")
        
    except Exception as e:
        print(f"Fehler: {e}")
    
    return 0


def cmd_chart(args):
    """Generate ASCII chart for a metric."""
    client = get_client()
    if not client:
        return 1
    
    metric = args.metric.lower()
    days = args.days or 7
    
    data = []
    for i in range(days):
        d = date.today() - timedelta(days=i)
        stats = client.get_stats(d.isoformat())
        
        if metric in ["steps", "schritte"]:
            data.append((d.isoformat()[5:], stats.get('totalSteps') or 0))
        elif metric in ["hr", "heart", "herz"]:
            data.append((d.isoformat()[5:], stats.get('restingHeartRate') or 0))
        elif metric in ["calories", "kalorien"]:
            data.append((d.isoformat()[5:], stats.get('totalKilocalories') or 0))
        elif metric in ["stress"]:
            data.append((d.isoformat()[5:], stats.get('averageStressLevel') or 0))
        elif metric in ["floors", "stockwerke"]:
            data.append((d.isoformat()[5:], stats.get('floorsAscended') or 0))
        elif metric in ["battery"]:
            bb = client.get_body_battery(d.isoformat())
            if bb and isinstance(bb, list) and bb:
                data.append((d.isoformat()[5:], bb[0].get('charged', 0) - bb[0].get('drained', 0)))
        else:
            print(f"Unknown metric: {metric}")
            print("Available: steps, hr, calories, stress, floors, battery")
            return 1
    
    # Reverse for chronological order
    data = list(reversed(data))
    
    title = f"📊 {metric.upper()} - Letzte {days} Tage"
    print(ascii_chart(data, title=title))
    return 0


def cmd_download(args):
    """Download FIT/GPX files for an activity."""
    client = get_client()
    if not client:
        return 1
    
    if not args.id:
        # List recent activities with IDs
        activities = client.get_activities(0, 10)
        print("Recent activities:")
        for act in activities:
            print(f"  {act.get('activityId')}: {act.get('activityName')} ({act.get('startTimeLocal', '')[:10]})")
        print("\nUse: python3 garmin.py download --id <activity_id> --format fit")
        return 0
    
    activity_id = args.id
    fmt = args.format or "fit"
    
    try:
        print(f"Downloading activity {activity_id} as {fmt.upper()}...")
        data = client.download_activity(activity_id, fmt=fmt)
        
        filename = f"activity_{activity_id}.{fmt}"
        with open(filename, "wb") as f:
            f.write(data)
        
        print(f"✅ Saved to: {filename}")
        print(f"   Size: {len(data)} bytes")
    except Exception as e:
        print(f"Error downloading activity: {e}")
    
    return 0


def cmd_trends(args):
    """7-day trends with ASCII charts."""
    client = get_client()
    if not client:
        return 1
    
    days = args.days or 7
    
    # Collect data
    steps_data = []
    hr_data = []
    stress_data = []
    
    for i in range(days):
        d = date.today() - timedelta(days=i)
        stats = client.get_stats(d.isoformat())
        steps_data.append((d.isoformat()[5:], stats.get('totalSteps') or 0))
        hr_data.append((d.isoformat()[5:], stats.get('restingHeartRate') or 0))
        stress_data.append((d.isoformat()[5:], stats.get('averageStressLevel') or 0))
    
    # Reverse for chronological order
    steps_data = list(reversed(steps_data))
    hr_data = list(reversed(hr_data))
    stress_data = list(reversed(stress_data))
    
    if args.json:
        output = {
            "steps": steps_data,
            "heart_rate": hr_data,
            "stress": stress_data,
        }
        print(json.dumps(output, indent=2))
        return 0
    
    print(f"\n📈 TRENDS - Letzte {days} Tage\n")
    
    # Steps chart
    print(ascii_chart(steps_data, title="👣 Schritte"))
    
    # HR chart
    print(ascii_chart(hr_data, title="❤️ Ruhe-Herzfrequenz"))
    
    # Stress chart
    print(ascii_chart(stress_data, title="😤 Stress"))
    
    # Summary
    steps = [s for _, s in steps_data if s > 0]
    hrs = [h for _, h in hr_data if h > 0]
    
    print(f"\n📊 Zusammenfassung:")
    print(f"   Schritte: {sum(steps):,} total, Ø {sum(steps)//len(steps):,}/Tag")
    if hrs:
        print(f"   Ruhe-HF: Ø {sum(hrs)/len(hrs):.1f} bpm")
    
    return 0


# [Include all other functions from the original script: cmd_summary, cmd_activities, cmd_week, etc.]
# For brevity, I'm showing only the new/updated functions above

def main():
    parser = argparse.ArgumentParser(
        description="Garmin Connect CLI - Pro Edition",
        formatter_class=argparse.RawDescriptionHelpFormatter
    )
    parser.add_argument("--json", action="store_true", help="Output as JSON")
    parser.add_argument("--full", action="store_true", help="Full detailed output")
    
    subparsers = parser.add_subparsers(dest="command", help="Command")
    
    # Ask (natural language)
    p_ask = subparsers.add_parser("ask", help="Natural language query")
    p_ask.add_argument("query", nargs="*", help="Your question")
    p_ask.add_argument("--question", help="Your question (alternative)")
    
    # Download
    p_download = subparsers.add_parser("download", help="Download FIT/GPX files")
    p_download.add_argument("--id", help="Activity ID (omit to list recent)")
    p_download.add_argument("--format", choices=["fit", "gpx", "tcx"], default="fit", help="File format")
    
    # Body composition
    p_body = subparsers.add_parser("body", help="Body composition data")
    p_body.add_argument("--date", help="Date (YYYY-MM-DD)")
    
    # VO2 max
    p_vo2max = subparsers.add_parser("vo2max", help="VO2 max and cardio fitness")
    
    # Chart
    p_chart = subparsers.add_parser("chart", help="ASCII chart for a metric")
    p_chart.add_argument("metric", help="Metric to chart (steps, hr, calories, stress, floors, battery)")
    p_chart.add_argument("--days", type=int, default=7, help="Number of days")
    
    # [Include all other subparsers from the original script]
    # Summary
    p_summary = subparsers.add_parser("summary", help="Full daily summary")
    p_summary.add_argument("--date", help="Date (YYYY-MM-DD)")
    
    # Week
    p_week = subparsers.add_parser("week", help="Weekly overview")
    p_week.add_argument("--days", type=int, default=7, help="Number of days")
    
    # Trends
    p_trends = subparsers.add_parser("trends", help="7-day trends with charts")
    p_trends.add_argument("--days", type=int, default=7, help="Number of days")
    
    # Activities
    p_activities = subparsers.add_parser("activities", help="Recent activities")
    p_activities.add_argument("--date", help="Date (YYYY-MM-DD)")
    p_activities.add_argument("--limit", type=int, default=10, help="Limit results")
    
    # Login/Logout
    subparsers.add_parser("login", help="Login and save tokens")
    subparsers.add_parser("logout", help="Delete saved tokens")
    
    args = parser.parse_args()
    
    # Handle commands that don't need client
    if args.command == "login":
        from getpass import getpass
        email = input("Email: ")
        password = getpass("Password: ")
        TOKEN_DIR.mkdir(parents=True, exist_ok=True)
        with open(CREDENTIALS_FILE, "w") as f:
            json.dump({"email": email, "password": password}, f)
        os.chmod(CREDENTIALS_FILE, 0o600)
        
        client = Garmin(email=email, password=password)
        client.login(tokenstore=str(TOKEN_DIR))
        print(f"{EMOJI['good']} Login successful!")
        profile = client.get_user_profile()
        if profile:
            print(f"Logged in as: {profile.get('fullName', 'Unknown')}")
        return 0
    
    if args.command == "logout":
        import shutil
        if TOKEN_DIR.exists():
            shutil.rmtree(TOKEN_DIR)
        if CREDENTIALS_FILE.exists():
            CREDENTIALS_FILE.unlink()
        print("Logged out and credentials removed")
        return 0
    
    if not args.command:
        args.command = "summary"
    
    # Map commands to functions (need to include all from original)
    commands = {
        "ask": cmd_ask,
        "download": cmd_download,
        "body": cmd_body,
        "vo2max": cmd_vo2max,
        "chart": cmd_chart,
        "trends": cmd_trends,
        "week": cmd_week,
        # Add all other commands here...
        "summary": lambda a: cmd_summary(a) if (client := get_client()) else 1,
        "activities": lambda a: cmd_activities(a) if (client := get_client()) else 1,
        # ... etc
    }
    
    if args.command in commands:
        return commands[args.command](args)
    
    # Default: run summary
    client = get_client()
    if client:
        return cmd_summary(args)
    
    return 1


if __name__ == "__main__":
    # Import remaining functions from original script
    # This is a simplified version - the full script includes all functions
    sys.exit(main())