#!/usr/bin/env python3
"""gallery-dl wrapper with Telegram/ImgBB integration"""

import argparse
import subprocess
import os
import sys
import json
from pathlib import Path

def run_gallery_dl(args_list):
    """Run gallery-dl command"""
    result = subprocess.run(['gallery-dl'] + args_list, capture_output=True, text=True)
    return result.returncode, result.stdout, result.stderr

def get_config():
    """Get config file path"""
    config_dir = Path(os.path.expanduser('~/.config/gallery-dl'))
    return config_dir / 'config.json'

def main():
    parser = argparse.ArgumentParser(description='gallery-dl wrapper')
    parser.add_argument('url', help='URL to download')
    parser.add_argument('-D', '--directory', default='./gallery-dl-output', help='Download directory')
    parser.add_argument('-f', '--filename', help='Filename format')
    parser.add_argument('--limit', type=int, help='Limit downloads')
    parser.add_argument('--range', help='Download range (e.g., 1-10)')
    parser.add_argument('-v', '--verbose', action='store_true', help='Verbose output')
    parser.add_argument('--telegram', action='store_true', help='Send to Telegram after download')
    parser.add_argument('--imgbb', action='store_true', help='Upload to ImgBB after download')
    parser.add_argument('--config', help='gallery-dl config file')
    
    args = parser.parse_args()
    
    # Build gallery-dl command
    cmd = [args.url, '-D', args.directory]
    
    if args.filename:
        cmd.extend(['-f', args.filename])
    if args.limit:
        cmd.extend(['--limit', str(args.limit)])
    if args.range:
        cmd.extend(['--range', args.range])
    if args.verbose:
        cmd.append('-v')
    if args.config:
        cmd.extend(['--config', args.config])
    
    print(f"📥 Downloading: {args.url}")
    print(f"   To: {args.directory}")
    
    # Run gallery-dl
    code, stdout, stderr = run_gallery_dl(cmd)
    
    if code != 0:
        print(f"❌ Error: {stderr}")
        sys.exit(1)
    
    print(f"✅ Downloaded to {args.directory}")
    
    # Find downloaded files
    folder = Path(args.directory)
    if not folder.exists():
        print("❌ Output folder not found")
        sys.exit(1)
    
    files = list(folder.rglob('*'))
    images = [f for f in files if f.is_file() and f.suffix.lower() in ['.jpg', '.jpeg', '.png', '.gif', '.webp']]
    
    print(f"📁 Found {len(images)} images")
    
    if args.telegram:
        print("📤 Sending to Telegram...")
        # Would integrate with Telegram here
        pass
    
    if args.imgbb:
        print("☁️ Uploading to ImgBB...")
        # Would integrate with ImgBB here
        pass
    
    return 0

if __name__ == '__main__':
    sys.exit(main())
