---
name: fsxmemory
version: 1.3.1
description: Structured memory system for AI agents. Context death resilience (checkpoint/recover), structured storage, Obsidian-compatible markdown, and local semantic search.
author: Foresigxt
repository: https://github.com/Foresigxt/foresigxt-cli-memory
---

# Foresigxt Memory

Structured memory system for AI agents.

## Install

```bash
npm install -g @foresigxt/foresigxt-cli-memory
```

## Setup

### Option 1: Initialize New Vault

```bash
# Initialize vault (creates folder structure + templates)
fsxmemory init ~/memory
```

### Option 2: Use Existing Vault

**For isolated workspace memory** (each workspace has its own vault):

```bash
# Create .env in workspace root
echo 'FSXMEMORY_PATH=/path/to/workspace/memory' > .env

# All agents in THIS workspace use this isolated vault
fsxmemory stats  # Works automatically!
```

**For shared memory across all workspaces**:

```bash
# Set global environment variable (in ~/.bashrc or ~/.zshrc)
export FSXMEMORY_PATH=/path/to/shared/memory

# All agents in ALL workspaces share the same vault
```

**Or**: Use `--vault` flag for one-time override:

```bash
fsxmemory stats --vault /path/to/other/vault
```

## Core Commands

### Store memories by type

```bash
# Types: fact, feeling, decision, lesson, commitment, preference, relationship, project, procedural, semantic, episodic
fsxmemory remember decision "Use Postgres over SQLite" --content "Need concurrent writes for multi-agent setup"
fsxmemory remember lesson "Context death is survivable" --content "Checkpoint before heavy work"
fsxmemory remember relationship "Justin Dukes" --content "Client contact at Hale Pet Door"
fsxmemory remember procedural "Deploy to Production" --content "1. Run tests 2. Build 3. Deploy"
fsxmemory remember semantic "Event Loop Concept" --content "JavaScript's concurrency model..."
fsxmemory remember episodic "First Production Deploy" --content "Deployed v2.0 today, team was nervous but it went well"
```

### Quick capture to inbox

```bash
fsxmemory capture "TODO: Review PR tomorrow"
```

### Search (requires qmd installed)

```bash
# Keyword search (fast)
fsxmemory search "client contacts"

# Semantic search (slower, more accurate)
fsxmemory vsearch "what did we decide about the database"
```

## Context Death Resilience

### Checkpoint (save state frequently)

```bash
fsxmemory checkpoint --working-on "PR review" --focus "type guards" --blocked "waiting for CI"
```

### Recover (check on wake)

```bash
fsxmemory recover --clear
# Shows: death time, last checkpoint, recent handoff
```

### Handoff (before session end)

```bash
fsxmemory handoff \
  --working-on "Foresigxt Memory improvements" \
  --blocked "npm token" \
  --next "publish to npm, create skill" \
  --feeling "productive"
```

### Recap (bootstrap new session)

```bash
fsxmemory recap
# Shows: recent handoffs, active projects, pending commitments, lessons
```

## Migration from Other Formats

Migrate existing vaults from OpenClaw, Obsidian, or other markdown-based systems:

### Analyze First (Dry Run)

```bash
# See what would be changed without modifying files
fsxmemory migrate --from openclaw --vault /path/to/vault --dry-run
```

### Migrate with Backup

```bash
# Recommended: Creates automatic backup before migration
fsxmemory migrate --from openclaw --vault /path/to/vault --backup

# The migration:
# ✅ Adds YAML frontmatter to all markdown files
# ✅ Renames directories (procedural→procedures, semantic→knowledge, episodic→episodes)
# ✅ Creates .fsxmemory.json config file
# ✅ Preserves all content and custom categories
# ✅ Creates timestamped backup for rollback
```

### Rollback if Needed

```bash
# Restore from backup if something went wrong
fsxmemory migrate --rollback --vault /path/to/vault
```

### Migration Options

```bash
# Available source formats
--from openclaw      # OpenClaw vault format
--from obsidian      # Obsidian vault format
--from generic       # Generic markdown vault

# Migration flags
--dry-run           # Preview changes without modifying files
--backup            # Create backup before migration (recommended)
--force             # Skip confirmation prompts
--verbose           # Show detailed progress
--rollback          # Restore from last backup
```

### Example: Migrate OpenClaw Vault

```bash
# 1. Analyze first
fsxmemory migrate --from openclaw --vault ~/.openclaw/workspace/memory --dry-run

# 2. Run migration with backup
fsxmemory migrate --from openclaw --vault ~/.openclaw/workspace/memory --backup --verbose

# 3. Verify migration worked
fsxmemory stats --vault ~/.openclaw/workspace/memory
fsxmemory doctor --vault ~/.openclaw/workspace/memory
```

**Migration Speed**: ~53 files in 0.07 seconds ⚡

## Auto-linking

Wiki-link entity mentions in markdown files:

```bash
# Link all files
fsxmemory link --all

# Link single file
fsxmemory link memory/2024-01-15.md
```

## Templates Reference

Foresigxt Memory includes structured templates for consistent documentation. Location: `templates/` directory.

### Available Templates

| Template | Type | Use For | Sections |
|----------|------|---------|----------|
| `decision.md` | decision | Key choices, architecture decisions | Context, Options, Decision, Outcome |
| `procedure.md` | procedural | How-to guides, workflows, SOPs | Purpose, Prerequisites, Steps, Pitfalls, Verification |
| `knowledge.md` | semantic | Concepts, definitions, mental models | Definition, Key Concepts, Examples, Why It Matters |
| `episode.md` | episodic | Events, experiences, meetings | What Happened, Context, Key Moments, Reflection |
| `person.md` | person | Contacts, relationships | Contact, Role, Working With, Interactions |
| `project.md` | project | Active work, initiatives | Goal, Status, Next Actions, Blockers |
| `lesson.md` | lesson | Insights, patterns learned | Situation, Lesson, Application |
| `handoff.md` | handoff | Session continuity | Working On, Context, Next Steps, Blockers |
| `daily.md` | daily | Daily notes, journal | Focus, Done, Notes |

### Template Usage

Templates are automatically selected by memory type:

```bash
fsxmemory remember decision "Title" --content "..."    # → templates/decision.md
fsxmemory remember procedural "Title" --content "..."  # → templates/procedure.md
fsxmemory remember semantic "Title" --content "..."    # → templates/knowledge.md
fsxmemory remember episodic "Title" --content "..."    # → templates/episode.md
fsxmemory remember relationship "Name" --content "..." # → templates/person.md
fsxmemory remember lesson "Title" --content "..."      # → templates/lesson.md
```

**To view template structure**: Read the template file in `templates/` directory before creating a memory document.

**Template features**:
- YAML frontmatter with metadata (title, date, type, status)
- Structured sections with placeholder guidance
- Wiki-link suggestions for connections
- Auto-generated tags

## Folder Structure

```
vault/
├── .fsxmemory/           # Internal state
│   ├── last-checkpoint.json
│   └── dirty-death.flag
├── decisions/            # Key choices with reasoning
├── lessons/              # Insights and patterns
├── people/               # One file per person
├── projects/             # Active work tracking
├── procedures/           # How-to guides and workflows
├── knowledge/            # Concepts and definitions
├── episodes/             # Personal experiences
├── handoffs/             # Session continuity
├── inbox/                # Quick captures
└── templates/            # Document templates (9 types)
```

## Best Practices

1. **Checkpoint every 10-15 min** during heavy work
2. **Handoff before session end** — future you will thank you
3. **Recover on wake** — check if last session died
4. **Use types** — knowing WHAT you're storing helps WHERE to put it
5. **Wiki-link liberally** — `[[person-name]]` builds your knowledge graph

## Integration with qmd

Foresigxt Memory uses [qmd](https://github.com/tobi/qmd) for search:

```bash
# Install qmd
bun install -g github:tobi/qmd

# Add vault as collection
qmd collection add /path/to/vault --name my-memory --mask "**/*.md"

# Update index
qmd update && qmd embed
```

## Configuration

Foresigxt Memory supports three ways to set the vault path (in order of precedence):

### 1. Command-line flag (highest priority)
```bash
fsxmemory stats --vault /path/to/vault
```

### 2. Environment variable
```bash
export FSXMEMORY_PATH=/path/to/memory
fsxmemory stats
```

### 3. .env file (for workspace-isolated memory)
```bash
# Create .env in workspace root
cat > .env << 'EOF'
FSXMEMORY_PATH=/home/user/.openclaw/workspace/memory
EOF

# All fsxmemory commands in this workspace use this isolated vault
fsxmemory stats
fsxmemory checkpoint --working-on "task"
```

**Use .env when:**
- ✅ **Isolating workspace memory** — Each project has its own separate vault
- ✅ **Per-project configuration** — Different agents in different workspaces use different vaults
- ✅ **Portable** — Workspace agents automatically use the right vault
- ✅ **Git-safe** — Add `.env` to `.gitignore` to protect paths

**Use global export when:**
- ✅ **Sharing memory across workspaces** — All agents everywhere use one vault
- ✅ **Centralized knowledge** — One source of truth for all projects

**Environment Variables:**
- `FSXMEMORY_PATH` — Vault path (can be set in shell or `.env` file)

## Publishing Skill Package

To create a distributable skill package (includes SKILL.md and templates/):

```bash
# Package the skill
npm run package-skill

# Output: dist-skill/fsxmemory-skill.zip (~8KB)
```

**Package contents:**
- `SKILL.md` - Complete documentation and reference
- `templates/` - All 9 memory templates
- `.env.example` - Configuration template
- `INSTALL.md` - Quick setup guide

**Distribution:**
Share the `fsxmemory-skill.zip` file with other agents/teams. They can extract it to get:
- Complete skill documentation
- Ready-to-use templates
- Configuration examples

**For OpenClaw/ClaudeHub:**
The packaged skill is ready for upload to skill repositories.

## Links

- npm: https://www.npmjs.com/package/@foresigxt/foresigxt-cli-memory
- GitHub: https://github.com/Foresigxt/foresigxt-cli-memory
- Issues: https://github.com/Foresigxt/foresigxt-cli-memory/issues
