"""
Extract content from a PowerPoint file for Frontend Slides (Phase 4).
Returns list of slide dicts with title, content, images, notes.
Dependency: pip install python-pptx
"""
from pptx import Presentation
import os


def extract_pptx(file_path, output_dir):
    """
    Extract all content from a PowerPoint file.
    Returns a list of slide dicts: number, title, content[], images[], notes.
    Saves images to output_dir/assets/.
    """
    prs = Presentation(file_path)
    slides_data = []
    assets_dir = os.path.join(output_dir, "assets")
    os.makedirs(assets_dir, exist_ok=True)

    for slide_num, slide in enumerate(prs.slides):
        slide_data = {
            "number": slide_num + 1,
            "title": "",
            "content": [],
            "images": [],
            "notes": "",
        }

        title_shape = getattr(slide.shapes, "title", None)
        for shape in slide.shapes:
            if shape.has_text_frame:
                if title_shape is not None and shape == title_shape:
                    slide_data["title"] = shape.text
                else:
                    slide_data["content"].append({"type": "text", "content": shape.text})

            if shape.shape_type == 13:  # Picture
                image = shape.image
                image_bytes = image.blob
                image_ext = image.ext
                image_name = f"slide{slide_num + 1}_img{len(slide_data['images']) + 1}.{image_ext}"
                image_path = os.path.join(assets_dir, image_name)

                with open(image_path, "wb") as f:
                    f.write(image_bytes)

                slide_data["images"].append({
                    "path": f"assets/{image_name}",
                    "width": shape.width,
                    "height": shape.height,
                })

        if slide.has_notes_slide:
            notes_frame = slide.notes_slide.notes_text_frame
            slide_data["notes"] = notes_frame.text

        slides_data.append(slide_data)

    return slides_data
