---
name: freedcamp
description: "Manage Freedcamp tasks, projects, groups, comments, notifications, and task lists via HMAC-SHA1 API credentials."
homepage: https://freedcamp.com
user-invocable: true
metadata: {"openclaw":{"requires":{"env":["FREEDCAMP_API_KEY","FREEDCAMP_API_SECRET"]},"primaryEnv":"FREEDCAMP_API_KEY","homepage":"https://freedcamp.com"}}
---

# Freedcamp

This skill provides a dependency-free Node.js CLI that calls the Freedcamp REST API (v1) using **HMAC-SHA1 secured credentials** (API Key + API Secret).

- Script: `{baseDir}/scripts/freedcamp.mjs`
- Auth: `FREEDCAMP_API_KEY` + `FREEDCAMP_API_SECRET`
- Output: **JSON only** (stdout), suitable for agents and automation

## Setup

1. Obtain your Freedcamp API key and secret from your Freedcamp account settings.
2. Provide both values as environment variables.

### Common injection patterns

- Shell env (local testing):

  ```
  export FREEDCAMP_API_KEY="..."
  export FREEDCAMP_API_SECRET="..."
  ```

- OpenClaw config (recommended): set `skills.entries.freedcamp.apiKey` and `skills.entries.freedcamp.env.FREEDCAMP_API_SECRET` so secrets are injected only for the agent run.

### Configure via OpenClaw CLI (recommended)

```bash
openclaw config set skills.entries.freedcamp.enabled true
openclaw config set skills.entries.freedcamp.apiKey "YOUR_API_KEY"
openclaw config set skills.entries.freedcamp.env.FREEDCAMP_API_SECRET "YOUR_API_SECRET"
```

**Verify what is stored:**

```bash
openclaw config get skills.entries.freedcamp
```

**Remove stored credentials:**

```bash
openclaw config unset skills.entries.freedcamp.apiKey
openclaw config unset skills.entries.freedcamp.env.FREEDCAMP_API_SECRET
```

## First calls (sanity + discovery)

- Who am I / session info:

  `node {baseDir}/scripts/freedcamp.mjs me`

- List all groups, projects, and apps:

  `node {baseDir}/scripts/freedcamp.mjs groups-projects`

## ID resolution

When the user provides project names, resolve to IDs using:

- `groups-projects` returns all groups with their projects, including project IDs and names
- Use the exact `project_name` from the output for other commands

Avoid guessing a project ID when multiple matches exist.

## Core: tasks

### List tasks in a project

`node {baseDir}/scripts/freedcamp.mjs tasks --project <project_id> --all`

### List tasks with filters

`node {baseDir}/scripts/freedcamp.mjs tasks --project <project_id> --status in_progress,not_started --assigned_to 2,-1`

Useful filters:
- `--status` comma-separated: `not_started`, `completed`, `in_progress`, `invalid`, `review`
- `--assigned_to` comma-separated user IDs. `0` = unassigned, `-1` = everyone
- `--due_from YYYY-MM-DD` / `--due_to YYYY-MM-DD`
- `--created_from YYYY-MM-DD` / `--created_to YYYY-MM-DD`
- `--list_status active|archived|all`
- `--with_archived true` to include tasks from archived projects
- `--limit <n>` (max 200 per page, default 200)
- `--offset <n>` for pagination

### Get a single task (with comments and files)

`node {baseDir}/scripts/freedcamp.mjs task <task_id>`

### Create a task

`node {baseDir}/scripts/freedcamp.mjs create-task --project <project_id> --title "Task title"`

With optional description and task list:

`node {baseDir}/scripts/freedcamp.mjs create-task --project <project_id> --title "Task title" --description "Details here" --task_group <task_group_id>`

### Update a task

`node {baseDir}/scripts/freedcamp.mjs update-task <task_id> --title "New title" --status in_progress`

Status values: `not_started` (0), `completed` (1), `in_progress` (2), `invalid` (3), `review` (4)

### Create a task by project name

`node {baseDir}/scripts/freedcamp.mjs create-task-by-name --project_name "My Project" --app_name "Tasks" --title "New task"`

Resolves the project name to an ID using session data. Currently supports the Tasks app.

## Task lists (groups)

- List task lists for a project:

  `node {baseDir}/scripts/freedcamp.mjs task-lists --project <project_id>`

- Specify app (default is Tasks / app_id 2):

  `node {baseDir}/scripts/freedcamp.mjs task-lists --project <project_id> --app_id 2`

## Comments

- Add a comment to any item:

  `node {baseDir}/scripts/freedcamp.mjs comment <item_id> --app_name "Tasks" --text "My comment"`

Comments are automatically wrapped in `<p>` tags. You can also pass raw HTML:

  `node {baseDir}/scripts/freedcamp.mjs comment <item_id> --app_name "Tasks" --html "<p>Bold <b>text</b></p>"`

### App names for comments

When adding comments, the `--app_name` must be one of:
Tasks, Discussions, Milestones, Time, Files, Issue Tracker, Wikis, CRM, Passwords, Calendar, Planner, Translations

## Notifications

- Fetch recent notifications (last 60 days):

  `node {baseDir}/scripts/freedcamp.mjs notifications`

- Mark a notification as read:

  `node {baseDir}/scripts/freedcamp.mjs mark-read <notification_uid>`

## Data model reference

### Task statuses

| Name | Value | CLI flag |
|---|---|---|
| Not Started | 0 | `not_started` |
| Completed | 1 | `completed` |
| In Progress | 2 | `in_progress` |
| Invalid | 3 | `invalid` |
| Review | 4 | `review` |

### Priorities

| Name | Value |
|---|---|
| None | 0 |
| Low | 1 |
| Medium | 2 |
| High | 3 |

### App types

| ID | Name | Key |
|---|---|---|
| 2 | Tasks | TODOS |
| 3 | Discussions | DISCUSSIONS |
| 4 | Milestones | MILESTONES |
| 5 | Time | TIME |
| 6 | Files | FILES |
| 13 | Issue Tracker | BUGTRACKER |
| 14 | Wikis | WIKI |
| 16 | CRM | CRM |
| 17 | Passwords | PASSMAN |
| 19 | Calendar | CALENDAR |
| 47 | Planner | PLANNER |
| 48 | Translations | TRANSLATIONS |

## Important notes

- Comments must contain HTML. Plain text passed via `--text` is auto-wrapped in `<p>` tags.
- Task pagination max is 200 per request; use `--offset` for more.
- Session is cached locally and auto-refreshes on 401 errors.
- The `--all` flag on `tasks` auto-paginates to fetch every result.

## Out of scope

- Invoices and Invoices Plus APIs are not exposed.
- "Bot personality" is not embedded; configure behavior in your agent prompt.
