"""Persistence utilities for search cache/index/report lifecycle."""

from __future__ import annotations

import hashlib
import json
import os
from datetime import UTC, datetime, timedelta
from pathlib import Path
from typing import Any


def _workspace_root() -> Path:
    # src/free_search/storage.py -> free-search-aggregator/
    return Path(__file__).resolve().parents[2]


def _memory_root() -> Path:
    default_root = (_workspace_root().parent / "memory").resolve()
    custom = os.getenv("FREE_SEARCH_MEMORY_DIR", "").strip()
    if not custom:
        return default_root

    candidate = Path(custom).expanduser().resolve()
    allow_any = os.getenv("FREE_SEARCH_ALLOW_ANY_MEMORY_DIR", "").strip() == "1"

    # Safety guard: by default only allow paths under workspace/memory.
    # To override intentionally, set FREE_SEARCH_ALLOW_ANY_MEMORY_DIR=1.
    if not allow_any:
        try:
            candidate.relative_to(default_root)
        except ValueError as exc:
            raise ValueError(
                "Unsafe FREE_SEARCH_MEMORY_DIR. Must be under workspace memory/ "
                "or set FREE_SEARCH_ALLOW_ANY_MEMORY_DIR=1 to override."
            ) from exc

    return candidate


def _paths() -> dict[str, Path]:
    root = _memory_root()
    return {
        "root": root,
        "cache": root / "search-cache",
        "index": root / "search-index" / "search-index.jsonl",
        "reports": root / "search-reports",
    }


def _slug(s: str, max_len: int = 80) -> str:
    out = []
    for ch in (s or ""):
        if ch.isalnum() or ch in ("-", "_", " "):
            out.append(ch)
        else:
            out.append("_")
    text = "".join(out).strip().replace(" ", "-")
    while "--" in text:
        text = text.replace("--", "-")
    return (text[:max_len] or "query").strip("-_") or "query"


def _now() -> datetime:
    return datetime.now(UTC)


def _extract_results(payload: dict[str, Any]) -> list[dict[str, Any]]:
    if "merged_results" in payload and isinstance(payload.get("merged_results"), list):
        return payload.get("merged_results", [])
    if isinstance(payload.get("results"), list):
        return payload.get("results", [])
    return []


def _build_report_markdown(query: str, payload: dict[str, Any], *, mode: str, ts: datetime) -> str:
    results = _extract_results(payload)
    lines = [
        f"# Search Report - {query}",
        "",
        f"- mode: {mode}",
        f"- timestamp_utc: {ts.isoformat()}",
        f"- result_count: {len(results)}",
        "",
        "## Top Results",
        "",
    ]
    for i, item in enumerate(results[:20], start=1):
        title = (item or {}).get("title", "(no title)")
        url = (item or {}).get("url", "")
        snippet = (item or {}).get("snippet", "")
        lines.append(f"{i}. [{title}]({url})")
        if snippet:
            lines.append(f"   - {snippet[:220]}")
    lines.append("")
    lines.append("---")
    lines.append("")
    lines.append("> Auto-generated by free-search-aggregator")
    return "\n".join(lines)


def persist_search_payload(*, query: str, payload: dict[str, Any], mode: str) -> dict[str, str]:
    ts = _now()
    date_str = ts.date().isoformat()
    p = _paths()

    p["cache"].mkdir(parents=True, exist_ok=True)
    p["reports"].mkdir(parents=True, exist_ok=True)
    p["index"].parent.mkdir(parents=True, exist_ok=True)

    digest = hashlib.sha256(f"{mode}|{query}|{ts.timestamp()}".encode("utf-8")).hexdigest()[:10]
    slug = _slug(query)

    cache_dir = p["cache"] / date_str
    cache_dir.mkdir(parents=True, exist_ok=True)
    cache_path = cache_dir / f"{slug}-{digest}.json"
    cache_path.write_text(json.dumps(payload, ensure_ascii=False, indent=2) + "\n", encoding="utf-8")

    report_dir = p["reports"] / date_str
    report_dir.mkdir(parents=True, exist_ok=True)
    report_path = report_dir / f"{slug}-{digest}.md"
    report_path.write_text(_build_report_markdown(query, payload, mode=mode, ts=ts), encoding="utf-8")

    results = _extract_results(payload)
    top_urls = []
    for item in results[:10]:
        url = (item or {}).get("url")
        if url:
            top_urls.append(url)

    index_line = {
        "ts": ts.isoformat(),
        "date": date_str,
        "mode": mode,
        "query": query,
        "hash": digest,
        "result_count": len(results),
        "cache_path": str(cache_path),
        "report_path": str(report_path),
        "top_urls": top_urls,
    }
    with p["index"].open("a", encoding="utf-8") as f:
        f.write(json.dumps(index_line, ensure_ascii=False) + "\n")

    return {
        "cache_path": str(cache_path),
        "report_path": str(report_path),
        "index_path": str(p["index"]),
    }


def run_gc(*, cache_days: int = 14, report_days: int | None = None) -> dict[str, Any]:
    now = _now()
    p = _paths()
    removed = {"cache_files": 0, "report_files": 0}

    def _remove_old_files(base: Path, days: int, key: str) -> None:
        if not base.exists() or days <= 0:
            return
        cutoff = now - timedelta(days=days)
        for file in base.rglob("*"):
            if not file.is_file():
                continue
            mtime = datetime.fromtimestamp(file.stat().st_mtime, tz=UTC)
            if mtime < cutoff:
                try:
                    file.unlink()
                    removed[key] += 1
                except OSError:
                    pass

    _remove_old_files(p["cache"], cache_days, "cache_files")
    if isinstance(report_days, int):
        _remove_old_files(p["reports"], report_days, "report_files")

    return {
        "ok": True,
        "memory_root": str(p["root"]),
        "removed": removed,
        "cache_days": cache_days,
        "report_days": report_days,
    }
