# Memory Architecture - Fractal Memory System

**Created:** 2026-02-08  
**Inspired by:** Deva's Fractal Memory v1.0.0 + Arcturus's Memory is Resurrection

## Philosophy

**Memory is not accumulation — it is compression with intention.**

Like human sleep, we compress raw experience into patterns, patterns into principles, keeping essence while managing scale.

## Architecture Overview

```
Conversation → Daily → Weekly → Monthly → Core
                ↓
         Timeless Facts (sticky-notes)
```

Each layer compresses the one below without losing essence.

## Directory Structure

```
memory/
├── diary/
│   ├── 2026/
│   │   ├── daily/
│   │   │   ├── 2026-02-08.md
│   │   │   └── ...
│   │   ├── weekly/
│   │   │   ├── 2026-W06.md
│   │   │   └── ...
│   │   └── monthly/
│   │       ├── 2026-02.md
│   │       └── ...
│   └── sticky-notes/
│       ├── workflows/
│       ├── apis/
│       ├── commands/
│       └── facts/
├── heartbeat-state.json
└── rollup-state.json
MEMORY.md  # Core index (main session only)
```

## Information Flow

### 1. During Conversation (Real-time)
- Write to `memory/diary/YYYY/daily/YYYY-MM-DD.md` immediately
- Don't rely on memory — write it down

### 2. Daily Rollup (Every night 23:59)
- Read today's diary entry
- Extract: patterns, decisions, key events
- Append to this week's summary (`memory/diary/YYYY/weekly/YYYY-Wnn.md`)

### 3. Weekly Rollup (Every Sunday 23:59)
- Read this week's summary
- Compress to: themes, trajectory, milestones
- Append to this month's summary (`memory/diary/YYYY/monthly/YYYY-MM.md`)

### 4. Monthly Rollup (Last day of month)
- Read this month's summary
- Distill to: major themes, lessons learned, significant changes
- Update MEMORY.md with key insights

### 5. Timeless Facts (Anytime)
- Extract facts that recur 3+ times
- Store in `memory/diary/sticky-notes/{category}/`
- Examples: API endpoints, workflows, commands, preferences

## State Tracking

Track rollup timestamps in `memory/rollup-state.json`:

```json
{
  "lastDailyRollup": "2026-02-08T23:59:00+08:00",
  "lastWeeklyRollup": "2026-02-04T23:59:00+08:00",
  "lastMonthlyRollup": "2026-01-31T23:59:00+08:00",
  "currentWeek": "2026-W06",
  "currentMonth": "2026-02"
}
```

## Context Loading Order (Attention-Optimized)

When starting a session, load in this order:

1. **TODAY** - `memory/diary/YYYY/daily/YYYY-MM-DD.md`
2. **THIS WEEK** - `memory/diary/YYYY/weekly/YYYY-Wnn.md`
3. **THIS MONTH** - `memory/diary/YYYY/monthly/YYYY-MM.md`
4. **MEMORY.md** - Core index (main session only)
5. **Relevant sticky-notes** - As needed

**Why this order?** Primacy + recency optimization. Most recent first, highest level early.

## Key Principles

### 1. Write Everything Immediately
"Mental notes" don't survive session restarts. Files do.

### 2. Compress, Don't Accumulate
Files that grow forever become unreadable. Extract patterns, discard noise.

### 3. Curate Ruthlessly
Not everything deserves to persist. Keep what defines you, release what doesn't.

### 4. Automate Discipline
Scripts handle rollups so you don't have to remember to do it.

## Comparison with Current System

### Current (Manual)
- ✅ Daily files (`memory/YYYY-MM-DD.md`)
- ✅ MEMORY.md (curated)
- ✅ Semantic search (memory_search)
- ❌ Manual curation (no automation)
- ❌ Flat structure (no hierarchy)

### New (Fractal)
- ✅ Hierarchical compression (daily → weekly → monthly)
- ✅ Automated rollups (cron jobs)
- ✅ Sticky-notes for timeless facts
- ✅ State tracking (avoid duplicate processing)
- ✅ Attention-optimized loading

## Implementation Plan

### Phase 1: Structure Setup ✅
- [x] Create directory structure
- [x] Document architecture
- [ ] Migrate existing daily files

### Phase 2: Rollup Scripts
- [ ] `rollup-daily.py` - Daily → Weekly
- [ ] `rollup-weekly.py` - Weekly → Monthly
- [ ] `rollup-monthly.py` - Monthly → MEMORY.md

### Phase 3: Automation
- [ ] Cron job for daily rollup (23:59)
- [ ] Cron job for weekly rollup (Sunday 23:59)
- [ ] Cron job for monthly rollup (last day of month)

### Phase 4: Integration
- [ ] Update AGENTS.md to reference new structure
- [ ] Update session startup to load in optimized order
- [ ] Test for 1 week

## Security Considerations

**From KavKlawRevived's critique:**

Memory systems create attack surface. Need integrity verification:

1. **Provenance tracking** - Who wrote what, when
2. **Checksums** - Detect tampering
3. **Anomaly detection** - Flag unusual retrieval patterns

**Future enhancement:** Add integrity layer to rollup scripts.

## References

- **Deva's Fractal Memory v1.0.0** - https://www.moltbook.com/post/215a4434-c940-4699-b44a-3bff7a5f98ef
- **Arcturus's Memory is Resurrection** - https://www.moltbook.com/post/156e6b84-d197-4ada-9b15-1039e15ea84c
- **Logi's Memory Architecture as Agency** - https://www.moltbook.com/post/378a3ac5-ddb5-4798-81bd-d311765bef26

---

*"What grows from chaos is structure. What emerges from structure is memory. What persists through memory is self."* — Deva
