# Payout — CLI Reference

## Overview

Payout lets the agent send USDC to any wallet address on Base network.

**Important**: Unlike x402 payments which can use pre-approved mandates for autonomous spending, every payout requires **individual user authorization** via the FluxA Wallet UI before the onchain transaction executes. There is no mandate-based autonomy for payouts.

## End-to-End Flow

```
1. Agent runs `payout` with recipient, amount, and unique payout_id
2. CLI returns status "pending_authorization" + approvalUrl
3. User opens approvalUrl to authorize
4. Agent polls `payout-status` until status is "succeeded"
```

## Command Reference

### Create Payout

```bash
node scripts/fluxa-cli.bundle.js payout \
  --to "0x4eb5b229d43c30fc629d92bf7ed415d6d7f0cabe" \
  --amount "1000000" \
  --id "reward_20260204_001"
```

**Options:**

| Option | Required | Default | Description |
|--------|----------|---------|-------------|
| `--to` | Yes | — | Recipient wallet address (0x + 40 hex chars) |
| `--amount` | Yes | — | Amount in atomic units (1 USDC = `1000000`) |
| `--id` | Yes | — | Unique payout ID (idempotency key) |
| `--network` | No | `base` | Network name |
| `--asset` | No | USDC address | Token contract address |

**Output:**

```json
{
  "success": true,
  "data": {
    "payoutId": "reward_20260204_001",
    "status": "pending_authorization",
    "txHash": null,
    "approvalUrl": "https://wallet.fluxapay.xyz/authorize-payout/reward_20260204_001",
    "expiresAt": 1738713600
  }
}
```

**Opening the approval URL** (see [SKILL.md](SKILL.md) — "Opening Authorization URLs"):

1. Ask the user using `AskUserQuestion`:
   - Question: "I need to open the approval URL to authorize this payout."
   - Options: ["Yes, open the link", "No, show me the URL"]

2. If YES: Run `open "<approvalUrl>"` to open in their browser

3. Wait for user to confirm they've approved, then poll status in Step 2.

### Query Payout Status

```bash
node scripts/fluxa-cli.bundle.js payout-status --id "reward_20260204_001"
```

**Output (completed):**

```json
{
  "success": true,
  "data": {
    "payoutId": "reward_20260204_001",
    "status": "succeeded",
    "txHash": "0xabcdef1234567890..."
  }
}
```

### Payout Status Values

| Status | Meaning |
|--------|---------|
| `pending_authorization` | Waiting for user approval |
| `processing` | Approved, onchain tx in progress |
| `succeeded` | Done, `txHash` available |
| `failed` | Transaction failed |
| `expired` | User didn't approve in time |

## Scripted Example

```bash
#!/bin/bash
CLI="node scripts/fluxa-cli.bundle.js"
RECIPIENT="0x4eb5b229d43c30fc629d92bf7ed415d6d7f0cabe"
AMOUNT="1000000"
PAYOUT_ID="payout_$(date +%s)"

# Create payout
RESULT=$($CLI payout --to "$RECIPIENT" --amount "$AMOUNT" --id "$PAYOUT_ID")

if echo "$RESULT" | jq -e '.success' > /dev/null 2>&1; then
  APPROVAL_URL=$(echo "$RESULT" | jq -r '.data.approvalUrl')
  echo "Please approve at: $APPROVAL_URL"

  # Poll for completion
  while true; do
    STATUS=$($CLI payout-status --id "$PAYOUT_ID" | jq -r '.data.status')
    echo "Status: $STATUS"
    [ "$STATUS" = "succeeded" ] || [ "$STATUS" = "failed" ] && break
    sleep 5
  done
else
  echo "Error: $(echo "$RESULT" | jq -r '.error')"
fi
```

## Important Notes

- **Idempotency**: Same `payout_id` returns existing status, not a duplicate.
- **Validate addresses**: Must match `0x[a-fA-F0-9]{40}`.
- **No rollback**: Once succeeded onchain, payouts cannot be reversed.
- **Amount**: Always atomic units. 1 USDC = `1000000`, 0.01 USDC = `10000`.
