"""Google Flights search wrapper using fast-flights."""

from dataclasses import dataclass
from typing import Optional


@dataclass
class Flight:
    """Represents a single flight option."""

    airline: str
    departure_time: str
    arrival_time: str
    duration: str
    stops: int
    stop_info: Optional[str]
    price: Optional[int]
    is_best: bool
    arrival_ahead: Optional[str]  # e.g., "+1" for next day arrival

    def to_dict(self) -> dict:
        return {
            "airline": self.airline,
            "departure_time": self.departure_time,
            "arrival_time": self.arrival_time,
            "duration": self.duration,
            "stops": self.stops,
            "stop_info": self.stop_info,
            "price": self.price,
            "is_best": self.is_best,
            "arrival_ahead": self.arrival_ahead,
        }


@dataclass
class SearchResult:
    """Results from a flight search."""

    flights: list[Flight]
    current_price: Optional[str]  # "low", "typical", or "high"
    origin: str
    destination: str
    date: str
    return_date: Optional[str]

    def to_dict(self) -> dict:
        return {
            "origin": self.origin,
            "destination": self.destination,
            "date": self.date,
            "return_date": self.return_date,
            "current_price": self.current_price,
            "flights": [f.to_dict() for f in self.flights],
        }


def search_flights(
    origin: str,
    destination: str,
    date: str,
    return_date: Optional[str] = None,
    adults: int = 1,
    children: int = 0,
    seat_class: str = "economy",
    max_results: int = 20,
) -> SearchResult:
    """
    Search for flights on Google Flights.

    Args:
        origin: Origin airport code (e.g., "DEN", "LAX")
        destination: Destination airport code
        date: Departure date (YYYY-MM-DD)
        return_date: Return date for round trips (YYYY-MM-DD)
        adults: Number of adult passengers
        children: Number of child passengers
        seat_class: Seat class (economy, premium-economy, business, first)
        max_results: Maximum number of results to return

    Returns:
        SearchResult with flight options
    """
    from fast_flights import FlightData, Passengers, get_flights

    # Map seat class names
    seat_map = {
        "economy": "economy",
        "premium-economy": "premium-economy",
        "business": "business",
        "first": "first",
    }
    seat = seat_map.get(seat_class.lower(), "economy")

    # Build flight data
    flight_data = [FlightData(date=date, from_airport=origin, to_airport=destination)]

    # Add return flight for round trips
    trip = "one-way"
    if return_date:
        flight_data.append(FlightData(date=return_date, from_airport=destination, to_airport=origin))
        trip = "round-trip"

    # Build passengers
    passengers = Passengers(
        adults=adults,
        children=children,
        infants_in_seat=0,
        infants_on_lap=0,
    )

    # Perform search
    # Use "common" mode (direct scraping) - "fallback" requires a token
    result = get_flights(
        flight_data=flight_data,
        trip=trip,
        seat=seat,
        passengers=passengers,
        fetch_mode="common",
    )

    # Convert results
    flights = []
    for flight in result.flights[:max_results]:
        # Parse price - it might be a string like "$123" or an int
        price = None
        if flight.price:
            price_str = str(flight.price)
            # Remove currency symbols and commas
            price_clean = price_str.replace("$", "").replace(",", "").strip()
            try:
                price = int(price_clean)
            except ValueError:
                pass

        flights.append(Flight(
            airline=flight.name or "Unknown",
            departure_time=flight.departure or "",
            arrival_time=flight.arrival or "",
            duration=flight.duration or "",
            stops=flight.stops if isinstance(flight.stops, int) else 0,
            stop_info=str(flight.stops) if flight.stops and not isinstance(flight.stops, int) else None,
            price=price,
            is_best=flight.is_best or False,
            arrival_ahead=flight.arrival_time_ahead,
        ))

    return SearchResult(
        flights=flights,
        current_price=result.current_price,
        origin=origin,
        destination=destination,
        date=date,
        return_date=return_date,
    )
