#!/usr/bin/env python3
"""
Notify when new compliance violations/partials appear between runs.

This script compares the current compliance summary (as generated by
`compliance_dashboard.py render --output-summary ...`) against a persisted
snapshot and emits a message when new items appear.

If OPENCLAW_VIOLATION_NOTIFY_CMD is set, the message is sent to that command via
stdin (shell=True). Otherwise the message is printed to stdout.
"""

from __future__ import annotations

import argparse
import json
import os
import subprocess
import sys
import time
from typing import Any, Dict, List, Tuple


def _default_state_file() -> str:
    base = os.path.join(os.path.expanduser("~"), ".openclaw", "security")
    return os.path.join(base, "violation-notify-state.json")


def _load_json(path: str) -> Dict[str, Any]:
    with open(path, "r", encoding="utf-8") as f:
        return json.load(f)


def _safe_mkdir_for(path: str) -> None:
    d = os.path.dirname(path)
    if d:
        os.makedirs(d, exist_ok=True)


def _extract_violation_state(summary: Dict[str, Any]) -> Dict[str, Dict[str, Any]]:
    """
    Return a mapping: check_id -> {status, risk, title}
    Only includes items whose status is violation or partial.
    """
    out: Dict[str, Dict[str, Any]] = {}
    for item in summary.get("violations", []) or []:
        try:
            check_id = str(item.get("check_id", "")).strip()
        except Exception:
            check_id = ""
        if not check_id:
            continue
        status = str(item.get("status", "")).strip() or "violation"
        risk = str(item.get("risk", "")).strip() or "unknown"
        title = str(item.get("title", "")).strip()
        out[check_id] = {"status": status, "risk": risk, "title": title}
    return out


def _severity_rank(status: str) -> int:
    # Higher = more severe.
    if status == "violation":
        return 2
    if status == "partial":
        return 1
    return 0


def _diff_new(prev: Dict[str, Dict[str, Any]], cur: Dict[str, Dict[str, Any]]) -> List[Tuple[str, Dict[str, Any]]]:
    new_items: List[Tuple[str, Dict[str, Any]]] = []
    for check_id, meta in cur.items():
        if check_id not in prev:
            new_items.append((check_id, meta))
            continue
        prev_status = str(prev.get(check_id, {}).get("status", ""))
        cur_status = str(meta.get("status", ""))
        if _severity_rank(cur_status) > _severity_rank(prev_status):
            # Escalation (e.g., partial -> violation) counts as "new".
            new_items.append((check_id, meta))
    return sorted(new_items, key=lambda x: x[0])


def _format_message(new_items: List[Tuple[str, Dict[str, Any]]], summary_file: str) -> str:
    lines = []
    lines.append("OpenClaw Cyber Security Engineer: new compliance findings detected")
    lines.append(f"Source: {summary_file}")
    lines.append(f"Time (UTC): {time.strftime('%Y-%m-%dT%H:%M:%SZ', time.gmtime())}")
    lines.append("")
    for check_id, meta in new_items:
        status = meta.get("status", "")
        risk = meta.get("risk", "")
        title = meta.get("title", "")
        if title:
            lines.append(f"- {check_id}: {status} ({risk}) - {title}")
        else:
            lines.append(f"- {check_id}: {status} ({risk})")
    return "\n".join(lines) + "\n"


def _send_notification(message: str) -> int:
    cmd = os.environ.get("OPENCLAW_VIOLATION_NOTIFY_CMD", "").strip()
    if not cmd:
        sys.stdout.write(message)
        return 0

    # Best-effort: never crash the cycle due to a notifier.
    try:
        p = subprocess.run(cmd, shell=True, input=message, text=True)
        return int(p.returncode)
    except Exception:
        return 1


def main() -> int:
    ap = argparse.ArgumentParser()
    ap.add_argument(
        "--summary-file",
        default="cyber-security-engineer/assessments/compliance-summary.json",
        help="Path to compliance summary JSON.",
    )
    ap.add_argument(
        "--state-file",
        default=os.environ.get("OPENCLAW_VIOLATION_NOTIFY_STATE", _default_state_file()),
        help="Path to persisted state JSON (default: ~/.openclaw/security/violation-notify-state.json).",
    )
    args = ap.parse_args()

    try:
        summary = _load_json(args.summary_file)
    except FileNotFoundError:
        # Nothing to do yet.
        return 0
    except Exception as e:
        sys.stderr.write(f"notify_on_violation: failed to read summary: {e}\n")
        return 0

    cur_state = _extract_violation_state(summary)

    prev_state: Dict[str, Dict[str, Any]] = {}
    try:
        prev_obj = _load_json(args.state_file)
        prev_state = prev_obj.get("violations", {}) or {}
    except FileNotFoundError:
        prev_state = {}
    except Exception:
        prev_state = {}

    new_items = _diff_new(prev_state, cur_state)
    if new_items:
        msg = _format_message(new_items, args.summary_file)
        _send_notification(msg)

    # Persist current state for next run.
    try:
        _safe_mkdir_for(args.state_file)
        with open(args.state_file, "w", encoding="utf-8") as f:
            json.dump({"violations": cur_state}, f, indent=2, sort_keys=True)
            f.write("\n")
    except Exception:
        # Best-effort.
        pass

    return 0


if __name__ == "__main__":
    raise SystemExit(main())

