---
name: fireflies
description: Access Fireflies.ai meeting transcripts, summaries, action items, and analytics via GraphQL API
metadata: {"clawdbot":{"secrets":["FIREFLIES_API_KEY"]}}
---

# Fireflies.ai Skill

Query meeting transcripts, summaries, action items, and analytics from Fireflies.ai.

## Setup

Set your Fireflies API key:
```bash
FIREFLIES_API_KEY=your_api_key_here
```

Get your API key from: https://app.fireflies.ai/integrations (scroll to Fireflies API section)

## API Base

GraphQL Endpoint: `https://api.fireflies.ai/graphql`

Authorization header: `Bearer $FIREFLIES_API_KEY`

---

## Core Queries

### Get Current User

```bash
curl -s -X POST https://api.fireflies.ai/graphql \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer $FIREFLIES_API_KEY" \
  -d '{"query":"{ user { user_id name email is_admin minutes_consumed num_transcripts recent_meeting } }"}' | jq
```

### Get Single Transcript

```bash
curl -s -X POST https://api.fireflies.ai/graphql \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer $FIREFLIES_API_KEY" \
  -d '{"query":"query($id:String!){transcript(id:$id){id title date duration participants fireflies_users summary{keywords action_items overview topics_discussed} speakers{name duration} sentences{speaker_name text start_time}}}","variables":{"id":"TRANSCRIPT_ID"}}' | jq
```

### Search Transcripts by Date Range

```bash
# ISO 8601 format: YYYY-MM-DDTHH:mm:ss.sssZ
curl -s -X POST https://api.fireflies.ai/graphql \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer $FIREFLIES_API_KEY" \
  -d '{"query":"query($from:DateTime,$to:DateTime,$limit:Int){transcripts(fromDate:$from,toDate:$to,limit:$limit){id title date duration organizer_email participants summary{keywords action_items overview}}}","variables":{"from":"2024-01-01T00:00:00.000Z","to":"2024-01-31T23:59:59.999Z","limit":50}}' | jq
```

### Search Transcripts by Participant

```bash
# Search meetings where specific people participated
curl -s -X POST https://api.fireflies.ai/graphql \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer $FIREFLIES_API_KEY" \
  -d '{"query":"query($participants:[String],$limit:Int){transcripts(participants:$participants,limit:$limit){id title date participants organizer_email summary{action_items}}}","variables":{"participants":["john@example.com","jane@example.com"],"limit":20}}' | jq
```

### Search Transcripts by Organizer

```bash
curl -s -X POST https://api.fireflies.ai/graphql \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer $FIREFLIES_API_KEY" \
  -d '{"query":"query($organizers:[String],$limit:Int){transcripts(organizers:$organizers,limit:$limit){id title date organizer_email participants}}","variables":{"organizers":["sales@example.com"],"limit":25}}' | jq
```

### Search by Keyword (Title and/or Transcript)

```bash
# scope: "TITLE", "SENTENCES", or "ALL"
curl -s -X POST https://api.fireflies.ai/graphql \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer $FIREFLIES_API_KEY" \
  -d '{"query":"query($keyword:String,$scope:String){transcripts(keyword:$keyword,scope:$scope,limit:10){id title date summary{overview}}}","variables":{"keyword":"pricing","scope":"ALL"}}' | jq
```

### Get My Recent Transcripts

```bash
curl -s -X POST https://api.fireflies.ai/graphql \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer $FIREFLIES_API_KEY" \
  -d '{"query":"{ transcripts(mine:true,limit:10) { id title date duration summary { action_items keywords } } }"}' | jq
```

---

## Advanced Queries

### Get Full Transcript with Summary & Action Items

```bash
curl -s -X POST https://api.fireflies.ai/graphql \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer $FIREFLIES_API_KEY" \
  -d '{"query":"query($id:String!){transcript(id:$id){id title date duration organizer_email participants fireflies_users workspace_users meeting_attendees{displayName email} summary{keywords action_items outline overview bullet_gist topics_discussed meeting_type} speakers{name duration word_count} sentences{speaker_name text start_time end_time}}}","variables":{"id":"TRANSCRIPT_ID"}}' | jq
```

### Get Transcript with Analytics

```bash
# Requires Pro plan or higher
curl -s -X POST https://api.fireflies.ai/graphql \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer $FIREFLIES_API_KEY" \
  -d '{"query":"query($id:String!){transcript(id:$id){id title analytics{sentiments{positive_pct neutral_pct negative_pct} speakers{name duration word_count filler_words questions longest_monologue words_per_minute}}}}","variables":{"id":"TRANSCRIPT_ID"}}' | jq
```

### Get Contacts

```bash
curl -s -X POST https://api.fireflies.ai/graphql \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer $FIREFLIES_API_KEY" \
  -d '{"query":"{ contacts { email name picture last_meeting_date } }"}' | jq
```

### Get Active Meetings

```bash
curl -s -X POST https://api.fireflies.ai/graphql \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer $FIREFLIES_API_KEY" \
  -d '{"query":"{ active_meetings { id title organizer_email meeting_link start_time state } }"}' | jq
```

---

## Pipeline Review Example

Get all meetings from last 7 days with specific participants:

```bash
# Date commands (pick based on your OS):
# macOS:
FROM_DATE=$(date -u -v-7d +"%Y-%m-%dT00:00:00.000Z")
# Linux:
# FROM_DATE=$(date -u -d '7 days ago' +"%Y-%m-%dT00:00:00.000Z")

TO_DATE=$(date -u +"%Y-%m-%dT23:59:59.999Z")

curl -s -X POST https://api.fireflies.ai/graphql \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer $FIREFLIES_API_KEY" \
  -d "{\"query\":\"query(\$from:DateTime,\$to:DateTime,\$participants:[String]){transcripts(fromDate:\\\"\$FROM_DATE\\\",toDate:\\\"\$TO_DATE\\\",participants:\$participants,limit:50){id title date duration organizer_email participants summary{keywords action_items topics_discussed meeting_type}}}\",\"variables\":{\"from\":\"$FROM_DATE\",\"to\":\"$TO_DATE\",\"participants\":[\"prospect@company.com\"]}}" | jq
```

---

## Key Schema Fields

### Transcript Fields
- `id` - Unique identifier
- `title` - Meeting title
- `date` - Unix timestamp (milliseconds)
- `dateString` - ISO 8601 datetime
- `duration` - Duration in minutes
- `organizer_email` - Meeting organizer
- `participants` - All participant emails
- `fireflies_users` - Fireflies users who participated
- `workspace_users` - Team members who participated
- `meeting_attendees` - Detailed attendee info (displayName, email)
- `transcript_url` - View in dashboard
- `audio_url` - Download audio (Pro+, expires 24h)
- `video_url` - Download video (Business+, expires 24h)

### Summary Fields
- `keywords` - Key topics
- `action_items` - Extracted action items
- `overview` - Meeting overview
- `topics_discussed` - Main topics
- `meeting_type` - Meeting category
- `outline` - Structured outline
- `bullet_gist` - Bullet point summary

### Sentence Fields
- `text` - Sentence text
- `speaker_name` - Who said it
- `start_time` - Timestamp (seconds)
- `end_time` - End timestamp
- `ai_filters` - Filters (task, question, pricing, etc.)

### Speaker Fields
- `name` - Speaker name
- `duration` - Speaking time
- `word_count` - Words spoken
- `filler_words` - Filler word count
- `questions` - Questions asked
- `longest_monologue` - Longest uninterrupted speech
- `words_per_minute` - Speaking pace

---

## Filter Examples

### By Date Range (ISO 8601)
```json
{
  "fromDate": "2024-01-01T00:00:00.000Z",
  "toDate": "2024-01-31T23:59:59.999Z"
}
```

### By Multiple Participants
```json
{
  "participants": ["user1@example.com", "user2@example.com"]
}
```

### By Channel
```json
{
  "channel_id": "channel_id_here"
}
```

### Combined Filters
```json
{
  "fromDate": "2024-01-01T00:00:00.000Z",
  "toDate": "2024-01-31T23:59:59.999Z",
  "participants": ["sales@example.com"],
  "keyword": "pricing",
  "scope": "ALL",
  "limit": 50
}
```

---

## PowerShell Examples

```powershell
$headers = @{
  "Authorization" = "Bearer $env:FIREFLIES_API_KEY"
  "Content-Type" = "application/json"
}

# Get recent transcripts
$body = @{
  query = "{ transcripts(mine:true,limit:10) { id title date } }"
} | ConvertTo-Json

Invoke-RestMethod -Uri "https://api.fireflies.ai/graphql" -Method POST -Headers $headers -Body $body
```

---

## Shareable Recording Links

The API provides `transcript_url`, `video_url`, and `audio_url`, but for **sharing with external parties** (prospects, clients), use the **embed URL format**:

```
API transcript_url:  https://app.fireflies.ai/view/{id}           (requires Fireflies login)
Embed URL:           https://share.fireflies.ai/embed/meetings/{id}  (no login required, permanent)
```

**Why use embed URLs:**
- No Fireflies account required to view
- Permanent link (doesn't expire like video_url/audio_url)
- Better viewing experience (embedded player)

**Construction:**
```bash
# Get meeting ID from API
MEETING_ID=$(curl -s -X POST https://api.fireflies.ai/graphql \
  -H "Authorization: Bearer $FIREFLIES_API_KEY" \
  -H "Content-Type: application/json" \
  -d '{"query":"{ transcripts(mine:true,limit:1) { id } }"}' | jq -r '.data.transcripts[0].id')

# Construct embed URL
EMBED_URL="https://share.fireflies.ai/embed/meetings/${MEETING_ID}"
echo "Share this: $EMBED_URL"
```

**Embed in HTML:**
```html
<iframe 
  src="https://share.fireflies.ai/embed/meetings/{id}" 
  width="640" 
  height="360" 
  frameborder="0" 
  allow="autoplay; fullscreen; picture-in-picture" 
  allowfullscreen>
</iframe>
```

---

## Notes

- **Dependencies**: Requires `curl` and `jq` (install: `sudo apt install jq` or `brew install jq`)
- **Rate Limits**: Check with Fireflies support for current limits
- **Pagination**: Use `limit` (max 50) and `skip` for large result sets
- **Date Format**: Always use ISO 8601 format: `YYYY-MM-DDTHH:mm:ss.sssZ`
- **Audio/Video URLs**: Expire after 24 hours, regenerate as needed (use embed URLs for permanent sharing)
- **Analytics**: Requires Pro plan or higher
- **Video Recording**: Must be enabled in dashboard settings

---

## Common Use Cases

1. **Weekly Pipeline Review**: Search transcripts by date + participants
2. **Follow-up Tasks**: Extract action items from recent meetings
3. **Competitor Mentions**: Search keyword in sentences
4. **Speaking Analytics**: Analyze talk time, questions asked
5. **Meeting Insights**: Get summaries and key topics
