"""
Finance Tracker — Storage Module
Handles JSON and Markdown storage with automatic sync
"""

import json
from datetime import datetime
from pathlib import Path
from typing import Optional, List, Dict, Any

try:
    from .categories import get_emoji, detect_category
except ImportError:
    from categories import get_emoji, detect_category


class FinanceStorage:
    """Handles all storage operations for the finance tracker."""
    
    VERSION = "1.0"
    
    def __init__(self, data_dir: Optional[Path] = None):
        """Initialize storage with data directory."""
        if data_dir is None:
            data_dir = Path.home() / ".finance-tracker"
        
        self.data_dir = Path(data_dir)
        self.data_dir.mkdir(parents=True, exist_ok=True)
        
        self.json_file = self.data_dir / "transactions.json"
        self.md_file = self.data_dir / "FINANCE_LOG.md"
        
        self._init_files()
    
    def _init_files(self):
        """Initialize data files if they don't exist."""
        if not self.json_file.exists():
            self._save_json({
                "version": self.VERSION,
                "currency": "UZS",
                "created_at": datetime.now().isoformat(),
                "transactions": []
            })
        
        if not self.md_file.exists():
            self._init_markdown()
    
    def _init_markdown(self):
        """Create initial markdown file."""
        content = f"""# 💰 Finance Log

**Currency:** UZS  
**Started:** {datetime.now().strftime("%Y-%m-%d")}

---

## Transactions

| Date | Category | Amount | Description |
|------|----------|--------|-------------|
<!-- TRANSACTIONS -->

---

*Auto-generated by Finance Tracker*
"""
        self.md_file.write_text(content)
    
    def _load_json(self) -> Dict[str, Any]:
        """Load data from JSON file."""
        try:
            with open(self.json_file) as f:
                return json.load(f)
        except (json.JSONDecodeError, FileNotFoundError):
            return {"version": self.VERSION, "currency": "UZS", "transactions": []}
    
    def _save_json(self, data: Dict[str, Any]):
        """Save data to JSON file."""
        with open(self.json_file, 'w') as f:
            json.dump(data, f, indent=2, ensure_ascii=False, default=str)
    
    def _append_markdown(self, tx: Dict[str, Any]):
        """Add transaction to markdown file."""
        content = self.md_file.read_text()
        
        date_str = datetime.fromisoformat(tx["date"]).strftime("%Y-%m-%d %H:%M")
        emoji = get_emoji(tx["category"])
        amount_str = f"{tx['amount']:,}"
        
        new_row = f"| {date_str} | {emoji} {tx['category']} | {amount_str} | {tx['description']} |\n"
        
        # Insert after marker
        marker = "<!-- TRANSACTIONS -->"
        if marker in content:
            content = content.replace(marker, new_row + marker)
        else:
            # Fallback: append to file
            content += f"\n{new_row}"
        
        self.md_file.write_text(content)
    
    def add_transaction(
        self,
        amount: int,
        description: str,
        category: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Add a new transaction.
        Returns the created transaction.
        """
        data = self._load_json()
        
        # Auto-detect category if not provided
        if category is None or category == "auto":
            category = detect_category(description)
        
        tx = {
            "id": len(data["transactions"]) + 1,
            "amount": amount,
            "category": category,
            "description": description,
            "date": datetime.now().isoformat(),
            "timestamp": int(datetime.now().timestamp())
        }
        
        data["transactions"].append(tx)
        self._save_json(data)
        self._append_markdown(tx)
        
        return tx
    
    def get_transactions(
        self,
        limit: Optional[int] = None,
        category: Optional[str] = None,
        days: Optional[int] = None
    ) -> List[Dict[str, Any]]:
        """Get transactions with optional filters."""
        data = self._load_json()
        transactions = data["transactions"]
        
        # Filter by days
        if days is not None:
            cutoff = datetime.now().timestamp() - (days * 86400)
            transactions = [tx for tx in transactions if tx["timestamp"] >= cutoff]
        
        # Filter by category
        if category is not None:
            transactions = [tx for tx in transactions if tx["category"] == category]
        
        # Sort by most recent
        transactions = sorted(transactions, key=lambda x: x["timestamp"], reverse=True)
        
        # Limit
        if limit is not None:
            transactions = transactions[:limit]
        
        return transactions
    
    def search(self, query: str) -> List[Dict[str, Any]]:
        """Search transactions by description."""
        data = self._load_json()
        query_lower = query.lower()
        
        results = [
            tx for tx in data["transactions"]
            if query_lower in tx["description"].lower()
        ]
        
        return sorted(results, key=lambda x: x["timestamp"], reverse=True)
    
    def get_currency(self) -> str:
        """Get the configured currency."""
        data = self._load_json()
        return data.get("currency", "UZS")
    
    def set_currency(self, currency: str):
        """Set the currency."""
        data = self._load_json()
        data["currency"] = currency.upper()
        self._save_json(data)
    
    def export_csv(self) -> str:
        """Export all transactions as CSV."""
        data = self._load_json()
        
        lines = ["date,category,amount,description"]
        for tx in data["transactions"]:
            date = datetime.fromisoformat(tx["date"]).strftime("%Y-%m-%d %H:%M")
            lines.append(f'{date},{tx["category"]},{tx["amount"]},"{tx["description"]}"')
        
        return "\n".join(lines)
    
    def undo_last(self) -> Optional[Dict[str, Any]]:
        """Remove the last transaction. Returns the removed transaction or None."""
        data = self._load_json()
        
        if not data["transactions"]:
            return None
        
        # Remove last transaction
        removed = data["transactions"].pop()
        self._save_json(data)
        
        # Also update markdown (remove last entry)
        self._remove_from_markdown(removed)
        
        return removed
    
    def _remove_from_markdown(self, tx: Dict[str, Any]):
        """Remove a transaction from the markdown file."""
        if not self.md_file.exists():
            return
        
        content = self.md_file.read_text()
        
        # Find and remove the line containing this transaction
        date_str = datetime.fromisoformat(tx["date"]).strftime("%Y-%m-%d %H:%M")
        lines = content.split("\n")
        new_lines = [line for line in lines if date_str not in line or tx["description"] not in line]
        
        self.md_file.write_text("\n".join(new_lines))
    
    def edit_transaction(
        self,
        tx_id: int,
        amount: Optional[int] = None,
        description: Optional[str] = None,
        category: Optional[str] = None
    ) -> Optional[Dict[str, Any]]:
        """Edit an existing transaction. Returns updated transaction or None."""
        data = self._load_json()
        
        for tx in data["transactions"]:
            if tx["id"] == tx_id:
                old_tx = tx.copy()
                
                if amount is not None:
                    tx["amount"] = amount
                if description is not None:
                    tx["description"] = description
                    # Re-detect category if description changed and no category specified
                    if category is None:
                        from categories import detect_category
                        tx["category"] = detect_category(description)
                if category is not None:
                    tx["category"] = category
                
                tx["updated_at"] = datetime.now().isoformat()
                
                self._save_json(data)
                
                # Update markdown
                self._remove_from_markdown(old_tx)
                self._append_markdown(tx)
                
                return tx
        
        return None
    
    def delete_transaction(self, tx_id: int) -> bool:
        """Delete a specific transaction by ID."""
        data = self._load_json()
        original_len = len(data["transactions"])
        
        removed = None
        new_transactions = []
        for tx in data["transactions"]:
            if tx["id"] == tx_id:
                removed = tx
            else:
                new_transactions.append(tx)
        
        if removed:
            data["transactions"] = new_transactions
            self._save_json(data)
            self._remove_from_markdown(removed)
            return True
        
        return False
    
    def get_stats(self, days: Optional[int] = None) -> Dict[str, Any]:
        """Get spending statistics."""
        transactions = self.get_transactions(days=days)
        
        if not transactions:
            return {
                "total": 0,
                "count": 0,
                "by_category": {},
                "average": 0
            }
        
        total = sum(tx["amount"] for tx in transactions)
        
        by_category = {}
        for tx in transactions:
            cat = tx["category"]
            if cat not in by_category:
                by_category[cat] = {"amount": 0, "count": 0}
            by_category[cat]["amount"] += tx["amount"]
            by_category[cat]["count"] += 1
        
        # Sort by amount
        by_category = dict(sorted(
            by_category.items(),
            key=lambda x: x[1]["amount"],
            reverse=True
        ))
        
        return {
            "total": total,
            "count": len(transactions),
            "by_category": by_category,
            "average": total // len(transactions) if transactions else 0
        }


# Global instance for convenience
_storage: Optional[FinanceStorage] = None


def get_storage(data_dir: Optional[Path] = None) -> FinanceStorage:
    """Get or create the global storage instance."""
    global _storage
    if _storage is None:
        _storage = FinanceStorage(data_dir)
    return _storage
