const fs = require('fs');
const path = require('path');
const { program } = require('commander');

// Configuration
const STICKER_DIR = process.env.STICKER_DIR || "/home/crishaocredits/.openclaw/media/stickers";
const INDEX_FILE = path.join(STICKER_DIR, 'index.json');

program
  .option('-q, --query <text>', 'Search query (e.g., "angry cat", "happy")')
  .option('--content <text>', 'Search query (alias for --query)')
  .option('-e, --emotion <emotion>', 'Filter by emotion (e.g., "happy", "sad")')
  .option('--random', 'Pick a random result if multiple match', false)
  .option('--json', 'Output JSON result', false);

program.parse(process.argv);
const options = program.opts();

// Alias mapping
if (options.content && !options.query) {
    options.query = options.content;
}

function loadIndex() {
    if (!fs.existsSync(INDEX_FILE)) {
        return {};
    }
    try {
        return JSON.parse(fs.readFileSync(INDEX_FILE, 'utf8'));
    } catch (e) {
        console.error("Failed to parse index file:", e.message);
        return {};
    }
}

function search(index, query, emotion) {
    const results = [];
    const searchTerms = query ? query.toLowerCase().split(/\s+/) : [];

    for (const [filename, data] of Object.entries(index)) {
        let score = 0;
        
        // Filter by emotion if specified
        if (emotion && data.emotion && data.emotion.toLowerCase() !== emotion.toLowerCase()) {
            continue;
        }

        // Keyword matching
        if (searchTerms.length > 0) {
            const keywords = (data.keywords || []).map(k => k.toLowerCase());
            const fileEmotion = (data.emotion || "").toLowerCase();
            
            // Check keywords
            for (const term of searchTerms) {
                if (keywords.includes(term)) score += 3;
                else if (keywords.some(k => k.includes(term))) score += 1;
                
                // Check emotion field matches query
                if (fileEmotion.includes(term)) score += 2;
            }
        } else {
            // No query, base score (e.g. random pick from emotion filter)
            score = 1;
        }

        if (score > 0) {
            results.push({ filename, ...data, score });
        }
    }

    return results.sort((a, b) => b.score - a.score);
}

function findSticker(query, emotion, random = false) {
    const index = loadIndex();
    const results = search(index, query, emotion);

    if (results.length === 0) return null;

    let selected;
    if (random) {
        // Pick random from top 3 (to add variety)
        const topN = results.slice(0, 3);
        selected = topN[Math.floor(Math.random() * topN.length)];
    } else {
        selected = results[0];
    }
    return selected;
}

if (require.main === module) {
    program
      .option('-q, --query <text>', 'Search query (e.g., "angry cat", "happy")')
      .option('-e, --emotion <emotion>', 'Filter by emotion (e.g., "happy", "sad")')
      .option('--random', 'Pick a random result if multiple match', false)
      .option('--json', 'Output JSON result', false);

    program.parse(process.argv);
    const options = program.opts();

    const result = findSticker(options.query, options.emotion, options.random);

    if (!result) {
        if (options.json) console.log(JSON.stringify({ found: false }));
        else console.log("No matching stickers found.");
    } else {
        if (options.json) {
            console.log(JSON.stringify({ found: true, sticker: result }));
        } else {
            console.log(`Found: ${result.filename}`);
            console.log(`Path: ${result.path}`);
            console.log(`Emotion: ${result.emotion}`);
            console.log(`Keywords: ${result.keywords.join(', ')}`);
        }
    }
}

module.exports = { findSticker };
