#!/usr/bin/env python3
"""
Feishu Folder Summary Generator
Helper script to generate summary reports from Feishu folders.
"""

import json
import sys
from datetime import datetime
from typing import Dict, List, Any

def generate_report_header(folder_name: str, url: str, total_docs: int) -> str:
    """Generate the report header section."""
    timestamp = datetime.now().strftime("%Y-%m-%d %H:%M")
    return f"""# {folder_name} Document Summary Report

**Source:** {url}  
**Generated:** {timestamp}  
**Total Documents:** {total_docs}

---

"""

def generate_directory_tree(structure: List[Dict], level: int = 0) -> str:
    """Generate a tree visualization of the folder structure."""
    tree = "## Directory Structure\n\n```\n"
    
    def add_node(node: Dict, prefix: str = "", is_last: bool = True):
        connector = "└── " if is_last else "├── "
        tree_line = f"{prefix}{connector}{node['title']}\n"
        nonlocal tree
        tree += tree_line
        
        if 'children' in node and node['children']:
            new_prefix = prefix + ("    " if is_last else "│   ")
            for i, child in enumerate(node['children']):
                add_node(child, new_prefix, i == len(node['children']) - 1)
    
    for i, node in enumerate(structure):
        add_node(node, "", i == len(structure) - 1)
    
    tree += "```\n\n---\n\n"
    return tree

def generate_document_summary(doc: Dict, index: str) -> str:
    """Generate summary for a single document."""
    title = doc.get('title', 'Untitled')
    doc_type = doc.get('type', 'unknown')
    status = doc.get('status', 'unknown')
    summary = doc.get('summary', 'No summary available')
    
    status_emoji = {
        'complete': '✅',
        'in_progress': '🚧',
        'empty': '⚠️',
        'unknown': '❓'
    }.get(status, '❓')
    
    return f"""### {index} {title}
- **Type:** {doc_type}
- **Status:** {status_emoji} {status.replace('_', ' ').title()}
- **Summary:** {summary}

"""

def generate_statistics(stats: Dict) -> str:
    """Generate the statistics table."""
    return f"""## Statistics

| Category | Count | Status |
|----------|-------|--------|
| Total | {stats.get('total', 0)} | - |
| Complete | {stats.get('complete', 0)} | ✅ |
| In Progress | {stats.get('in_progress', 0)} | 🚧 |
| Empty | {stats.get('empty', 0)} | ⚠️ |

---

*Report generated by feishu-folder-summary skill*
"""

def classify_content_status(content: str, block_count: int) -> str:
    """Classify document status based on content."""
    if block_count <= 2 or len(content.strip()) < 50:
        return 'empty'
    
    progress_keywords = ['待补充', '编写中', '开发中', '里程碑', 'TODO', '🚧']
    if any(kw in content for kw in progress_keywords):
        return 'in_progress'
    
    return 'complete'

def extract_key_summary(content: str, title: str) -> str:
    """Extract key summary points from document content."""
    # Remove markdown headers for analysis
    lines = content.split('\n')
    
    # Look for key sections
    summary_points = []
    
    for line in lines[:50]:  # Check first 50 lines
        line = line.strip()
        # Extract headings as key points
        if line.startswith('#') or line.startswith('**'):
            clean_line = line.lstrip('#').lstrip('*').strip()
            if len(clean_line) > 10 and len(clean_line) < 200:
                summary_points.append(clean_line)
        # Extract list items
        elif line.startswith('- ') or line.startswith('* '):
            if len(line) > 10 and len(line) < 150:
                summary_points.append(line[2:])
    
    # Limit summary length
    if summary_points:
        summary = '; '.join(summary_points[:5])
        if len(summary) > 300:
            summary = summary[:300] + '...'
        return summary
    
    # Fallback: use first non-empty paragraph
    for line in lines:
        line = line.strip()
        if line and not line.startswith('#') and len(line) > 20:
            return line[:200] + ('...' if len(line) > 200 else '')
    
    return "Document content available but no clear summary extracted"

if __name__ == '__main__':
    print("Feishu Folder Summary Helper")
    print("This script provides utility functions for generating summary reports.")
    print("Use through the skill interface for full functionality.")
