const Lark = require('@larksuiteoapi/node-sdk');
require('dotenv').config({ path: require('path').resolve(__dirname, '../../.env') });

const APP_ID = process.env.FEISHU_APP_ID;
const APP_SECRET = process.env.FEISHU_APP_SECRET;
const client = new Lark.Client({ appId: APP_ID, appSecret: APP_SECRET });

(async () => {
    console.log('🧹 Deep Cleaning Calendar...');
    
    // 1. Find Bot Calendar
    let botCalendarId;
    const calList = await client.calendar.calendar.list();
    if (calList.code === 0 && calList.data.calendar_list) {
        const botCal = calList.data.calendar_list.find(c => c.summary === 'OpenClaw Assistant');
        if (botCal) botCalendarId = botCal.calendar_id;
    }

    if (!botCalendarId) {
        // Try fallback
        try {
            const primary = await client.calendar.calendar.get({ calendar_id: 'primary' });
            if (primary.code === 0) botCalendarId = 'primary';
        } catch(e) {}
    }

    if (!botCalendarId) {
        console.log("No calendar to clean.");
        return;
    }

    // 2. List ALL Future Events
    const now = Math.floor(Date.now() / 1000);
    const endTime = now + 30 * 24 * 3600; // 30 days
    
    // SDK approach with raw fallback if needed
    let res = await client.request({
        method: 'GET',
        url: `/open-apis/calendar/v4/calendars/${encodeURIComponent(botCalendarId)}/events`,
        params: {
            start_time: String(now),
            end_time: String(endTime),
            page_size: 100
        }
    });
    
    if (res.code !== 0 && botCalendarId !== 'primary') {
         console.log(`Access failed for ${botCalendarId}. Retrying primary...`);
         botCalendarId = 'primary';
         res = await client.request({
            method: 'GET',
            url: `/open-apis/calendar/v4/calendars/primary/events`,
            params: {
                start_time: String(now),
                end_time: String(endTime),
                page_size: 100
            }
        });
    }

    if (res.code === 0 && res.data.items) {
        for (const evt of res.data.items) {
            // Delete everything EXCEPT the "System Maintenance" one?
            // Or just delete the "undefined" ones.
            if (!evt.summary || evt.summary === 'undefined' || evt.summary.trim() === '') {
                // If it is already deleted/cancelled (status="cancelled"), skip
                if (evt.status === 'cancelled') continue;

                console.log(`🗑️ Deleting Empty Event: ${evt.event_id}`);
                try {
                    await client.request({
                        method: 'DELETE',
                        url: `/open-apis/calendar/v4/calendars/${encodeURIComponent(botCalendarId)}/events/${evt.event_id}`
                    });
                } catch (delErr) {
                    console.log(`Failed to delete ${evt.event_id} (might be already gone): ${delErr.message}`);
                }
            }
        }
        console.log("Cleanup Done.");
    }
})();
