# Weekly Report Automation Workflow

Complete guide to automating weekly reports using Feishu Bitable and Documents.

## Overview

This workflow automates the creation of weekly status reports by:
1. Pulling data from Feishu Bitable (tasks, progress, metrics)
2. Generating a formatted markdown report
3. Creating/updating a Feishu document
4. Sharing with the team via wiki or chat

## Prerequisites

- Feishu app with permissions:
  - `bitable:app` (read Bitable)
  - `docx:document` (create/update documents)
  - `wiki:wiki` (optional, for wiki posting)
- Bitable with weekly data (tasks, metrics, etc.)
- OpenClaw with Feishu integration configured

## Step-by-Step Implementation

### 1. Design Your Bitable Structure

Create a Bitable with these suggested tables:

#### Tasks Table
| Field | Type | Description |
|-------|------|-------------|
| Task Name | Text | Task description |
| Assignee | User | Who's responsible |
| Status | Single Select | Not Started/In Progress/Blocked/Done |
| Progress | Number | 0-100 percentage |
| Week | Date | Week this task belongs to |
| Category | Single Select | Feature/Bug/Improvement |

#### Metrics Table
| Field | Type | Description |
|-------|------|-------------|
| Metric Name | Text | What's being measured |
| Value | Number | Current value |
| Target | Number | Target value |
| Week | Date | Week of measurement |
| Trend | Single Select | Up/Down/Stable |

### 2. Create Report Template

Save as `templates/weekly_report.md`:

```markdown
# Weekly Report: {{week}}

**Report Date:** {{report_date}}  
**Generated By:** OpenClaw Automation

## Executive Summary

{{summary}}

## Tasks Overview

{{tasks_table}}

## Metrics

{{metrics_table}}

## Key Accomplishments

{{accomplishments}}

## Risks and Blockers

{{risks}}

## Next Week's Focus

{{next_week_focus}}

---

*This report was automatically generated. Data source: {{bitable_url}}*
```

### 3. Automation Script

Use `bitable_to_doc.py` with custom logic:

```bash
python scripts/bitable_to_doc.py \
  --app-token basXXX \
  --table-id tblTasks \
  --query "Week='2026-02-10'" \
  --template templates/weekly_report.md \
  --title "Weekly Report 2026-02-10" \
  --output-folder fldcnReports
```

### 4. Custom Data Processing

For advanced reports, modify the script to:

```python
def generate_weekly_summary(task_records, metric_records):
    """Generate executive summary from data."""
    
    total_tasks = len(task_records)
    completed = sum(1 for t in task_records if t["fields"].get("Status") == "Done")
    completion_rate = (completed / total_tasks * 100) if total_tasks else 0
    
    summary = f"""
This week had {total_tasks} tasks with {completion_rate:.1f}% completion rate.
    
Key highlights:
"""
    
    # Add metrics summary
    for metric in metric_records:
        name = metric["fields"].get("Metric Name", "Unknown")
        value = metric["fields"].get("Value", 0)
        target = metric["fields"].get("Target", 0)
        trend = metric["fields"].get("Trend", "Stable")
        
        summary += f"- {name}: {value} (target: {target}, trend: {trend})\n"
    
    return summary
```

### 5. Schedule Automation

Set up a cron job to run weekly:

```yaml
# OpenClaw cron job configuration
schedule:
  kind: cron
  expr: "0 9 * * 1"  # Monday 9 AM
payload:
  kind: agentTurn
  message: "Generate weekly report for last week"
  model: "deepseek/deepseek-reasoner"
```

Or use system cron:
```bash
# Every Monday at 9 AM
0 9 * * 1 cd /path/to/workspace && python scripts/weekly_report.py
```

## Advanced Features

### Trend Analysis
Compare current week with previous week:

```python
def calculate_week_over_week(current_data, previous_data):
    """Calculate week-over-week changes."""
    changes = {}
    
    for metric in current_data:
        name = metric["name"]
        current_val = metric["value"]
        prev_val = find_previous_week_value(name, previous_data)
        
        if prev_val is not None:
            change_pct = ((current_val - prev_val) / prev_val * 100) if prev_val != 0 else 0
            changes[name] = {
                "current": current_val,
                "previous": prev_val,
                "change": change_pct,
                "direction": "up" if change_pct > 0 else "down" if change_pct < 0 else "stable"
            }
    
    return changes
```

### Team Notifications
After report generation, notify the team:

```python
def notify_team(report_url, recipients):
    """Send notification about new report."""
    message = f"📊 Weekly report is ready: {report_url}"
    
    # Use OpenClaw message tool or Feishu bot API
    for recipient in recipients:
        # Send via Feishu chat
        pass
```

### Archive Old Reports
Automatically archive reports older than 4 weeks:

```python
def archive_old_reports(folder_token, archive_folder_token):
    """Move old reports to archive folder."""
    
    # List documents in reports folder
    # Filter by date in title
    # Move to archive folder
    pass
```

## Troubleshooting

### Common Issues

1. **Missing Data**: Ensure Bitable has data for the specified week
2. **Permission Errors**: Verify app has required scopes
3. **Template Not Found**: Check template file path
4. **Rate Limiting**: Implement retry logic for API calls

### Debug Checklist
- [ ] Feishu app permissions correct
- [ ] Bitable accessible via app
- [ ] Template file exists and is readable
- [ ] Output folder exists and writable
- [ ] Date format matches Bitable data

## Customization Ideas

### Department-Specific Reports
Create variants for different teams:
- Engineering: Focus on PRs, deployments, incidents
- Sales: Focus on leads, conversions, revenue
- Marketing: Focus on campaigns, traffic, conversions

### Multi-Language Support
Generate reports in different languages based on team preferences.

### Data Visualization
Embed charts by generating images and uploading to Feishu.

### Integration with Other Tools
- Pull data from Jira, GitHub, etc.
- Push final report to Slack/Email
- Update project management tools with status

## Best Practices

1. **Test First**: Run in dry-run mode before production
2. **Error Handling**: Implement robust error handling and notifications
3. **Version Control**: Keep templates and scripts in version control
4. **Documentation**: Document any customizations for team members
5. **Review Process**: Have humans review automated reports initially