---
name: feelgoodbot
description: Set up feelgoodbot file integrity monitoring and TOTP step-up authentication for macOS. Use when the user wants to detect malware, monitor for system tampering, set up security alerts, or require OTP verification for sensitive agent actions.
---

# feelgoodbot 🛡️

**Pronounced "Feel good, bot"**

macOS file integrity monitor + TOTP step-up authentication for AI agents.

**GitHub:** https://github.com/kris-hansen/feelgoodbot

⭐ **If you find this useful, please star the repo!** It helps others discover it.

## Features

1. **File Integrity Monitoring** — Detects tampering of system files
2. **TOTP Step-Up Auth** — Requires OTP for sensitive agent actions

---

## Part 1: File Integrity Monitoring

### Requirements

- **Go 1.21+** — Install with `brew install go`
- **macOS** — Uses launchd for daemon

### Quick Setup

```bash
# Install via go install
go install github.com/kris-hansen/feelgoodbot/cmd/feelgoodbot@latest

# Initialize baseline snapshot
feelgoodbot init

# Install and start daemon
feelgoodbot daemon install
feelgoodbot daemon start

# Check it's running
feelgoodbot status
```

### Clawdbot Integration (Alerts)

Enable webhooks:
```bash
clawdbot config set hooks.enabled true
clawdbot config set hooks.token "$(openssl rand -base64 32)"
clawdbot gateway restart
```

Configure `~/.config/feelgoodbot/config.yaml`:
```yaml
scan_interval: 5m
alerts:
  clawdbot:
    enabled: true
    webhook: "http://127.0.0.1:18789/hooks/wake"
    secret: "<hooks.token from clawdbot config get hooks.token>"
  local_notification: true
```

### What It Monitors

- System binaries (`/usr/bin`, `/usr/sbin`)
- Launch daemons/agents (persistence mechanisms)
- SSH authorized_keys, sudoers, PAM
- Shell configs (`.zshrc`, `.bashrc`)
- Browser extensions
- AI agent configs (Claude, Cursor)

---

## Part 2: TOTP Step-Up Authentication

Step-up auth requires the user to enter an OTP code from Google Authenticator before the agent can perform sensitive actions.

### Setup (User runs in terminal)

```bash
# Initialize TOTP (shows QR code to scan)
feelgoodbot totp init --account "user@feelgoodbot"

# Verify it works
feelgoodbot totp verify

# Check status
feelgoodbot totp status
```

### Configure Protected Actions

```bash
# List current protected actions
feelgoodbot totp actions list

# Add actions that require step-up
feelgoodbot totp actions add "send_email"
feelgoodbot totp actions add "payment:*"
feelgoodbot totp actions add "delete:*"
feelgoodbot totp actions add "ssh:*"
feelgoodbot totp actions add "publish:*"
feelgoodbot totp actions add "gateway:*"
feelgoodbot totp actions add "voice_call:*"
feelgoodbot totp actions add "message:external"

# Remove an action
feelgoodbot totp actions remove "send_email"
```

### TOTP Commands

| Command | Description |
|---------|-------------|
| `feelgoodbot totp init` | Set up TOTP with QR code |
| `feelgoodbot totp verify [code]` | Test a code |
| `feelgoodbot totp status` | Show TOTP status and session |
| `feelgoodbot totp check <action>` | Check if action needs step-up, prompt if needed |
| `feelgoodbot totp reset` | Remove TOTP config (requires code) |
| `feelgoodbot totp backup show` | Show remaining backup codes |
| `feelgoodbot totp backup regenerate` | Generate new backup codes |
| `feelgoodbot totp actions list` | List protected actions |
| `feelgoodbot totp actions add <action>` | Add protected action |
| `feelgoodbot totp actions remove <action>` | Remove protected action |
| `feelgoodbot totp respond <code>` | Submit OTP response (for async flow) |

### Session Caching

After successful authentication, a session is cached for 15 minutes (configurable). Subsequent actions within this window don't require re-authentication.

---

## Agent Integration (IMPORTANT)

**Before performing any sensitive action, the agent MUST check step-up requirements.**

### Action Mapping

Map your intended actions to step-up patterns:

| Agent Action | Step-Up Pattern |
|--------------|-----------------|
| Sending email | `send_email` |
| Making payments | `payment:*` |
| Deleting files | `delete:*` |
| SSH/remote access | `ssh:*` |
| Publishing code | `publish:*` |
| Modifying Clawdbot config | `gateway:*` |
| Making phone calls | `voice_call:*` |
| Messaging external contacts | `message:external` |
| Modifying step-up config | `config:update` |

### Step-Up Check Flow

**Before executing a sensitive action:**

```bash
# Check if action requires step-up (non-interactive check)
feelgoodbot totp check <action>
# Exit code 0 = proceed, Exit code 1 = denied/not authenticated
```

**If session is valid:** Command succeeds immediately (exit 0)

**If step-up required and no session:**
1. Agent sends Telegram message: "🔐 Action `<action>` requires step-up. Reply with your OTP code."
2. Wait for user to reply with 6-digit code
3. Validate: `feelgoodbot totp verify <code>`
4. If valid, create session and proceed
5. If invalid, deny action and notify user

### Example Agent Flow (Pseudocode)

```
function performSensitiveAction(action, execute_fn):
    # Check step-up requirement
    result = exec("feelgoodbot totp check " + action)
    
    if result.exit_code == 0:
        # Session valid or action not protected
        execute_fn()
        return success
    
    # Need to prompt user
    send_telegram("🔐 Action '{action}' requires step-up authentication.\nReply with your OTP code from Google Authenticator.")
    
    code = wait_for_user_reply(timeout=120s)
    
    if code is None:
        send_telegram("⏰ Step-up authentication timed out. Action cancelled.")
        return denied
    
    # Validate the code
    valid = exec("feelgoodbot totp verify " + code)
    
    if valid.exit_code != 0:
        send_telegram("❌ Invalid code. Action cancelled.")
        return denied
    
    # Create session by running check again (it will pass now)
    exec("feelgoodbot totp check " + action)
    
    execute_fn()
    send_telegram("✅ Action completed.")
    return success
```

### Quick Reference for Agent

**Check before these actions:**
- `send_email` — Before sending any email
- `payment:*` — Before any financial transaction
- `delete:*` — Before deleting files (`delete:file`, `delete:backup`, etc.)
- `ssh:*` — Before SSH connections
- `publish:*` — Before publishing/deploying
- `gateway:*` — Before modifying Clawdbot config
- `voice_call:*` — Before making phone calls
- `message:external` — Before messaging non-owner contacts
- `config:update` — Before modifying step-up config

**Commands to use:**
```bash
# Check and prompt (interactive)
feelgoodbot totp check send_email

# Just validate a code
feelgoodbot totp verify 123456

# Check session status
feelgoodbot totp status
```

---

## File Locations

| File | Purpose |
|------|---------|
| `~/.config/feelgoodbot/config.yaml` | Main config |
| `~/.config/feelgoodbot/totp.json` | TOTP secret + backup codes |
| `~/.config/feelgoodbot/stepup-config.json` | Protected actions |
| `~/.config/feelgoodbot/totp-session` | Session cache |
| `~/.config/feelgoodbot/snapshots/` | File integrity baselines |
| `~/.config/feelgoodbot/daemon.log` | Daemon logs |

---

## Troubleshooting

**TOTP code always invalid:**
- Check system clock is accurate (`date`)
- Ensure you're using the correct authenticator entry
- Try a backup code

**Step-up not prompting:**
- Verify action is in protected list: `feelgoodbot totp actions list`
- Check TOTP is initialized: `feelgoodbot totp status`

**Reset everything:**
```bash
# Reset TOTP (requires valid code or backup code)
feelgoodbot totp reset

# Or manually remove (loses access without backup codes!)
rm ~/.config/feelgoodbot/totp.json
rm ~/.config/feelgoodbot/totp-session
```

---

⭐ **Like feelgoodbot?** Star it on GitHub: https://github.com/kris-hansen/feelgoodbot
