import { JMAPClient } from '../jmap-client.js';
import { NotFoundError } from '../errors.js';

export interface Email {
  id: string;
  blobId: string;
  threadId: string;
  mailboxIds: Record<string, boolean>;
  subject: string;
  from: { name: string; email: string }[] | null;
  to: { name: string; email: string }[] | null;
  cc: { name: string; email: string }[] | null;
  receivedAt: string;
  preview: string;
  hasAttachment: boolean;
  keywords: Record<string, boolean>;
}

export interface Mailbox {
  id: string;
  name: string;
  role: string | null;
  totalEmails: number;
  unreadEmails: number;
}

export interface EmailThread {
  id: string;
  emails: Email[];
  subject: string;
  participants: { name: string; email: string }[];
  latestDate: string;
  emailCount: number;
}

export class EmailTools {
  constructor(private client: JMAPClient) {}
  
  async getMailboxes(): Promise<Mailbox[]> {
    const accountId = await this.client.getAccountId();
    
    const response = await this.client.call([
      ['Mailbox/get', { accountId, properties: ['id', 'name', 'role', 'totalEmails', 'unreadEmails'] }, 'a'],
    ]);
    
    const result = response.methodResponses[0][1] as { list: Mailbox[] };
    return result.list;
  }
  
  async getInboxId(): Promise<string> {
    const mailboxes = await this.getMailboxes();
    const inbox = mailboxes.find(m => m.role === 'inbox');
    if (!inbox) throw new Error('Inbox not found');
    return inbox.id;
  }
  
  async listEmails(mailboxId?: string, limit = 20): Promise<Email[]> {
    const accountId = await this.client.getAccountId();
    const targetMailbox = mailboxId || await this.getInboxId();
    
    const response = await this.client.call([
      ['Email/query', {
        accountId,
        filter: { inMailbox: targetMailbox },
        sort: [{ property: 'receivedAt', isAscending: false }],
        limit,
      }, 'a'],
      ['Email/get', {
        accountId,
        '#ids': { resultOf: 'a', name: 'Email/query', path: '/ids' },
        properties: ['id', 'blobId', 'threadId', 'mailboxIds', 'subject', 'from', 'to', 'cc', 'receivedAt', 'preview', 'hasAttachment', 'keywords'],
      }, 'b'],
    ]);
    
    const result = response.methodResponses[1][1] as { list: Email[] };
    return result.list;
  }
  
  async getEmail(emailId: string): Promise<Email & { textBody: string; htmlBody: string }> {
    const accountId = await this.client.getAccountId();
    
    const response = await this.client.call([
      ['Email/get', {
        accountId,
        ids: [emailId],
        properties: ['id', 'blobId', 'threadId', 'mailboxIds', 'subject', 'from', 'to', 'cc', 'receivedAt', 'preview', 'hasAttachment', 'keywords', 'textBody', 'htmlBody', 'bodyValues'],
        fetchTextBodyValues: true,
        fetchHTMLBodyValues: true,
      }, 'a'],
    ]);
    
    const result = response.methodResponses[0][1] as { list: (Email & { textBody: string; htmlBody: string })[] };
    if (result.list.length === 0) throw new Error('Email not found');
    return result.list[0];
  }
  
  async searchEmails(query: string, limit = 20): Promise<Email[]> {
    const accountId = await this.client.getAccountId();
    
    const response = await this.client.call([
      ['Email/query', {
        accountId,
        filter: { text: query },
        sort: [{ property: 'receivedAt', isAscending: false }],
        limit,
      }, 'a'],
      ['Email/get', {
        accountId,
        '#ids': { resultOf: 'a', name: 'Email/query', path: '/ids' },
        properties: ['id', 'blobId', 'threadId', 'mailboxIds', 'subject', 'from', 'to', 'cc', 'receivedAt', 'preview', 'hasAttachment', 'keywords'],
      }, 'b'],
    ]);
    
    const result = response.methodResponses[1][1] as { list: Email[] };
    return result.list;
  }
  
  async sendEmail(options: {
    to: { name?: string; email: string }[];
    cc?: { name?: string; email: string }[];
    bcc?: { name?: string; email: string }[];
    subject: string;
    textBody: string;
    htmlBody?: string;
    inReplyTo?: string;
    references?: string[];
  }): Promise<{ emailId: string; submissionId: string }> {
    const accountId = await this.client.getAccountId();
    
    // Get identity for sending
    const identityResponse = await this.client.call([
      ['Identity/get', { accountId }, 'a'],
    ]);
    
    const identities = (identityResponse.methodResponses[0][1] as { list: { id: string; email: string }[] }).list;
    if (identities.length === 0) throw new Error('No identity found');
    const identityId = identities[0].id;
    
    // Create email
    const emailCreate: Record<string, unknown> = {
      'mailboxIds': {}, // Will be set by submission
      'to': options.to,
      'subject': options.subject,
      'bodyValues': {
        'body': { value: options.textBody, isEncodingProblem: false, isTruncated: false },
      },
      'textBody': [{ partId: 'body', type: 'text/plain' }],
    };
    
    if (options.cc) emailCreate['cc'] = options.cc;
    if (options.bcc) emailCreate['bcc'] = options.bcc;
    if (options.inReplyTo) emailCreate['inReplyTo'] = [options.inReplyTo];
    if (options.references) emailCreate['references'] = options.references;
    
    if (options.htmlBody) {
      emailCreate['bodyValues'] = {
        'textBody': { value: options.textBody, isEncodingProblem: false, isTruncated: false },
        'htmlBody': { value: options.htmlBody, isEncodingProblem: false, isTruncated: false },
      };
      emailCreate['textBody'] = [{ partId: 'textBody', type: 'text/plain' }];
      emailCreate['htmlBody'] = [{ partId: 'htmlBody', type: 'text/html' }];
    }
    
    const response = await this.client.call([
      ['Email/set', {
        accountId,
        create: { 'draft': emailCreate },
      }, 'a'],
      ['EmailSubmission/set', {
        accountId,
        create: {
          'send': {
            identityId,
            emailId: '#draft',
          },
        },
        onSuccessUpdateEmail: {
          '#send': {
            'mailboxIds/drafts': null,
            'mailboxIds/sent': true,
            'keywords/$draft': null,
          },
        },
      }, 'b'],
    ]);
    
    const emailResult = response.methodResponses[0][1] as { created?: { draft: { id: string } } };
    const submissionResult = response.methodResponses[1][1] as { created?: { send: { id: string } } };
    
    return {
      emailId: emailResult.created?.draft?.id || '',
      submissionId: submissionResult.created?.send?.id || '',
    };
  }
  
  async moveToFolder(emailId: string, targetMailboxId: string, sourceMailboxId?: string): Promise<void> {
    const accountId = await this.client.getAccountId();
    
    const patch: Record<string, boolean | null> = {
      [`mailboxIds/${targetMailboxId}`]: true,
    };
    
    const mailboxesToRemove = new Set<string>();
    if (sourceMailboxId) {
      mailboxesToRemove.add(sourceMailboxId);
    } else {
      const currentEmail = await this.getEmail(emailId);
      for (const mailboxId of Object.keys(currentEmail.mailboxIds || {})) {
        if (mailboxId !== targetMailboxId) {
          mailboxesToRemove.add(mailboxId);
        }
      }
    }
    for (const mailboxId of mailboxesToRemove) {
      patch[`mailboxIds/${mailboxId}`] = null;
    }
    
    await this.client.call([
      ['Email/set', {
        accountId,
        update: {
          [emailId]: patch,
        },
      }, 'a'],
    ]);
  }
  
  async setKeywords(emailId: string, keywords: Record<string, boolean>): Promise<void> {
    const accountId = await this.client.getAccountId();
    
    const patch: Record<string, boolean | null> = {};
    for (const [keyword, value] of Object.entries(keywords)) {
      patch[`keywords/${keyword}`] = value || null;
    }
    
    await this.client.call([
      ['Email/set', {
        accountId,
        update: {
          [emailId]: patch,
        },
      }, 'a'],
    ]);
  }
  
  async deleteEmail(emailId: string): Promise<void> {
    // Move to trash
    const mailboxes = await this.getMailboxes();
    const trash = mailboxes.find(m => m.role === 'trash');
    if (!trash) throw new Error('Trash folder not found');
    
    await this.moveToFolder(emailId, trash.id);
  }

  async getThread(emailId: string): Promise<EmailThread> {
    const accountId = await this.client.getAccountId();
    
    // First get the email to find its threadId
    const response = await this.client.call([
      ['Email/get', {
        accountId,
        ids: [emailId],
        properties: ['id', 'threadId'],
      }, 'a'],
    ]);
    
    const emailResult = response.methodResponses[0][1] as { list: { id: string; threadId: string }[] };
    if (emailResult.list.length === 0) {
      throw new NotFoundError('email', emailId);
    }
    
    const threadId = emailResult.list[0].threadId;
    
    // Now get all emails in this thread
    const threadResponse = await this.client.call([
      ['Email/query', {
        accountId,
        filter: { inThread: threadId },
        sort: [{ property: 'receivedAt', isAscending: true }], // Oldest first in thread
      }, 'a'],
      ['Email/get', {
        accountId,
        '#ids': { resultOf: 'a', name: 'Email/query', path: '/ids' },
        properties: ['id', 'blobId', 'threadId', 'mailboxIds', 'subject', 'from', 'to', 'cc', 'receivedAt', 'preview', 'hasAttachment', 'keywords'],
      }, 'b'],
    ]);
    
    const emails = (threadResponse.methodResponses[1][1] as { list: Email[] }).list;
    
    if (emails.length === 0) {
      throw new NotFoundError('thread', threadId);
    }
    
    // Extract unique participants
    const participantMap = new Map<string, { name: string; email: string }>();
    for (const email of emails) {
      const addParticipant = (p: { name: string; email: string } | null) => {
        if (p && !participantMap.has(p.email)) {
          participantMap.set(p.email, p);
        }
      };
      
      email.from?.forEach(addParticipant);
      email.to?.forEach(addParticipant);
      email.cc?.forEach(addParticipant);
    }
    
    return {
      id: threadId,
      emails,
      subject: emails[0].subject,
      participants: Array.from(participantMap.values()),
      latestDate: emails[emails.length - 1].receivedAt,
      emailCount: emails.length,
    };
  }

  async bulkMoveToFolder(
    emailIds: string[],
    targetMailboxId: string,
    sourceMailboxId?: string
  ): Promise<{ succeeded: string[]; failed: { id: string; error: string }[] }> {
    const accountId = await this.client.getAccountId();
    
    // Build update object for all emails
    const update: Record<string, Record<string, boolean | null>> = {};
    
    for (const emailId of emailIds) {
      update[emailId] = {
        [`mailboxIds/${targetMailboxId}`]: true,
      };
      
      if (sourceMailboxId) {
        update[emailId][`mailboxIds/${sourceMailboxId}`] = null;
      }
    }
    
    const response = await this.client.call([
      ['Email/set', {
        accountId,
        update,
      }, 'a'],
    ]);
    
    const result = response.methodResponses[0][1] as {
      updated?: Record<string, unknown>;
      notUpdated?: Record<string, { type: string; description: string }>;
    };
    
    const succeeded = Object.keys(result.updated || {});
    const failed = Object.entries(result.notUpdated || {}).map(([id, err]) => ({
      id,
      error: err.description || err.type,
    }));
    
    return { succeeded, failed };
  }

  async bulkSetKeywords(
    emailIds: string[],
    keywords: Record<string, boolean>
  ): Promise<{ succeeded: string[]; failed: { id: string; error: string }[] }> {
    const accountId = await this.client.getAccountId();
    
    // Build patch for keywords
    const patch: Record<string, boolean | null> = {};
    for (const [keyword, value] of Object.entries(keywords)) {
      patch[`keywords/${keyword}`] = value || null;
    }
    
    // Apply to all emails
    const update: Record<string, Record<string, boolean | null>> = {};
    for (const emailId of emailIds) {
      update[emailId] = { ...patch };
    }
    
    const response = await this.client.call([
      ['Email/set', {
        accountId,
        update,
      }, 'a'],
    ]);
    
    const result = response.methodResponses[0][1] as {
      updated?: Record<string, unknown>;
      notUpdated?: Record<string, { type: string; description: string }>;
    };
    
    const succeeded = Object.keys(result.updated || {});
    const failed = Object.entries(result.notUpdated || {}).map(([id, err]) => ({
      id,
      error: err.description || err.type,
    }));
    
    return { succeeded, failed };
  }

  async bulkDeleteEmails(
    emailIds: string[]
  ): Promise<{ succeeded: string[]; failed: { id: string; error: string }[] }> {
    // Get trash folder
    const mailboxes = await this.getMailboxes();
    const trash = mailboxes.find(m => m.role === 'trash');
    if (!trash) {
      throw new NotFoundError('mailbox', 'Trash');
    }
    
    return this.bulkMoveToFolder(emailIds, trash.id);
  }
}
